/*
Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.aws.presenter;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import com.clustercontrol.ws.cloud.CloudEndpoint;
import com.clustercontrol.ws.cloud.CloudInstanceBackup;

public class InstanceBackupRoot extends Element implements IInstanceBackupRoot {
	private BackupManagement backupManagement;
	private List<InstanceBackup> instanceBackups;

	public InstanceBackupRoot(BackupManagement backupManagement) {
		this.backupManagement = backupManagement;
	}	

	@Override
	public void update() throws CloudModelException {
		List<CloudInstanceBackup> cloudInstanceBackup = Collections.emptyList();

		try {
			CloudEndpoint endpoint = getCloudResourceManager().getEndpoint(CloudEndpoint.class);
			cloudInstanceBackup = endpoint.getInstanceBackupsByRegion(getBackupManagement().getRegion().getRegionName());
		}
		catch (Exception e) {
			throw new CloudModelException(e);
		}

		internalUpdate(cloudInstanceBackup);

		super.update();
	}
	
	public boolean isInitialized() {
		return instanceBackups != null;
	}
	
	private List<InstanceBackup> getInstanceBackupsList() {
		if (!isInitialized()) {
			update();
		}
		return instanceBackups;
	}
	
	public void internalUpdate(List<CloudInstanceBackup> cloudInstancesBackups) throws CloudModelException {
		if (isInitialized()) {
			List<InstanceBackup> instancesBackupsTemp = new ArrayList<InstanceBackup>(instanceBackups);
			List<CloudInstanceBackup> cloudInstanceBackupsTemp = new ArrayList<CloudInstanceBackup>(cloudInstancesBackups);
			Iterator<CloudInstanceBackup> cloudInstanceBackupsIter = cloudInstanceBackupsTemp.iterator();
			while (cloudInstanceBackupsIter.hasNext()) {
				CloudInstanceBackup cib = cloudInstanceBackupsIter.next();
				
				Iterator<InstanceBackup> instanceBackupsIter = instancesBackupsTemp.iterator();
				while (instanceBackupsIter.hasNext()) {
					InstanceBackup i = instanceBackupsIter.next();
					
					if (cib.getImageId().equals(i.getImageId()) && getBackupManagement().getRegion().getRegionName().equals(cib.getRegion())) {
						i.internalUpdate(cib);

						instanceBackupsIter.remove();
						cloudInstanceBackupsIter.remove();
						break;
					}
				}
			}

			for (InstanceBackup i: instancesBackupsTemp) {
				getInstanceBackupsList().remove(i);
				fireElementRemoved(p2.instanceBackups, i);
			}

			for (CloudInstanceBackup cib: cloudInstanceBackupsTemp) {
				InstanceBackup newInstanceBackup = InstanceBackup.createInstanceBackup(this, cib);
				getInstanceBackupsList().add(newInstanceBackup);
				this.fireElementAdded(p2.instanceBackups, newInstanceBackup);
			}
		}
		else {
			List<InstanceBackup> tempInstanceBackups = new ArrayList<InstanceBackup>();
			for (CloudInstanceBackup instanceBackup: cloudInstancesBackups) {
				tempInstanceBackups.add(InstanceBackup.createInstanceBackup(this, instanceBackup));
			}
			instanceBackups = tempInstanceBackups;
		}
	}

	@Override
	public BackupManagement getBackupManagement() {
		return backupManagement;
	}

	@Override
	public InstanceBackup[] getInstanceBackups() {
		return getInstanceBackupsList().toArray(new InstanceBackup[0]);
	}

	@Override
	public CloudResourceManager getCloudResourceManager() {
		return backupManagement.getCloudResourceManager();
	}

	@Override
	public IInstanceBackup createInstanceBackup(CreateInstanceBackupRequest request) {
		com.clustercontrol.ws.cloud.CreateInstanceBackupRequest wsRequest = new com.clustercontrol.ws.cloud.CreateInstanceBackupRequest();
		wsRequest.setRegion(getBackupManagement().getRegion().getRegionName());
		wsRequest.setInstanceId(request.getInstanceId());
		wsRequest.setImageName(request.getImageName());
		wsRequest.setDescription(request.getDescription());
		wsRequest.setNoReboot(request.getNoReboot());
		wsRequest.setWithEBSVolume(request.getWithEBSVolume());

		try {
			CloudEndpoint endpoint = getCloudResourceManager().getEndpoint(CloudEndpoint.class);
			CloudInstanceBackup backup = endpoint.addInstanceBackup(wsRequest);
			InstanceBackup newBackup = InstanceBackup.createInstanceBackup(this, backup);
			instanceBackups.add(newBackup);
			fireElementAdded(IInstanceBackupRoot.p2.instanceBackups, newBackup);
			return newBackup;
		}
		catch (Exception e) {
			throw new CloudModelException(e);
		}
	}

	@Override
	public void deleteInstanceBackup(String imageId) {
		try {
			CloudEndpoint endpoint = getCloudResourceManager().getEndpoint(CloudEndpoint.class);
			endpoint.removeInstanceBackup(backupManagement.getRegion().getRegionName(), imageId);
			IInstanceBackup removeBackup = getInstanceBackup(imageId);
			instanceBackups.remove(removeBackup);
			fireElementRemoved(InstanceBackupRoot.p2.instanceBackups, removeBackup);
		}
		catch (Exception e) {
			throw new CloudModelException(e);
		}
	}
	
	public InstanceBackup getInstanceBackup(String imageId) {
		for(InstanceBackup backup: instanceBackups){
			if(backup.getImageId().equals(imageId)){
				return backup;
			}
		}
		return null;
	}
}
