/*
Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.aws.presenter;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import com.clustercontrol.ws.cloud.CloudEndpoint;
import com.clustercontrol.ws.cloud.CloudUser;
import com.clustercontrol.ws.cloud.CloudUserType;
import com.clustercontrol.ws.cloud.Filter;

public class Identity extends Element implements IIdentity {
	private String name;
	private String id;
	private AccessKey key;
	private Category category;
	private Identity parent;
	private List<Identity> identities = new ArrayList<Identity>();
	private String description;
	private String hinemosUserId;
	private Date regDate;
	private String regUser;
	private Date updateDate;
	private String updateUser;
	
	public Identity() {
	}
	
	@Override
	public String getName() {
		return name;
	}
	@Override
	public String getId() {
		return id;
	}
	@Override
	public AccessKey getAccessKey() {
		return key;
	}
	@Override
	public String getDescription() {
		return description;
	}
	@Override
	public Identity[] getIdentities() {
		return identities.toArray(new Identity[0]);
	}
	@Override
	public Category getCategory() {
		return category;
	}
	@Override
	public Identity getParent() {
		return parent;
	}
	@Override
	public String getHinemosUserId() {
		return hinemosUserId;
	}
	@Override
	public Date getRegDate() {
		return regDate;
	}
	@Override
	public Date getUpdateDate() {
		return updateDate;
	}
	@Override
	public String getRegUser() {
		return regUser;
	}
	@Override
	public String getUpdateUser() {
		return updateUser;
	}

	public void setName(String name) {
		if (this.name == null) {
			if (name == null) {
				return;
			}
		}
		else {
			if (this.name.equals(name)) {
				return;
			}
		}

		String oldName = this.name;
		this.name = name;
		
		// 更新分を通知。
		firePropertyChanged2(this, IIdentity.p2.name, oldName, this.name);
	}

	public void setId(String id) {
		if (this.id == null) {
			if (id == null) {
				return;
			}
		}
		else {
			if (this.id.equals(id)) {
				return;
			}
		}

		String oldId = this.id;
		this.id = id;
		
		// 更新分を通知。
		firePropertyChanged2(this, IIdentity.p2.id, oldId, this.name);
	}

	public void setAccessKey(AccessKey key) {
		if (this.key == null) {
			if (key == null) {
				return;
			}
		}
		else {
			if (this.key.equals(key)) {
				return;
			}
		}

		AccessKey oldkey = this.key;
		this.key = key;
		
		// 更新分を通知。
		firePropertyChanged2(this, IIdentity.p2.accessKey, oldkey, this.key);
	}

	public void setDescription(String description) {
		if (this.description == null) {
			if (description == null) {
				return;
			}
		}
		else {
			if (this.description.equals(description)) {
				return;
			}
		}

		String oldDescription = this.description;
		this.description = description;
		// 更新分を通知。
		firePropertyChanged2(this, IIdentity.p2.description, oldDescription, this.description);
	}

	public void setCategory(Category category) {
		if (this.category == null) {
			if (category == null) {
				return;
			}
		}
		else {
			if (this.category.equals(category)) {
				return;
			}
		}

		Category oldCategory = this.category;
		this.category = category;
		firePropertyChanged2(this, IIdentity.p2.category, oldCategory, this.category);
	}

	public void setParent(Identity parent) {
		if (this.parent == null) {
			if (parent == null) {
				return;
			}
		}
		else {
			if (this.parent.equals(parent)) {
				return;
			}
		}

		Identity oldParent = this.parent;
		this.parent = parent;
		
		// 更新分を通知。
		firePropertyChanged2(this, IIdentity.p2.parent, oldParent, this.parent);
	}

	public void setHinemosUserId(String hinemosUserId) {
		if (this.hinemosUserId == null) {
			if (hinemosUserId == null) {
				return;
			}
		}
		else {
			if (this.hinemosUserId.equals(hinemosUserId)) {
				return;
			}
		}

		String oldhinemosUserId = this.hinemosUserId;
		this.hinemosUserId = hinemosUserId;
		
		// 更新分を通知。
		firePropertyChanged2(this, IIdentity.p2.hinemosUserId, oldhinemosUserId, this.hinemosUserId);
	}

	public void setRegDate(Date regDate) {
		if (this.regDate == null) {
			if (regDate == null) {
				return;
			}
		}
		else {
			if (this.regDate.equals(regDate)) {
				return;
			}
		}

		Date oldRegDate = this.regDate;
		this.regDate = regDate;
		
		// 更新分を通知。
		firePropertyChanged2(this, IIdentity.p2.regDate, oldRegDate, this.regDate);
	}

	public void setUpdateDate(Date updateDate) {
		if (this.updateDate == null) {
			if (updateDate == null) {
				return;
			}
		}
		else {
			if (this.updateDate.equals(updateDate)) {
				return;
			}
		}

		Date oldUpdateDate = this.updateDate;
		this.updateDate = updateDate;
		
		// 更新分を通知。
		firePropertyChanged2(this, IIdentity.p2.regDate, oldUpdateDate, this.updateDate);

	}
	public void setRegUser(String regUser) {
		if (this.regUser == null) {
			if (regUser == null) {
				return;
			}
		}
		else {
			if (this.regUser.equals(regUser)) {
				return;
			}
		}
		String oldRegUser = this.regUser;
		this.regUser = regUser;
		
		// 更新分を通知。
		firePropertyChanged2(this, IIdentity.p2.regUser, oldRegUser, this.regUser);
	}

	public void setUpdateUser(String updateUser) {
		if (this.updateUser == null) {
			if (updateUser == null) {
				return;
			}
		}
		else {
			if (this.updateUser.equals(updateUser)) {
				return;
			}
		}

		String oldUpdateUser = this.updateUser;
		this.updateUser = updateUser;
		
		// 更新分を通知。
		firePropertyChanged2(this, IIdentity.p2.updateUser, oldUpdateUser, this.updateUser);
	}

	public CloudResourceManager getCloudResourceManager() {
		return category.getCloudManager();
	}

	public void addChildIdentity(Identity child) {
		if (!identities.contains(child)) {
			identities.add(child);
			fireElementAdded(IIdentity.p2.identities, child);
		}
	}
	
	public void removeChildIdentity(IIdentity child) {
		if (identities.contains(child)) {
			identities.remove(child);
			fireElementRemoved(IIdentity.p2.identities, child);
		}
	}
	@Override
	public void update() throws InvalidStateException {
		// Web サービス経由で新規追加。
		CloudEndpoint ce = getCloudResourceManager().getEndpoint(CloudEndpoint.class);

		CloudUser cu = null; 
		try {
			// サーバーから最新の情報を取得。
			cu = ce.getCloudUser(this.getId());
		}
		catch (Exception e) {
			throw new CloudModelException(e);
		}

		// 階層があっているか判定。不変情報なので、ずれている場合、上位のアップデートが必要。
		if (cu.getParentCloudUserId() == null) {
			if (getParent() != null ) {
				// ローカルは、親アカウントを持っているので、不一致。
				throw new CloudModelException();
			}
		}
		else {
			if (getParent() != null) {
				if (!cu.getParentCloudUserId().equals(getParent().getId())) {
					// 親アカウントが一致しない。
					throw new CloudModelException();
				}
			}
			else {
				// ローカルは、親アカウントを持っていないので、不一致。
				throw new CloudModelException();
			}
		}

		this.setName(cu.getCloudUserName());
		this.setDescription(cu.getDescription());
		AccessKey key = new AccessKey();
		key.setAccessKey(cu.getAccessKey());
		key.setSecretKey(cu.getSecretKey());
		this.setAccessKey(key);
		this.setHinemosUserId(cu.getUserId());
		this.setRegDate(new Date(cu.getRegDate()));
		this.setRegUser(cu.getRegUser());
		this.setUpdateDate(new Date(cu.getUpdateDate()));
		this.setUpdateUser(cu.getUpdateUser());
		
		// 子供の情報を更新。
		if (cu.getParentCloudUserId() == null) {
			try {
				Filter f = new Filter();
				f.setName("parentCloudUserId");
				f.getValues().add(cu.getCloudUserId());

				List<CloudUser> users = ce.getCloudUsers(Arrays.asList(f));

				List<Identity> identityList = new ArrayList<Identity>(identities);
				List<Identity> newIdentityList = new ArrayList<Identity>();
				Iterator<CloudUser> userIter = users.iterator();
				while (userIter.hasNext()) {
					CloudUser childCu = userIter.next();

					Iterator<Identity> identityIter = identityList.iterator();
					while (identityIter.hasNext()) {
						Identity id = identityIter.next();
						
						if (childCu.getCloudUserId().equals(id.getId())) {
							newIdentityList.add(id);
							userIter.remove();
							identityIter.remove();
						}
					}
				}

				for (Identity id: identityList) {
					try {
						removeChildIdentity(id);
					}
					catch (Exception e) {
						throw new CloudModelException(e);
					}
				}

				for (CloudUser childCu: users) {
					try {
						Identity child = Identity.createIdentiry(childCu, getCloudResourceManager(), getCategory(), this);
						addChildIdentity(child);
					}
					catch (Exception e) {
						throw new CloudModelException(e);
					}
				}
				
				for (Identity id: identities) {
					id.update();
				}
			}
			catch (Exception e) {
				throw new CloudModelException(e);
			}
		}

		super.update();
	}
	@Override
	public void createChildIdentity(CreateIdentityRequest request) {
		// Web サービス経由で新規追加。
		CloudEndpoint ce = getCloudResourceManager().getEndpoint(CloudEndpoint.class);

		CloudUser cu = new CloudUser();
		cu.setCloudUserName(request.name);
		cu.setCloudUserId(request.id);
		cu.setDescription(request.description);
		cu.setAccessKey(request.key.getAccessKey());
		cu.setSecretKey(request.key.getSecretKey());
		cu.setCloudUserType(CloudUserType.CHILD);
		cu.setParentCloudUserId(getId());
		cu.setUserId(request.hinemosUserId);
		
		try {
			// 追加依頼実施。
			CloudUser newCu = ce.addCloudUser(cu);
			Identity childId = createIdentiry(newCu, getCloudResourceManager(), category, this);
			addChildIdentity(childId);
		}
		catch (Exception e) {
			throw new CloudModelException(e);
		}
	}
	@Override
	public void modifyIdentity(ModifyIdentityRequest request) {
		// Web サービス経由で新規追加。
		CloudEndpoint ce = getCloudResourceManager().getEndpoint(CloudEndpoint.class);

		CloudUser cu = new CloudUser();
		cu.setCloudUserName(request.name);
		cu.setCloudUserId(getId());
		cu.setDescription(request.description);
		cu.setAccessKey(request.key.getAccessKey());
		cu.setSecretKey(request.key.getSecretKey());
		cu.setUserId(request.hinemosUserId);
		
		try {
			// 追加依頼実施。
			CloudUser updateCu = ce.modifyCloudUser(cu);
			
			this.setName(updateCu.getCloudUserName());
			this.setDescription(updateCu.getDescription());
			AccessKey key = new AccessKey();
			key.setAccessKey(updateCu.getAccessKey());
			key.setSecretKey(updateCu.getSecretKey());
			this.setAccessKey(key);

			this.setHinemosUserId(updateCu.getUserId());
			
			this.setRegDate(new Date(updateCu.getRegDate()));
			this.setRegUser(updateCu.getRegUser());
			
			this.setUpdateDate(new Date(updateCu.getUpdateDate()));
			this.setUpdateUser(updateCu.getUpdateUser());

		}
		catch (Exception e) {
			throw new CloudModelException(e);
		}
	}
	@Override
	public void deleteChildIdentity(String childId) {
		// Web サービス経由で新規追加。
		CloudEndpoint ce = getCloudResourceManager().getEndpoint(CloudEndpoint.class);

		try {
			ce.removeCloudUser(childId);
			IIdentity child = getCategory().getIdentityManagement().getIdentity(childId);
			removeChildIdentity(child);
		}
		catch (Exception e) {
			throw new CloudModelException(e);
		}
	}
	
	/**
	 * IIdentity の実装オブジェクトを作成。
	 * Web サービスへのアクセスは行わない。
	 * 
	 * @param cu
	 * @param root
	 * @param category
	 * @param parent
	 * @return
	 */
	public static Identity createIdentiry(CloudUser cu, CloudResourceManager root, Category category, Identity parent) {
		Identity id = new Identity();
		
		id.setId(cu.getCloudUserId());
		id.setName(cu.getCloudUserName());
		id.setDescription(cu.getDescription());
		
		AccessKey key = new AccessKey();
		key.setAccessKey(cu.getAccessKey());
		key.setSecretKey(cu.getSecretKey());
		
		id.setAccessKey(key);
		id.setCategory(category);

		id.setHinemosUserId(cu.getUserId());
		
		id.setRegDate(new Date(cu.getRegDate()));
		id.setRegUser(cu.getRegUser());
		
		id.setUpdateDate(new Date(cu.getUpdateDate()));
		id.setUpdateUser(cu.getUpdateUser());
		
		if (parent != null) {
			// 子として設定。
			id.setParent(parent);
			parent.addChildIdentity(id);
		}

		return id;
	}
}