/*

 Copyright (C) 2011 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.agent.log;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.agent.SendQueue;
import com.clustercontrol.agent.util.AgentProperties;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.ws.agent.OutputBasicInfo;
import com.clustercontrol.ws.monitor.MonitorInfo;

/**
 * ログ転送スレッドを管理するクラス<BR>
 * 
 * 転送対象ログファイル情報を受け取り、ログ転送スレッドを制御します。
 * 
 */
public class LogfileMonitorManager {

	//ロガー
	private Log m_log = LogFactory.getLog(LogfileMonitorManager.class);

	/** ファイルパスとファイルの読み込み状態を保持しているマップ */
	private static ConcurrentHashMap<String, LogfileMonitor> m_logfileMonitorCache =
		new ConcurrentHashMap<String, LogfileMonitor>();

	/** Queue送信  */
	private SendQueue m_sendQueue;

	/** ログファイル監視間隔 */
	private static int m_runInterval = 10000; // 10sec

	/**
	 * ログ転送管理情報クラス<BR>
	 *
	 */
	class LogThreadManageInfo{

		String m_fileName;

		LogfileMonitor m_thread;

		public String getFileName() {
			return m_fileName;
		}

		public LogfileMonitor getThread() {
			return m_thread;
		}

		public LogThreadManageInfo(String fileName) {
			super();
			m_fileName = fileName;
		}

		public void setThread(LogfileMonitor thread) {
			m_thread = thread;
		}
	}

	/**
	 * コンストラクタ
	 * 
	 * @param ejbConnectionManager EJBコネクション管理
	 * @param sendQueue 監視管理Queue送信
	 * @param props ログ転送エージェントプロパティ
	 */
	public LogfileMonitorManager(SendQueue sendQueue) {
		m_sendQueue = sendQueue;
	}

	private static Object globalLock = new Object();

	/**
	 * 監視設定をスレッドに反映します。<BR>
	 * 
	 * @param list 転送対象ログファイル情報一覧
	 */
	public void setLogfileMonitor(ArrayList<MonitorInfo> monitorList) {
		HashMap <String, ArrayList<MonitorInfo>> newMonitorMap =
			new HashMap<String, ArrayList<MonitorInfo>>();

		try {
			String runIntervalStr = AgentProperties.getProperty("run.interval",
					Integer.toString(m_runInterval));
			m_runInterval = Integer.parseInt(runIntervalStr);
		} catch (Exception e) {
			m_log.warn("LogfileThread : " + e.getMessage());
		}
		/*
		 * logfileMonitorはログファイルごとにオブジェクトが生成される。
		 * logfileMonitor.monitorInfoListに監視設定が登録される。
		 * (logfileMonitorとmonitorInfoは1対多の関係)
		 */
		/*
		 * 1. logfileMonitorを生成する。
		 */
		for (MonitorInfo monitorInfo : monitorList) {
			if (monitorInfo.getMonitorFlg() == ValidConstant.TYPE_INVALID) {
				continue;
			}
			String filePath = monitorInfo.getLogfileCheckInfo().getLogfile();

			LogfileMonitor logfileMonitor = m_logfileMonitorCache.get(filePath);
			if(logfileMonitor == null){
				// ファイル監視オブジェクトを生成。
				logfileMonitor = new LogfileMonitor(this, filePath, m_runInterval);
				m_logfileMonitorCache.put(filePath, logfileMonitor);
			}

			ArrayList<MonitorInfo> list = newMonitorMap.get(filePath);
			if (list == null){
				list = new ArrayList<MonitorInfo> ();
				newMonitorMap.put(filePath, list);
			}
			list.add(monitorInfo);
		}

		/*
		 * 2. logfileMonitor.monitorInfoListを登録する。
		 */
		ArrayList<String> noMonitorFileList = new ArrayList<String>();
		for (String filePath : m_logfileMonitorCache.keySet()) {
			LogfileMonitor thread = m_logfileMonitorCache.get(filePath);
			ArrayList<MonitorInfo> list = newMonitorMap.get(filePath);
			thread.setMonitor(list);
			/*
			 * 監視設定が登録されていないファイルは閉じる。
			 */
			if (thread.clean()) {
				noMonitorFileList.add(filePath);
			}
		}
		for (String file : noMonitorFileList) {
			m_logfileMonitorCache.remove(file);
		}
	}

	public void start() {
		LogfileThread thread = new LogfileThread();
		thread.setName("LogFileMonitor");
		thread.start();
	}

	private class LogfileThread extends Thread {
		@Override
		public void run() {
			m_log.info("run LogfileThread");
			while (true) {
				try {
					for (String filePath : m_logfileMonitorCache.keySet()) {
						LogfileMonitor logfileMonitor = m_logfileMonitorCache.get(filePath);
						logfileMonitor.run();
					}
				} catch (Exception e) {
					m_log.warn("LogfileThread : " + e.getClass().getCanonicalName() + ", " +
							e.getMessage(), e);
				} catch (Throwable e) {
					m_log.error("LogfileThread : " + e.getClass().getCanonicalName() + ", " +
							e.getMessage(), e);
				}
				try {
					Thread.sleep(m_runInterval);
				} catch (InterruptedException e) {
					m_log.info("LogfileThread is Interrupted");
					break;
				}
			}
		}
	}

	/**
	 * 監視管理のJMSに情報を通知します。<BR>
	 * 
	 * @param priority 重要度
	 * @param app アプリケーション
	 * @param msgId メッセージID
	 * @param msg メッセージ
	 * @param msgOrg オリジナルメッセージ
	 */
	public void sendMessage(String filePath, int priority, String app, String msgId, String msg, String msgOrg, String monitorId) {
		// ログ出力情報
		OutputBasicInfo output = new OutputBasicInfo();
		output.setPluginId(HinemosModuleConstant.MONITOR_LOGFILE);
		output.setPriority(priority);
		output.setApplication(app);
		output.setMessageId(msgId);
		output.setMessage(msg);
		output.setMessageOrg(msgOrg);

		output.setGenerationDate(new Date().getTime());
		output.setMonitorId(monitorId);
		output.setFacilityId(""); // マネージャがセットする。
		output.setScopeText(""); // マネージャがセットする。

		this.m_sendQueue.put(output);
	}

}
