#!/bin/bash

# Copyright (C) since 2006 NTT DATA Corporation
# 
# This program is free software; you can redistribute it and/or
# Modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation, version 2.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details

# bash configuration
SCRIPT_DIR=$(cd $(dirname $0);pwd)
. ${SCRIPT_DIR}/../hinemos.cfg
. ${SCRIPT_DIR}/../sbin/hinemos_utility.sh

export PROG=`basename $0`
DIR=`dirname $0`
USER=`/usr/bin/whoami`
HOST=`hostname`

cd ${HINEMOS_HOME}/var/log

if [ "x${JBOSS_FD_MAXNUM}" != "x" ]
then
	ulimit -n ${JBOSS_FD_MAXNUM}
fi

if [ "x${NET_CORE_RMEM_MAX}" != "x" ]
then
	sysctl -w net.core.rmem_max=${NET_CORE_RMEM_MAX}
fi

if [ "x${NET_CORE_RMEM_DEF}" != "x" ]
then
	sysctl -w net.core.rmem_default=${NET_CORE_RMEM_DEF}
fi

########################################
#  Local Variable
########################################

# startup check interval [sec]
#   this variable is valid with -W option
JBOSS_CHECK_INTERVAL=3

# startup check timeout [sec]
#   this variable is valid with -W option
JBOSS_CHECK_TIMEOUT=300

# log file for monitor startup
STARTUP_MONITOR_FILE=${HINEMOS_HOME}/var/log/startup.log

# temporary export file dir
EXPORT_TEMPORARY_DIR=${HINEMOS_HOME}/var/export

########################################
#  Local Message
########################################

# INFO
MSG_I001="waiting for JBoss to start..."
MSG_I002="JBoss configuration assigned"
MSG_I003="removing temporally data (JMS) before JBoss start"
MSG_I004="JBoss started"
MSG_I005="starting Hinemos Manager(maintenance-mode)"
MSG_I006="removing temporally data (Export) before JBoss start"

# WARN
MSG_W001="WARN : PID file(/var/run/snmptrapd.pid) is found. stop snmptrapd service."

# ERROR
MSG_E001="usage : ${PROG} [-a|-c|-M|-W]"
MSG_E002="incorrect user (${USER})";
MSG_E003="configuration of high availability not found"
MSG_E004="PID file (${HINEMOS_JBOSS_PID}) exists. JBoss may be running."
MSG_E005="startup timeout. please check status of JBoss"
MSG_E006="PID file (${HINEMOS_PG_PID}) not found. PostgreSQL may be stopped."
MSG_E007="please check variable HINEMOS_JBOSS_USER (${HINEMOS_HOME}/hinemos.cfg)."

########################################
# Function
########################################


########################################
# SHELL
########################################

#
# check argument
#
OPTION=""
JMS_CLEANUP="false"
JBOSS_STARTUP_WAIT="true"
for ARGS in "$@"
do
	case ${ARGS} in
		-a)
			# for HA Option
			if [ ! -d ${JBOSS_HOME}/server/hinemos_ha ]
			then
				Logging "${MSG_E003}"
				exit -5
			fi
			OPTION="-c all"
			;;
		-c)
			JMS_CLEANUP="true"
			;;
		-M)
			Logging "${MSG_I005}"
			export JAVA_OPTS="-Dhinemos.manager.mode=maintenance"
			;;
		-W)
			JBOSS_STARTUP_WAIT="false"
			;;
		*)
			Logging "${MSG_E001}"
			exit -1
			;;
	esac
done

#
# check user executed
#
if [ "x"${USER} != "x"${HINEMOS_JBOSS_USER} ]
then
	Logging "${MSG_E002}"
	Logging "${MSG_E007}"
	exit -2
fi

#
# check process
#
if [ ! -f ${HINEMOS_PG_PID} ]
then
	Logging "${MSG_E006}"
	exit -6
fi
if [ -f /var/run/snmptrapd.pid ]
then
	Logging "${MSG_W001}" && echo
fi

if [ -f ${HINEMOS_JBOSS_PID} ]
then
	read JBOSS_PID < ${HINEMOS_JBOSS_PID}
	if [ "x"${JBOSS_PID} != "x" ] && [ `ps --no-headers --pid ${JBOSS_PID} e | grep "${JAVA_HOME}/bin/java.*org.jboss.Main" | wc -l` -gt 0 ]
	then
		Logging "${MSG_E004}"
		exit -7
	else
		rm -f ${HINEMOS_JBOSS_PID}
	fi
fi

#
# set composition
#
if [ -d ${JBOSS_HOME}/server/hinemos_ha ]
then
	CONFIGURATION="hinemos_ha"
else
	CONFIGURATION="hinemos"
fi

Logging "${MSG_I002} : ${CONFIGURATION}"

#
# cleanup JMS
#
if [ "x${JMS_CLEANUP}" = "xtrue" ]
then
	echo
	Logging "${MSG_I003}"
	${HINEMOS_HOME}/sbin/mng/hinemos_clear_tmp.sh
fi

#
# startup JBoss
#
echo && Logging "${MSG_I006}"
rm -Rf ${EXPORT_TEMPORARY_DIR}/*

echo && Logging "${MSG_I001}"

nohup ${JBOSS_HOME}/bin/run_hinemos.sh ${OPTION} -c ${CONFIGURATION} -b ${IP_ADDRESS} > ${HINEMOS_JBOSS_STDOUTERR} 2>&1 &

# wait for JBoss to start
TIME_COUNT=0
rm -f ${STARTUP_MONITOR_FILE}

while [ 1 ]
do
	if [ "x${JBOSS_STARTUP_WAIT}" = "xfalse" ]
	then
			Logging "${MSG_I004} (with -W option)"
			break
	fi
	
	if [ -f ${STARTUP_MONITOR_FILE} ]
	then
		if [ `grep -a 'JBoss (Microcontainer).*Started' ${STARTUP_MONITOR_FILE} | wc -l` -gt 0 ]
		then
			echo "done"
			Logging "${MSG_I004}"
			break
		fi
	fi

	if [ "${TIME_COUNT}" -ge ${JBOSS_CHECK_TIMEOUT} ]
	then
		Logging "${MSG_E005}"
		exit -8
	fi

	sleep ${JBOSS_CHECK_INTERVAL}
	TIME_COUNT=$((${TIME_COUNT} + ${JBOSS_CHECK_INTERVAL}))
	echo -n "."
done

exit 0
