/*
Copyright (C) 2010 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.ws.endpoint;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;

import javax.annotation.Resource;
import javax.ejb.CreateException;
import javax.naming.NamingException;
import javax.xml.ws.WebServiceContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jnp.interfaces.NamingContext;

import com.clustercontrol.fault.FacilityNotFound;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.fault.InvalidUserPass;
import com.clustercontrol.fault.NotifyDuplicate;
import com.clustercontrol.fault.NotifyNotFound;
import com.clustercontrol.accesscontrol.bean.RoleConstant;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.notify.bean.NotifyInfo;
import com.clustercontrol.notify.ejb.session.NotifyControllerLocal;
import com.clustercontrol.notify.ejb.session.NotifyControllerLocalHome;
import com.clustercontrol.notify.util.NotifyCache;
import com.clustercontrol.notify.util.NotifyRelationCache;

/**
 * 通知用のWebAPIエンドポイント
 */
@javax.jws.WebService(targetNamespace = "http://notify.ws.clustercontrol.com")
public class NotifyEndpoint {
	@Resource
	WebServiceContext wsctx;

	private static Log m_log = LogFactory.getLog( NotifyEndpoint.class );
	private static Log m_opelog = LogFactory.getLog("HinemosOperation");

	/**
	 * echo(WebサービスAPI疎通用)
	 * 
	 * 権限必要なし（ユーザ名チェックのみ実施）
	 * 
	 * @param str
	 * @return
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 */
	public String echo(String str) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		ArrayList<String> roleList = new ArrayList<String>();
		HttpAuthenticator.authCheck(wsctx, roleList);

		return str + ", " + str;
	}

	/**
	 * 通知用のSessionBeanの取得
	 * 
	 * @return
	 * @throws HinemosUnknown
	 * @throws InvalidUserPass
	 */
	private NotifyControllerLocal getNotifyControllerLocal() throws HinemosUnknown, InvalidUserPass {
		NotifyControllerLocal local = null;
		try {
			String account = HttpAuthenticator.getAccount(wsctx);
			NamingContext namingContext = HttpAuthenticator.getContext(account);
			NotifyControllerLocalHome localHome =
				(NotifyControllerLocalHome)
				namingContext.lookup(NotifyControllerLocalHome.JNDI_NAME);
			local = localHome.create();
		} catch (CreateException e) {
			m_log.error("getNotifyControllerLocal CreateException : " + e,e);
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error("getNotifyControllerLocal NamingException : " + e,e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return local;
	}

	/**
	 * 通知情報を作成します。
	 * 
	 * NotifyWrite権限が必要
	 * 
	 * @param info 作成対象の通知情報
	 * @return 作成に成功した場合、<code> true </code>
	 * @throws NotifyDuplicate
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * 
	 * @see com.clustercontrol.notify.factory.AddNotify#add(NotifyInfo)
	 */
	public boolean addNotify(NotifyInfo info) throws HinemosUnknown, NotifyDuplicate, InvalidUserPass, InvalidRole,InvalidSetting {
		m_log.debug("addNotify");

		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.NOTIFY_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if(info != null){
			StringBuffer msg = new StringBuffer();
			msg.append(", NotifyID=");
			msg.append(info.getNotifyId());
			m_opelog.info(HinemosModuleConstant.LOG_PREFIX_NOTIFY + " Add, Method=addNotify, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}

		try {
			boolean flag = getNotifyControllerLocal().addNotify(info);
			return flag;
		} finally {
			// FIXME:本来はSessionBeanに実装すべき
			// トランザクションのコミット後にキャッシュをクリアする。
			NotifyCache.refresh();
			NotifyRelationCache.refresh();
		}
	}

	/**
	 * 通知情報を変更します。
	 * 
	 * NotifyWrite権限が必要
	 * 
	 * @param info 変更対象の通知情報
	 * @return 変更に成功した場合、<code> true </code>
	 * @throws HinemosUnknown
	 * @throws NotifyDuplicate
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * 
	 * @see com.clustercontrol.notify.factory.ModifyNotify#modify(NotifyInfo)
	 */
	public boolean modifyNotify(NotifyInfo info) throws NotifyDuplicate, HinemosUnknown, InvalidUserPass, InvalidRole,InvalidSetting {
		m_log.debug("modifyNotify");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.NOTIFY_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if(info != null){
			StringBuffer msg = new StringBuffer();
			msg.append(", NotifyID=");
			msg.append(info.getNotifyId());
			m_opelog.info(HinemosModuleConstant.LOG_PREFIX_NOTIFY + " Change, Method=modifyNotify, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}

		try {
			boolean flag = getNotifyControllerLocal().modifyNotify(info);
			return flag;
		} finally {
			// FIXME:本来はSessionBeanに実装すべき
			// トランザクションのコミット後にキャッシュをクリアする。
			NotifyCache.refresh();
			NotifyRelationCache.refresh();
		}
	}

	/**
	 * 通知情報を削除します。
	 * 
	 * NotifyWrite権限が必要
	 * 
	 * @param notifyId 削除対象の通知ID
	 * @return 削除に成功した場合、<code> true </code>
	 * @throws HinemosUnknown
	 * @throws NotifyNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * 
	 * @see com.clustercontrol.notify.factory.DeleteNotify#delete(String)
	 */
	public boolean deleteNotify(String notifyId) throws NotifyNotFound, HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("deleteNotify");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.NOTIFY_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", NotifyID=");
		msg.append(notifyId);
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_NOTIFY + " Delete, Method=deleteNotify, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		try {
			boolean flag = getNotifyControllerLocal().deleteNotify(notifyId);
			return flag;
		} finally {
			// FIXME:本来はSessionBeanに実装すべき
			// トランザクションのコミット後にキャッシュをクリアする。
			NotifyCache.refresh();
			NotifyRelationCache.refresh();
		}
	}

	/**
	 * 引数で指定された通知情報を返します。
	 * 
	 * NotifyRead権限が必要
	 * 
	 * @param notifyId 取得対象の通知ID
	 * @return 通知情報
	 * @throws HinemosUnknown
	 * @throws NotifyNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * 
	 * @see com.clustercontrol.notify.factory.SelectNotify#getNotify(String)
	 */
	public NotifyInfo getNotify(String notifyId) throws NotifyNotFound, HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("getNotify");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.NOTIFY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", NotifyID=");
		msg.append(notifyId);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_NOTIFY + " Get, Method=getNotify, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return getNotifyControllerLocal().getNotify(notifyId);
	}

	/**
	 * 通知情報一覧を返します。
	 * 
	 * NotifyRead権限が必要
	 * 
	 * @return 通知情報一覧（Objectの2次元配列）
	 * @throws HinemosUnknown
	 * @throws NotifyNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * 
	 * @see com.clustercontrol.notify.factory.SelectNotify#getNotifyList()
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<NotifyInfo> getNotifyList() throws NotifyNotFound, HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("getNotifyList");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.NOTIFY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_NOTIFY + " Get, Method=getNotifyList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx));

		return getNotifyControllerLocal().getNotifyList();
	}

	/**
	 *　引数で指定した通知IDを利用している通知グループIDを取得する。
	 * 
	 * NotifyRead権限が必要
	 * 
	 * @param notifyId
	 * @return　通知グループIDのリスト
	 * @throws NotifyNotFound
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws InvalidRoleException
	 * @throws InvalidUserPassException
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<String> checkNotifyId(String notifyId) throws NotifyNotFound, HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("checkNotifyId");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.NOTIFY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", NotifyID=");
		msg.append(notifyId);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_NOTIFY + " Check, Method=checkNotifyId, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return getNotifyControllerLocal().checkNotifyId(notifyId);
	}

	/**
	 *　指定した通知IDを有効化する。(現在の有効/無効の判定なし)
	 *
	 * NotifyWrite権限が必要
	 *
	 * @param notifyId 通知ID
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws HinemosUnknown
	 * @throws NotifyNotFound
	 * @throws NotifyDuplicate
	 * @throws InvalidRoleException
	 * @throws InvalidUserPassException
	 */
	public void setNotifyStatus(String notifyId, boolean validFlag) throws InvalidUserPass, InvalidRole, HinemosUnknown, NotifyNotFound, NotifyDuplicate {
		m_log.debug("enableNotify() notifyId = " + notifyId);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.NOTIFY_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", NotifyID=");
		msg.append(notifyId);
		msg.append(", ValidFlag=");
		msg.append(validFlag);
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_NOTIFY + " Change Valid, Method=setNotifyStatus, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		try {
			getNotifyControllerLocal().setNotifyStatus(notifyId, validFlag);
		} finally {
			// FIXME:本来はSessionBeanに実装すべき
			// トランザクションのコミット後にキャッシュをクリアする。
			NotifyCache.refresh();
			NotifyRelationCache.refresh();
		}
	}

	/**
	 * 外部から直接通知処理を実行します。

	 * 
	 * @param pluginId プラグインID
	 * @param monitorId 監視項目ID
	 * @param facilityId ファシリティID
	 * @param subKey 抑制用のサブキー（任意の文字列）
	 * @param generationDate 出力日時（エポック秒）
	 * @param priority 重要度
	 * @param application アプリケーション
	 * @param messageId メッセージID
	 * @param message メッセージ
	 * @param messageOrg オリジナルメッセージ
	 * @param notifyIdList 通知IDのリスト
	 * @param srcId 送信元を特定するためのID
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws HinemosUnknown
	 * @throws NotifyNotFound
	 * @throws FacilityNotFound
	 */
	public void notify(
			String pluginId,
			String monitorId,
			String facilityId,
			String subKey,
			long generationDate,
			int priority,
			String application,
			String messageId,
			String message,
			String messageOrg,
			ArrayList<String> notifyIdList,
			String srcId)  throws InvalidRole, InvalidUserPass, HinemosUnknown, NotifyNotFound, FacilityNotFound {
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");
		String msg = ", pluginID=" + pluginId
		+ ", monitorID=" + monitorId
		+ ", facilityID=" + facilityId
		+ ", subKey=" + subKey
		+ ", generationDate=" + sdf.format(new Date(generationDate))
		+ ", priority=" + priority
		+ ", application=" + application
		+ ", messageID=" + messageId
		+ ", message=" + message
		+ ", messageOrg=" + messageOrg
		+ ", srcID=" + srcId;

		m_log.debug("notify() " + msg);

		// 認証済み操作ログ
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_NOTIFY + " Run, Method=notify, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		getNotifyControllerLocal().notify(
				pluginId,
				monitorId,
				facilityId,
				subKey,
				generationDate,
				priority,
				application,
				messageId,
				message,
				messageOrg,
				notifyIdList,
				srcId);
	}}