/*
Copyright (C) 2010 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.ws.endpoint;

import java.util.ArrayList;
import java.util.Collection;

import javax.annotation.Resource;
import javax.ejb.CreateException;
import javax.naming.NamingException;
import javax.xml.ws.WebServiceContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jnp.interfaces.NamingContext;

import com.clustercontrol.fault.BgFileNotFound;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.IconFileNotFound;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.InvalidUserPass;
import com.clustercontrol.accesscontrol.bean.RoleConstant;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.nodemap.NodeMapException;
import com.clustercontrol.nodemap.bean.NodeMapModel;
import com.clustercontrol.nodemap.ejb.session.NodeMapControllerLocal;
import com.clustercontrol.nodemap.ejb.session.NodeMapControllerLocalHome;

/**
 * ノードマップ用のWebAPIエンドポイント
 */
@javax.jws.WebService(targetNamespace = "http://nodemap.ws.clustercontrol.com")
public class NodeMapEndpoint {
	@Resource
	WebServiceContext wsctx;

	private static Log m_log = LogFactory.getLog( NodeMapEndpoint.class );
	private static Log m_opelog = LogFactory.getLog("HinemosOperation");

	/**
	 * echo(WebサービスAPI疎通用)
	 * 
	 * 権限必要なし（ユーザ名チェックのみ実施）
	 * 
	 * @param str
	 * @return
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 */
	public String echo(String str) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		ArrayList<String> roleList = new ArrayList<String>();
		HttpAuthenticator.authCheck(wsctx, roleList);

		return str + ", " + str;
	}

	/**
	 * ノードマップ用のSessionBeanの取得
	 * @return
	 * @throws HinemosUnknown
	 * @throws InvalidUserPass
	 */
	private NodeMapControllerLocal getNodeMapControllerLocal() throws HinemosUnknown, InvalidUserPass {
		NodeMapControllerLocal local = null;
		try {
			String account = HttpAuthenticator.getAccount(wsctx);
			NamingContext namingContext = HttpAuthenticator.getContext(account);
			NodeMapControllerLocalHome localHome =
				(NodeMapControllerLocalHome)
				namingContext.lookup(NodeMapControllerLocalHome.JNDI_NAME);
			local = localHome.create();
		} catch (CreateException e) {
			m_log.error("getNodeMapControllerLocal CreateException : " + e,e);
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error("getNodeMapControllerLocal NamingException : " + e,e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return local;
	}
	/**
	 * リポジトリ情報からマップのデータを生成します。<BR>
	 * 
	 * RepositoryRead権限が必要
	 * 
	 * @param parentFacilityId 描画対象スコープの親のファシリティID
	 * @param facilityId 描画対象スコープのファシリティID
	 * @return マップ描画用データ
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * 
	 * @see com.clustercontrol.nodemap.model.NodeMapModel
	 */
	public NodeMapModel createNodeMapModel(String facilityId) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("createNodeMapModel");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", FacilityID=");
		msg.append(facilityId);
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_NODEMAP + " Add Map, Method=createNodeMapModel, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return getNodeMapControllerLocal().createNodeMapModel(facilityId);
	}

	/**
	 * マップのデータをDBに登録します。<BR>
	 * 
	 * RepositoryWrite権限が必要
	 * 
	 * @param facilityId 描画対象スコープのファシリティID
	 * @return マップ描画用データ
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * 
	 * @see com.clustercontrol.nodemap.model.NodeMapModel
	 */
	public void registerNodeMapModel(NodeMapModel map) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("registerNodeMapModel");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if (map != null) {
			StringBuffer msg = new StringBuffer();
			msg.append(", MapID=");
			msg.append(map.getMapId());
			m_opelog.info(HinemosModuleConstant.LOG_PREFIX_NODEMAP + " Set Map, Method=registerNodeMapModel, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}

		getNodeMapControllerLocal().registerNodeMapModel(map);
	}

	/**
	 * マップのデータを取得します。<BR>
	 * 
	 * RepositoryRead権限が必要
	 * 
	 * @param facilityId 描画対象スコープのファシリティID
	 * @return マップ描画用データ
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws NodeMapException
	 * 
	 * @see com.clustercontrol.nodemap.model.NodeMapModel
	 */
	public NodeMapModel getNodeMapModel(String facilityId) throws InvalidUserPass, InvalidRole, HinemosUnknown, NodeMapException {
		m_log.debug("getNodeMapModel");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", FacilityID=");
		msg.append(facilityId);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_NODEMAP + " Get, Method=getNodeMapModel, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return getNodeMapControllerLocal().getNodeMapModel(facilityId);
	}


	/**
	 * 背景データを取得します。<BR>
	 * 
	 * RepositoryRead権限が必要
	 * 
	 * @param filename
	 * @return filedata
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws BgFileNotFound
	 */
	public byte[] getBgImage(String filename) throws InvalidUserPass, InvalidRole, HinemosUnknown, BgFileNotFound {
		m_log.debug("getBgImage");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", FileName=");
		msg.append(filename);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_NODEMAP + " Get, Method=getBgImage, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return getNodeMapControllerLocal().getBgImage(filename);
	}

	/**
	 * 背景データをDBに登録します。<BR>
	 * 
	 * RepositoryWrite権限が必要
	 * 
	 * @param filename
	 * @param filedata
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws HinemosUnknown
	 * @throws NodeMapException
	 */
	public void setBgImage(String filename, byte[] filedata) throws InvalidUserPass, InvalidRole, HinemosUnknown, NodeMapException {
		m_log.debug("setBgImage");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", FileName=");
		msg.append(filename);
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_NODEMAP + " Set Image, Method=setBgImage, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		// TODO byte配列のサイズが大きくなる可能性あり。MTOMで実装したほうがよいかも。
		getNodeMapControllerLocal().setBgImage(filename, filedata);
	}

	/**
	 * 背景画像のファイル名一覧を取得します。<BR>
	 * 
	 * RepositoryRead権限が必要
	 * 
	 * @return Collection<String>
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	@SuppressWarnings("unchecked")
	public Collection<String> getBgImagePK() throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("getBgImagePK");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_NODEMAP + " Get, Method=getBgImagePK, User="
				+ HttpAuthenticator.getUserAccountString(wsctx));

		return getNodeMapControllerLocal().getBgImagePK();
	}

	/**
	 * 背景画像のファイル名の存在有無を取得します。<BR>
	 * 
	 * RepositoryRead権限が必要
	 * 
	 * @return boolean
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public boolean isBgImage(String filename) throws HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("isBgImage");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", FileName=");
		msg.append(filename);
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_NODEMAP + " Get, Method=isBgImage, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return getNodeMapControllerLocal().isBgImage(filename);
	}

	/**
	 * アイコン画像を取得します。<BR>
	 * 
	 * RepositoryRead権限が必要
	 * 
	 * @param filename
	 * @return filedata
	 * @throws IconFileNotFound
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public byte[] getIconImage(String filename) throws HinemosUnknown, IconFileNotFound, InvalidUserPass, InvalidRole {
		m_log.debug("getIconImage");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", FileName=");
		msg.append(filename);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_NODEMAP + " Get, Method=getIconImage, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return getNodeMapControllerLocal().getIconImage(filename);
	}

	/**
	 * アイコン画像をDBに登録します。<BR>
	 * 
	 * RepositoryWrite権限が必要
	 * 
	 * @param filename
	 * @param filedata
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws NodeMapException
	 * 
	 */
	public void setIconImage(String filename, byte[] filedata) throws InvalidUserPass, InvalidRole, HinemosUnknown, NodeMapException {
		m_log.debug("setIconImage");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", FileName=");
		msg.append(filename);
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_NODEMAP + " Set Image, Method=setIconImage, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		getNodeMapControllerLocal().setIconImage(filename, filedata);
	}

	/**
	 * アイコンのファイル名一覧を取得します。<BR>
	 * 
	 * RepositoryRead権限が必要
	 * 
	 * @return Collection<String>
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	@SuppressWarnings("unchecked")
	public Collection<String> getIconImagePK() throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("getIconImagePK");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_NODEMAP + " Get, Method=getIconImagePK, User="
				+ HttpAuthenticator.getUserAccountString(wsctx));

		return getNodeMapControllerLocal().getIconImagePK();
	}

	/**
	 * アイコンのファイル名の存在有無を取得します。<BR>
	 * 
	 * RepositoryRead権限が必要
	 * 
	 * @return boolean
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public boolean isIconImage(String filename) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("isIconImage");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", FileName=");
		msg.append(filename);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_NODEMAP + " Get, Method=isIconImage, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return getNodeMapControllerLocal().isIconImage(filename);
	}
}