/*
Copyright (C) 2010 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.ws.endpoint;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;

import javax.annotation.Resource;
import javax.ejb.CreateException;
import javax.naming.NamingException;
import javax.xml.ws.WebServiceContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jnp.interfaces.NamingContext;

import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.fault.InvalidUserPass;
import com.clustercontrol.fault.UserDuplicate;
import com.clustercontrol.fault.UserNotFound;
import com.clustercontrol.accesscontrol.bean.RoleConstant;
import com.clustercontrol.accesscontrol.bean.UserFilterInfo;
import com.clustercontrol.accesscontrol.bean.UserInfo;
import com.clustercontrol.accesscontrol.ejb.session.AccessControllerLocal;
import com.clustercontrol.accesscontrol.ejb.session.AccessControllerLocalHome;
import com.clustercontrol.accesscontrol.util.VersionUtil;
import com.clustercontrol.bean.HinemosModuleConstant;

/**
 * アクセス用のWebAPIエンドポイント
 */
@javax.jws.WebService(targetNamespace = "http://access.ws.clustercontrol.com")
public class AccessEndpoint {
	@Resource
	WebServiceContext wsctx;

	private static Log m_log = LogFactory.getLog( AccessEndpoint.class );
	private static Log m_opelog = LogFactory.getLog("HinemosOperation");

	/**
	 * echo(WebサービスAPI疎通用)
	 * 
	 * 権限必要なし（ユーザ名チェックのみ実施）
	 * 
	 * @param str
	 * @return
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 */
	public String echo(String str) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		ArrayList<String> roleList = new ArrayList<String>();
		HttpAuthenticator.authCheck(wsctx, roleList);

		return str + ", " + str;
	}

	/**
	 * ユーザ管理用のSessionBeanの取得
	 * 
	 * @return
	 * @throws InvalidUserPass
	 * @throws HinemosUnknown
	 */
	private AccessControllerLocal getAccessControllerLocal() throws InvalidUserPass, HinemosUnknown {
		AccessControllerLocal access = null;
		try {
			String account = HttpAuthenticator.getAccount(wsctx);
			NamingContext namingContext = HttpAuthenticator.getContext(account);
			AccessControllerLocalHome localHome =
				(AccessControllerLocalHome)
				namingContext.lookup(AccessControllerLocalHome.JNDI_NAME);
			access = localHome.create();
		} catch (CreateException e) {
			m_log.error("getAccessControllerLocal CreateException : " + e,e);
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error("getAccessControllerLocal NamingException : " + e,e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return access;
	}

	/**
	 * ログインチェックの為、本メソッドを使用します。
	 * 
	 * 権限必要なし（ユーザ名チェックのみ実施）
	 * 
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * 
	 * @ejb.interface-method
	 */
	public void checkLogin() throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("checkLogin");
		ArrayList<String> roleList = new ArrayList<String>();
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_ACCESS + " Login, Method=checkLogin, User="
				+ HttpAuthenticator.getUserAccountString(wsctx));

		getAccessControllerLocal().checkLogin();
	}

	/**
	 * ユーザ検索条件に基づき、ユーザ一覧情報を取得する。<BR>
	 * 
	 * AccessControlRead権限が必要
	 * 
	 * @param filter ユーザ検索条件（ユーザフィルタダイアログの入力値）
	 * @return ユーザ情報のリスト
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws UserNotFound
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.LoginUserSelector#getUserList(Property)
	 */
	public ArrayList<UserInfo> getUserInfoList(UserFilterInfo filter) throws InvalidUserPass, InvalidRole, HinemosUnknown, UserNotFound {
		m_log.debug("getUserInfoList");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.ACCESSCONTROL_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if(filter != null){
			StringBuffer msg = new StringBuffer();
			msg.append(", UserID=");
			msg.append(filter.getUserId());
			msg.append(", UserName=");
			msg.append(filter.getUserName());
			msg.append(", Description=");
			msg.append(filter.getDescription());
			msg.append(", Role=");
			msg.append(Arrays.toString(filter.getRoleList().toArray()));

			m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_ACCESS + " Get, Method=getUserInfoList, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		} else {
			m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_ACCESS + " Get, Method=getUserInfoListAll, User="
					+ HttpAuthenticator.getUserAccountString(wsctx) );
		}

		return getAccessControllerLocal().getUserInfoList(filter);
	}

	/**
	 * 自身のユーザ情報を取得する。<BR>
	 * 
	 * AccessControlRead権限が必要
	 * 
	 * @return 自身のユーザ情報
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws UserNotFound
	 * 
	 */
	public UserInfo getOwnUserInfo() throws InvalidUserPass, InvalidRole, HinemosUnknown, UserNotFound {
		m_log.debug("getOwnUserInfo");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_ACCESS + " Get, Method=getOwnUserInfo, User="
				+ HttpAuthenticator.getUserAccountString(wsctx));

		return getAccessControllerLocal().getOwnUserInfo();
	}

	/**
	 * ユーザ情報を取得する。<BR>
	 * 
	 * AccessControlRead権限が必要
	 * 
	 * @param userId ユーザID
	 * @param mode 取得モード
	 * @param locale ロケール情報
	 * @return ユーザ情報
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * 
	 * @see com.clustercontrol.bean.PropertyConstant
	 * @see com.clustercontrol.accesscontrol.factory.UserProperty#getProperty(String, int, Locale)
	 */
	public UserInfo getUserInfo(String userId) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("getUserInfo");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.ACCESSCONTROL_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", UserID=");
		msg.append(userId);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_ACCESS + " Get, Method=getUserInfo, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return getAccessControllerLocal().getUserInfo(userId);
	}

	/**
	 * ユーザを追加する。<BR>
	 * 
	 * AccessControlWrite権限が必要
	 * 
	 * @param info ユーザ情報
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws HinemosUnknown
	 * @throws UserDuplicate
	 * @throws ParseException
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.LoginUserModifier#addUser(Property, String, String)
	 */
	public void addUserInfo(UserInfo info) throws InvalidUserPass, InvalidRole, HinemosUnknown, UserDuplicate, InvalidSetting {
		m_log.debug("addUserInfo");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.ACCESSCONTROL_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if(info != null){
			StringBuffer msg = new StringBuffer();
			msg.append(", UserID=");
			msg.append(info.getId());
			m_opelog.info(HinemosModuleConstant.LOG_PREFIX_ACCESS + " Add User, Method=addUser, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}

		getAccessControllerLocal().addUserInfo(info);
	}

	/**
	 * ユーザ情報を変更する。<BR>
	 * 
	 * AccessControlWrite権限が必要
	 * 
	 * @param info ユーザ情報
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws UserNotFound
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.LoginUserModifier#modifyUser(Property, String)
	 */
	public void modifyUserInfo(UserInfo info) throws InvalidUserPass, InvalidRole, HinemosUnknown, UserNotFound, InvalidSetting {
		m_log.debug("modifyUserInfo");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.ACCESSCONTROL_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if(info != null){
			StringBuffer msg = new StringBuffer();
			msg.append(", UserID=");
			msg.append(info.getId());
			m_opelog.info(HinemosModuleConstant.LOG_PREFIX_ACCESS + " Change User, Method=modifyUser, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}

		getAccessControllerLocal().modifyUserInfo(info);
	}

	/**
	 * ユーザ情報を削除する。<BR>
	 * 
	 * AccessControlWrite権限が必要
	 * 
	 * @param userId ユーザID
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws UserNotFound
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.LoginUserModifier#deleteUser(String, String)
	 */
	public void deleteUserInfo(String userId) throws InvalidUserPass, InvalidRole, HinemosUnknown, UserNotFound {
		m_log.debug("deleteUserInfo");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.ACCESSCONTROL_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", UserID=");
		msg.append(userId);
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_ACCESS + " Delete User, Method=deleteUser, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		getAccessControllerLocal().deleteUserInfo(userId);
	}

	/**
	 * 自分自身のパスワードを変更する。<BR>
	 * 
	 * AccessControlRead権限が必要
	 * 
	 * @param userId ユーザID
	 * @param password パスワード
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws UserNotFound
	 * 
	 */
	public void changeOwnPassword(String password) throws InvalidUserPass, InvalidRole, HinemosUnknown, UserNotFound {
		m_log.debug("changeOwnPassword");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_ACCESS + " Change Password, Method=changeOwnPassword, User="
				+ HttpAuthenticator.getUserAccountString(wsctx));

		getAccessControllerLocal().changeOwnPassword(password);
	}

	/**
	 * パスワードを変更する。<BR>
	 * 
	 * AccessControlWrite権限が必要
	 * 
	 * @param userId ユーザID
	 * @param password パスワード
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws UserNotFound
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.LoginUserModifier#modifyUserPassword(String, String)
	 */
	public void changePassword(String userId, String password) throws InvalidUserPass, InvalidRole, HinemosUnknown, UserNotFound {
		m_log.debug("changePassword");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.ACCESSCONTROL_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", UserID=");
		msg.append(userId);
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_ACCESS + " Change Password, Method=changePassword, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		getAccessControllerLocal().changePassword(userId, password);
	}

	/**
	 * ログインしているユーザが指定したユーザ権限を持っているかどうかを確認する。<BR>
	 * 
	 * AccessControlRead権限が必要
	 * 
	 * @param userRole ユーザ権限名
	 * @return ユーザ権限を保持していればtrue, そうでなければfalse
	 * @throws InvalidUserPass
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 */
	public boolean isPermission(String userRole) throws HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("isPermission");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", UserRole=");
		msg.append(userRole);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_ACCESS + " Check Permission, Method=isPermission, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return getAccessControllerLocal().isPermission(userRole);
	}

	/**
	 * ログインユーザのユーザ名を取得する。<BR>
	 * 
	 * AccessControlRead権限が必要
	 * 
	 * @return ユーザ名
	 * @throws HinemosUnknown
	 * @throws UserNotFound
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.LoginUserSelector#getUserName(String)
	 */
	public String getUserName() throws HinemosUnknown, UserNotFound, InvalidUserPass, InvalidRole {
		m_log.debug("getUserName");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.REPOSITORY_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_ACCESS + " Get, Method=getUserName, User="
				+ HttpAuthenticator.getUserAccountString(wsctx));

		return getAccessControllerLocal().getUserName();
	}

	/**
	 * 引数で指定されたユーザがジョブユニットの管理ユーザに設定されているか確認する<BR>
	 * 
	 * AccessControlWrite権限が必要
	 * 
	 * @param userId ユーザID
	 * @return 引数で指定されたユーザが管理ユーザに設定されているジョブユニットIDのリスト
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws UserNotFound
	 * 
	 * @see com.clustercontrol.bean.TableColumnInfo
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<String> checkJobManagementUser(String userId) throws InvalidUserPass, InvalidRole, HinemosUnknown, UserNotFound {
		m_log.debug("checkJobManagementUser");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.ACCESSCONTROL_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", UserID=");
		msg.append(userId);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_ACCESS + " Check Permission, Method=checkJobManagementUser, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return getAccessControllerLocal().checkJobManagementUser(userId);
	}

	/**
	 * バージョン番号を取得する（権限必要なし）。
	 * 
	 * @param userId ユーザID
	 * @return バージョン番号
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	@SuppressWarnings("unchecked")
	public String getVersion() throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("getVersion");

		// 操作ログ
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_ACCESS + " Get, Method=getVersion, User="
				+ HttpAuthenticator.getUserAccountString(wsctx));

		return VersionUtil.getVersion();
	}
}