/*

Copyright (C) 2011 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.systemlog.service;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.CalendarNotFound;
import com.clustercontrol.bean.ProcessConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.calendar.factory.SelectCalendar;
import com.clustercontrol.monitor.ejb.session.MonitorSettingControllerLocal;
import com.clustercontrol.monitor.ejb.session.MonitorSettingControllerUtil;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.MonitorStringValueInfo;
import com.clustercontrol.repository.bean.FacilityTreeAttributeConstant;
import com.clustercontrol.repository.factory.FacilitySelector;
import com.clustercontrol.systemlog.SyslogReceiverConfig;

/**
 * syslogを文字列として受け取り、パースしフィルタのマッチング処理を行う。
 * マッチしたものは通知キューにメッセージを送信する。
 */
public class MatcherTask implements Runnable {
	/** ログ出力のインスタンス  */
	private final static Log _log = LogFactory.getLog(MatcherTask.class);

	private final SyslogReceiverConfig _config;

	/** 通知した件数 */
	private volatile static long _notifyCounter = 0;

	/** パースに失敗したメッセージの数 */
	private volatile static long _invalidFormatMessageCounter = 0;

	/** マッチング処理対象文字列 */
	private final byte[] _msgData;

	/**
	 * 通知した件数を返します。
	 */
	public static long getNotifyCount(){
		return _notifyCounter;
	}

	/**
	 * パースに失敗したメッセージの数を返します。
	 */
	public static long getInvalidFormatMessageCount(){
		return _invalidFormatMessageCounter;
	}

	/**
	 * 新しいMatcherTaskオブジェクトを割り当てます
	 * @param multiId HA構成時にインスタンスを特定するためのID
	 * @param charsetName 文字セット
	 * @param msgData パース対象文字列データ
	 */
	public MatcherTask(SyslogReceiverConfig config, byte[] msgData) {
		_config = config;
		_msgData = msgData;
	}

	public String getMsgString() {
		if (_msgData == null) {
			return null;
		} else {
			return new String(_msgData, 0, _msgData.length, _config.getCharset());
		}
	}

	@Override
	public void run() {
		proccess();
	}

	private void proccess() {
		// 指定の文字コードで文字列を生成
		String message = null;
		message = new String(_msgData, 0, _msgData.length, _config.getCharset());

		//メッセージをパース
		MessageInfo logmsg = MessageParser.parse(message);
		if (logmsg == null) {
			synchronized (this) {
				if (_invalidFormatMessageCounter < Long.MAX_VALUE) {
					_invalidFormatMessageCounter++;
				} else {
					_invalidFormatMessageCounter = 0;
					_log.info("invalid syslog counter is reseted.");
				}
			}
			return;
		}

		_log.debug("get filter list.");
		Collection<MonitorInfo> monitorList;
		try {
			MonitorSettingControllerLocal controller = MonitorSettingControllerUtil.getLocalHome().create();
			monitorList = controller.getSystemlogList();
		} catch (Exception e) {
			_log.error("filter is invalid;" ,e);
			return ;
		}
		_log.debug("got filter list. size = " + monitorList.size());

		for(MonitorInfo monitorInfo : monitorList){
			if (monitorInfo.getMonitorFlg() == ValidConstant.TYPE_INVALID){
				continue;
			}

			// チェック対象ノードのファシリティIDをsyslogの「hostname」部分から逆引きする
			// 対応するノードがリポジトリに存在しない場合は
			// 管理対象フラグ「有効」のノードのみが入ったセットが返る。
			// 対応するノードが存在するが、その全てが有効/無効フラグ「無効」の場合は、
			Set<String> nodeFacilityIdList = resolveFacilitId(logmsg.getHostName());
			_log.debug("hostname : " + logmsg.getHostName());

			if(nodeFacilityIdList == null){
				_log.error("proccess() : nodeFacilityIdList is null.");
				return;
			}

			// 稼動日チェック(カレンダが指定されている場合)
			if(monitorInfo.getCalendarId() != null){
				// 稼動日かどうか
				boolean runFlg = true;
				try {
					runFlg = new SelectCalendar().isRun(monitorInfo.getCalendarId(), logmsg.getGenerationDate());
				} catch (CalendarNotFound e) {
					runFlg = true;
					// 指定されたカレンダIDがすでに存在しない
					_log.info("isRun() : calendarId = " + monitorInfo.getCalendarId() + " is not found.");
				} catch (Exception e) {
					_log.error("unexpected internal failure.", e);
				}

				// 非稼働日の場合はこの監視設定は無効扱いであるため以降のフィルタを処理する
				if (runFlg == false) {
					_log.debug("Skip:" + monitorInfo.getDescription() + ":" + message + " : CalendarId=" + monitorInfo.getCalendarId());
					continue;
				}
			}

			// ノード毎にマッチング処理
			for (String logFacilityId : nodeFacilityIdList) {
				_log.debug("Facility ID : " + logFacilityId);

				if(FacilityTreeAttributeConstant.UNREGISTEREFD_SCOPE.equals(logFacilityId)){
					// 未登録ノードの場合
					// 監視対象スコープが「未登録ノード」でない場合は処理しない
					if(!FacilityTreeAttributeConstant.UNREGISTEREFD_SCOPE.equals(monitorInfo.getFacilityId())){
						continue;
					}
				} else {
					// 未登録ノードでない場合
					// ノードがフィルタ条件で指定の監視対象スコープ配下に含まれない場合は処理しない
					_log.debug("call containsFaciliyId");
					if(!FacilitySelector.containsFaciliyId(monitorInfo.getFacilityId(), logFacilityId)){
						_log.debug("not contains!!");
						continue;
					}
					_log.debug("called containsFaciliyId");
				}

				//フィルタ条件のリストで順番にフィルタリング
				for (MonitorStringValueInfo rule : monitorInfo.getStringValueInfo()) {
					_log.debug("FaciliyId : " + monitorInfo.getFacilityId() + ", " + logFacilityId
							+ ", Pattern : " + rule.getPattern()
							+ ", ValidFlg : " + rule.isValidFlg()
							+ ", CaseSensitivityFlg : " + rule.getCaseSensitivityFlg());

					// フィルタが無効の場合は次のフィルタを処理する
					if(!rule.isValidFlg()){
						continue;
					}

					//メッセージ条件でマッチング
					_log.debug("call getPattern");
					Pattern pattern = null;
					// 大文字・小文字を区別しない場合
					if(rule.getCaseSensitivityFlg()){
						pattern = Pattern.compile(rule.getPattern(), Pattern.DOTALL | Pattern.CASE_INSENSITIVE);
					}
					// 大文字・小文字を区別する場合
					else{
						pattern = Pattern.compile(rule.getPattern(), Pattern.DOTALL);
					}
					Matcher matcher = pattern.matcher(logmsg.getMessage());
					_log.debug("called getPattern : " + rule.getPattern());

					try {
						if (matcher.matches()) {
							//処理する場合は、メッセージ送信
							if (rule.getProcessType() == ProcessConstant.TYPE_YES) {
								//メッセージ送信処理
								_log.debug("Process:" + rule.getDescription() + ":" + message);

								// 通知キューに投げる。
								_config.getLogNotifier().put(_config.getReceiverId(), message, logmsg, monitorInfo, rule, logFacilityId);
								synchronized (this) {
									if (_notifyCounter < Long.MAX_VALUE) {
										_notifyCounter++;
									} else {
										_notifyCounter = 0;
										_log.info("syslog notification counter is reseted.");
									}
								}
							} else {
								//処理しないならば、処理終了
								_log.debug("Non Process:" + rule.getDescription() + ":" + message);
							}
							break;
						}
					} catch (Exception e) {
						_log.error("filter is invalid;" + rule.getDescription(),e);
					}
				}
			}
		}

		return;
	}

	private Set<String> resolveFacilitId(String hostName){
		// ノード名による一致確認
		Set<String> facilityIdList = FacilitySelector.getNodeListByNodename(hostName);

		// ノード名で一致するノードがない場合
		if(facilityIdList == null){
			// IPアドレスによる一致確認
			try {
				facilityIdList = FacilitySelector.getNodeListByIpAddress(InetAddress.getByName(hostName));
			} catch (UnknownHostException e) {
				_log.debug(e.getMessage());
			}

			// ノード名でもIPアドレスでも一致するノードがない場合
			if(facilityIdList == null){
				// ホスト名による一致確認
				facilityIdList = FacilitySelector.getNodeListByHostname(hostName);
			}
		}

		if(facilityIdList == null){
			// 指定のノード名、IPアドレスで登録されているノードがリポジトリに存在しないため、
			// 「"UNREGISTEREFD"（FacilityTreeAttributeConstant.UNREGISTEREFD_SCOPE）」だけを
			// セットに含めたものをマップに登録する。
			facilityIdList = new HashSet<String>();
			facilityIdList.add(FacilityTreeAttributeConstant.UNREGISTEREFD_SCOPE);
		}

		return facilityIdList;
	}



	/**
	 * javaの正規表現の簡易スクリプトに利用。
	 * @param args
	 */
	public static void main(String args[]) {

		if (args.length < 2) {
			System.out.println("args error");
			return;
		}
		String a = args[0];
		String b = args[1];

		/*
		// 正規表現のコンパイル有無で性能差が出るか？
		// ・マッチする場合はコンパイルが速い。（2倍ぐらい）
		// ・マッチしない場合はmatchesが速い。（3倍ぐらい）
		// マッチしない場合の方がはるかに多いので、matchesを利用する。
		long time = 0;
		time = System.currentTimeMillis();
		for (int i = 0; i < 100000; i ++ ) {
			boolean flag = a.matches(b);
		}
		System.out.println("time " + (System.currentTimeMillis() - time) + "ms");

		Pattern p = Pattern.compile(b);
		time = System.currentTimeMillis();
		for (int i = 0; i < 100000; i ++ ) {
			boolean flag = p.matcher(a).find();
		}
		System.out.println("time " + (System.currentTimeMillis() - time) + "ms");
		 */

		System.out.println("args[0]:(" + a + ")");
		System.out.println("args[1]:(" + b + ")");
		System.out.println("args[0].matches(args[1]) : " + a.matches(b));
	}
}
