/*

Copyright (C) 2008 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

 */

package com.clustercontrol.sharedtable.jmx;

import javax.naming.CompositeName;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.Name;
import javax.naming.NamingException;

import org.jboss.naming.NonSerializableFactory;
import org.jboss.system.ServiceMBeanSupport;

import com.clustercontrol.commons.scheduler.TriggerConstant;
import com.clustercontrol.commons.scheduler.TriggerSchedulerException;
import com.clustercontrol.commons.scheduler.TriggerSchedulerFactory;
import com.clustercontrol.sharedtable.SharedTable;
import com.clustercontrol.sharedtable.ejb.session.CheckJobBean;
import com.clustercontrol.sharedtable.ejb.session.CheckJobUtil;


/**
 * @jmx.mbean
 *     name="user:service=SharedTableService,name=SharedTableService"
 *     description="SharedTable Service MBean"
 *     extends="org.jboss.system.ServiceMBean"
 * 
 * @jboss.service servicefile="sharedtable"
 */
public class SharedTableService extends ServiceMBeanSupport implements SharedTableServiceMBean{
	private String jndiName = "SharedTable";  // JNDIに登録する際の名前
	private String quartzJndiName = null;  // Quartzをlookupする際のJNDI名

	private String checkInterval = "34 */10 * * * ? *"; // チェックジョブの起動条件

	// 参照されなくなったデータテーブルの保持期間
	// このフィールドの値は、実際には、sharedTableのインスタンスに渡すだけのものであるが、
	// sharedTableのインスタンスが生成されるまでの間も値を保持する必要があるため、
	// フィールドとして値を保持する必要がある
	private long keepAlive;
	private final String m_checkJobName = "CHECK_JOB";
	private final String m_checkJobGroup = "SHAREDTABLE";


	private SharedTable sharedTable;

	/**
	 * @jmx.managed-attribute
	 * @return
	 */
	@Override
	public String getJndiName() {
		return jndiName;
	}

	/**
	 * @jmx.managed-attribute
	 * 
	 * @param jndiName
	 * @throws NamingException
	 * @throws NamingException
	 */
	@Override
	public void setJndiName(String jndiName) throws NamingException {
		log.info("setJndiName() SharedTableService : " + jndiName);

		String oldName = this.jndiName;
		this.jndiName = jndiName;

		if (super.getState() == STARTED) {
			try {
				unbind(oldName);
			} catch (NamingException e){
				NamingException ne = new NamingException("Failed to unbind");
				ne.setRootCause(e);
				log.error(ne);
			}

			try{
				rebind();
			} catch (NamingException e){
				NamingException ne = new NamingException("Failed to rebind");
				ne.setRootCause(e);
				log.error(ne);
			}
		}
	}

	/**
	 * QuartzのサービスのJNDI名を返します。
	 * @return QuartzサービスのJNDI名
	 *
	 * @jmx.managed-attribute
	 */
	@Override
	public String getQuartzJndiName() {
		return quartzJndiName;
	}

	/**
	 * QuartzのサービスのJNDI名を設定します。
	 * @param jndiName QuartzサービスのJNDI名
	 * 
	 * @jmx.managed-attribute
	 */
	@Override
	public void setQuartzJndiName(String jndiName) {
		this.quartzJndiName = jndiName;
	}

	/**
	 * 参照されなくなったデータテーブルの保持期間を返します。
	 * @return 参照されなくなったデータテーブルの保持期間
	 *
	 * @jmx.managed-attribute
	 */
	@Override
	public long getKeepAlive() {
		if(sharedTable != null){
			return sharedTable.getKeepAlive();
		} else {
			return -1;
		}
	}

	/**
	 * 参照されなくなったデータテーブルの保持期間を設定します。
	 * @param jndiName 参照されなくなったデータテーブルの保持期間
	 * 
	 * @jmx.managed-attribute
	 */
	@Override
	public void setKeepAlive(long keepAlive) {
		if(sharedTable != null){
			this.keepAlive = keepAlive;
			sharedTable.setKeepAlive(keepAlive);
		}
	}

	/**
	 * チェックジョブの実行スケジュール設定を返します。
	 * @return チェックジョブの実行スケジュール
	 *
	 * @jmx.managed-attribute
	 */
	@Override
	public String getCheckInterval() {
		return checkInterval;
	}

	/**
	 * チェックジョブの実行スケジュールを設定します。
	 * @param jndiName チェックジョブの実行スケジュール
	 * 
	 * @jmx.managed-attribute
	 */
	@Override
	public void setCheckInterval(String checkInterval) {
		this.checkInterval = checkInterval;
	}

	/**
	 * @jmx.managed-attribute
	 */
	@Override
	public String getName() {
		return "SharedTableService(" + jndiName + ")";
	}

	/**
	 * @jmx.managed-operation
	 */
	@Override
	public String printTableInfo() {
		return sharedTable.getTableListDebugInfo();
	}

	@Override
	public void createService() {
		log.info("Create SharedTableService(" + jndiName + ")");

		if(sharedTable == null){
			sharedTable = SharedTable.getInstance();

			sharedTable.setKeepAlive(keepAlive);
		}

		log.info("Created SharedTableService(" + jndiName + ")");
	}

	@Override
	public void startService() throws NamingException {
		log.info("Start SharedTableService(" + jndiName + ")");

		rebind();

		// 定期チェック用のジョブをスケジューリングする
		scheduleCheckJob();

		log.info("Started SharedTableService(" + jndiName + ")");
	}

	private void scheduleCheckJob(){
		// コールバック時の引数を設定
		Object[] jdArgs = new Object[1];
		Class[] jdArgsType = new Class[1];
		jdArgs[0] = jndiName;
		jdArgsType[0] = String.class;

		try {
			// SimpleTrigger でジョブをスケジューリング登録
			// 監視も収集も無効の場合、登録後にポーズするようにスケジュール
			TriggerSchedulerFactory.getScheduler(TriggerConstant.MONITOR_INSTANCE_NAME).scheduleEjbLocalInvokerJobWithCronTrigger(
					m_checkJobName,
					m_checkJobGroup,
					CheckJobUtil.getLocalHome(),
					CheckJobBean.METHOD_NAME,
					jdArgs,
					jdArgsType,
					null,
					true,
					true,
					checkInterval);
		} catch (TriggerSchedulerException e) {
			log.error(e.getMessage(), e);
		} catch (NamingException e) {
			log.error(e.getMessage(), e);
		}
	}

	@Override
	public void stopService() throws NamingException {
		log.info("Stop SharedTableService(" + jndiName + ")");

		unbind(jndiName);

		log.info("Stoped SharedTableService(" + jndiName + ")");
	}

	@Override
	public void destroyService() throws Exception {
		log.info("Destroy SharedTableService(" + jndiName + ")");

		sharedTable = null;

		log.info("Destroyed SharedTableService(" + jndiName + ")");
	}

	private static Context createContext(Context rootCtx, Name name) throws NamingException {
		Context subctx = rootCtx;

		for (int n = 0; n < name.size(); n++) {
			String atom = name.get(n);

			try {
				Object obj = subctx.lookup(atom);
				subctx = (Context) obj;
			} catch (NamingException e) {
				// 存在しない場合は、サブコンテキストを生成
				subctx = subctx.createSubcontext(atom);
			}
		}

		return subctx;
	}

	private void rebind() throws NamingException {
		InitialContext rootCtx = new InitialContext();

		Name fullName = rootCtx.getNameParser("").parse(jndiName);
		Name parentName = fullName;
		if(fullName.size() > 1){
			parentName = fullName.getPrefix(fullName.size()-1);
		} else {
			parentName = new CompositeName();
		}

		Context parentCtx = createContext(rootCtx, parentName);
		Name atomName = fullName.getSuffix(fullName.size()-1);
		String atomStirng = atomName.get(0);

		NonSerializableFactory.rebind(parentCtx, atomStirng, sharedTable);
	}

	private void unbind(String jndiName) throws NamingException {
		InitialContext rootCtx = null;

		try {
			rootCtx = new InitialContext();

			Name fullName = rootCtx.getNameParser("").parse(jndiName);
			Name atomName = fullName.getSuffix(fullName.size() - 1);
			String atom = atomName.get(0);

			rootCtx.unbind(jndiName);
			NonSerializableFactory.unbind(atom);
		} finally {
			if(rootCtx != null) {
				rootCtx.close();
			}
		}
	}
}
