/*

Copyright (C) 2008 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

 */
package com.clustercontrol.sharedtable;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.poller.NotInitializedException;

/**
 * 複数のデータホルダを管理するクラスです。
 * データホルダは、収集間隔ごとに保持されます。
 */
public class DataTableCollection {
	private static Log m_log = LogFactory.getLog( DataTableCollection.class );

	private Object modifyLock = new Object();

	/**
	 * 管理するデータホルダのページサイズ
	 */
	private final int m_pageSize;

	/**
	 * 参照されなくなったデータテーブルの保持期間
	 */
	private long m_keepAlive = 60000L;  // 60秒間

	/**
	 * 収集名ごとに収集間隔と最終参照時刻を保持する
	 * 
	 * Key   : 収集名
	 * Value : 収集間隔と最終参照時刻
	 */
	private ConcurrentHashMap<String, IntervalAndLastReference> m_collectorIntervalMap;

	/**
	 * 収集間隔ごとにDataTableHolderを保持する
	 * 
	 * Key   : 収集間隔
	 * Value : 収集間隔ごとにDataTableを複数まとめて管理するクラス
	 */
	private ConcurrentHashMap<Integer, DataTableHolder> m_tableHolderMap;

	/**
	 * コンストラクタ
	 * @param pageSize ページサイズ
	 * @param keepAlive 参照されなくなったデータテーブルの保持期間
	 */
	protected DataTableCollection(int pageSize, long keepAlive){
		m_pageSize = pageSize;
		m_keepAlive = keepAlive;
		m_collectorIntervalMap = new ConcurrentHashMap<String, IntervalAndLastReference>();
		m_tableHolderMap = new ConcurrentHashMap<Integer, DataTableHolder>();
	}

	/**
	 * 指定の収集名で指定の収集期間で参照、更新されるテーブルホルダを作成する
	 * @param collectorName 収集名
	 * @param interval 収集間隔
	 */
	protected void creatDataTableHolder(String collectorName, int interval){
		m_log.debug("Create DataTableHolder : " + collectorName + ", " + interval);

		synchronized (modifyLock) {
			m_collectorIntervalMap.put(collectorName, new IntervalAndLastReference(interval));

			// 指定の収集期間のテーブルホルダが存在しない場合は作成する
			if(m_tableHolderMap.get(interval) == null){
				DataTableHolder holder = new DataTableHolder(m_pageSize);
				m_tableHolderMap.put(interval, holder);
			}
		}
	}

	/**
	 * 指定の収集名の設定を削除する。
	 * 収集名を削除することで、参照されなくなるテーブルホルダも同時に削除する
	 * @param collectorName 収集名
	 */
	protected void removeCollectorName(String collectorName){
		synchronized (modifyLock) {
			// 指定の収集名で登録されている収集間隔、最終参照時刻情報を取得する
			IntervalAndLastReference ial = m_collectorIntervalMap.get(collectorName);

			if(ial == null){
				return;
			}

			// 今回更新したことで削除する必要があるかチェックする収集間隔を取得
			int removeInterval = ial.getInterval();

			// 指定の収集名の情報を削除する
			m_collectorIntervalMap.remove(collectorName);

			// 現在登録されている全ての収集間隔を走査し、
			// 今回収集名に関連する収集間隔を削除することで、
			// 参照されなくなるテーブルホルダが存在するか否かを調べる
			Iterator<IntervalAndLastReference> ials = m_collectorIntervalMap.values().iterator();

			// 収集間隔にマッピングされているテーブルホルダを削除するか否かのフラグ
			boolean removeFlg = true;
			while(ials.hasNext()){
				int interval = ials.next().getInterval();

				// ひとつでも別の収集名で参照されるものがある場合は、削除しない
				m_log.debug("interval=" + interval + ", removeInterval=" + removeInterval);
				if(interval == removeInterval){
					removeFlg = false;
				}
			}

			// テーブルホルダを削除する
			if(removeFlg){
				m_log.debug("Remove DataTableHolder : " + removeInterval);

				m_tableHolderMap.remove(removeInterval);
			}
		}
	}

	/**
	 * 指定の収集名で登録されているデータホルダが存在するか確認する。
	 * @param collectorName 収集名
	 * @return 存在する場合はtrue
	 */
	protected boolean containsCollectorName(String collectorName){
		IntervalAndLastReference ial = m_collectorIntervalMap.get(collectorName);

		if(ial == null){
			// エラー処理
			return false;
		}

		return m_tableHolderMap.containsKey(ial.getInterval());
	}

	protected Set<Integer> getIntervals(){
		return m_tableHolderMap.keySet();
	}

	protected int getInterval(String collectorName){
		IntervalAndLastReference ial = m_collectorIntervalMap.get(collectorName);

		if(ial == null){
			// エラー処理
			return -1;
		}

		return ial.getInterval();
	}

	protected long getLastReference(String collectorName){
		IntervalAndLastReference ial = m_collectorIntervalMap.get(collectorName);

		if(ial == null){
			// エラー処理
			return -1;
		}

		return ial.getLastReference();
	}

	protected int getPageSize() {
		return m_pageSize;
	}

	protected DataTableHolder getDataTableHolder(String collectorName){
		synchronized (modifyLock) {
			IntervalAndLastReference ial = m_collectorIntervalMap.get(collectorName);

			if(ial == null){
				// エラー処理
				return null;
			}

			//指定の収集名の最終参照日時を更新する
			ial.setLastReference();

			return m_tableHolderMap.get(ial.getInterval());
		}
	}

	/**
	 * 設定されている収集名のセットを取得します
	 * @return 設定されている収集名のセット
	 */
	protected Set<String> getCollectorNames(){
		return m_collectorIntervalMap.keySet();
	}

	//	public DataTable getDataTable(String collectorName, int page){
	//		IntervalAndLastReference ial = m_collectorIntervalMap.get(collectorName);
	//
	//		if(ial == null){
	//			// エラー処理
	//			return null;
	//		}
	//
	//		DataTableHolder holder = tableHolderMap.get(ial.getInterval());
	//
	//		if(holder == null){
	//			// エラー処理
	//			return null;
	//		}
	//
	//		return holder.get(page);
	//	}

	/**
	 * 収集間隔ごとに管理されているDataTableに新たなテーブルを挿入する。
	 * checkKeyは、テーブルホルダで管理されているページのキーと同じキーを指定する必要があります。
	 * キーが異なる場合は、テーブルホルダの全てのページがクリアされ、
	 * 今回与えられたデータテーブルのみが挿入された状態となります。
	 * 
	 * @param interval 収集間隔
	 * @param table 挿入するテーブル
	 * @param checkKey テーブルホルダにデータを格納する際のキー
	 * @throws NotInitializedException
	 */
	protected void insertDataTable(int interval, DataTable table, String checkKey)
	throws NotInitializedException{
		DataTableHolder holder = m_tableHolderMap.get(interval);

		if(holder == null){
			// エラー処理
			throw new NotInitializedException("DataTableHolder is not found.");
		}

		holder.insertDataTable(table, checkKey);
	}

	/**
	 * 収集名ごとに管理されているテーブルホルダのうち
	 * 最終参照時刻から生存期間以上経過しているものを削除する
	 * 
	 * @param now 基準時刻
	 * @return 管理している全てのテーブルが削除された場合にはfalseを返す
	 */
	protected boolean checkAlive(long now){
		synchronized (modifyLock) {
			m_log.debug("DataTableCollection check : " + now);

			// 削除対象の収集名を保持するリスト
			ArrayList<String> removeList = new ArrayList<String>();

			Set<String> collectorNames = m_collectorIntervalMap.keySet();
			Iterator<String> itr = collectorNames.iterator();

			while(itr.hasNext()){
				// 収集名
				String collcetorName = itr.next();

				// 現在チェックしている収集名の収集間隔と最終参照時刻を取得する
				int interval = m_collectorIntervalMap.get(collcetorName).getInterval();
				long lastRef = m_collectorIntervalMap.get(collcetorName).getLastReference();

				// 収集間隔2回分 ＋ 保持時間 を超えたものは削除対処
				long lastMoment = (lastRef + (interval * 1000L) * 2 + m_keepAlive);
				if(m_log.isDebugEnabled()){
					m_log.debug(collcetorName + " : " + "lastMoment=" + new Date(lastMoment));
				}

				if(now  >= lastMoment){
					m_log.debug("remove target = " + collcetorName);
					removeList.add(collcetorName);
				}
			}

			// 削除対象がある場合
			if(removeList.size() > 0){
				// 削除する
				itr = removeList.iterator();
				while(itr.hasNext()){
					String collcetorName = itr.next();

					m_log.info("remove CollectorName : " + collcetorName + ", "
							+ m_collectorIntervalMap.get(collcetorName).getInterval());

					m_collectorIntervalMap.remove(collcetorName);
				}

				// 今後も参照される可能性のある参照間隔を保持する
				HashSet<Integer> activeIntrevalSet = new HashSet<Integer>();

				collectorNames = m_collectorIntervalMap.keySet();
				itr = collectorNames.iterator();

				while(itr.hasNext()){
					String collcetorName = itr.next();
					int interval = m_collectorIntervalMap.get(collcetorName).getInterval();

					activeIntrevalSet.add(interval);
				}

				// 削除対象の収集間隔を求める
				// 現在登録されている収集間隔を全てセット
				Set<Integer> removeIntrevalSet = new HashSet<Integer>(m_tableHolderMap.keySet());

				// 今後も参照されるものは除く
				removeIntrevalSet.removeAll(activeIntrevalSet);

				// 削除する
				Iterator<Integer> intItr = removeIntrevalSet.iterator();
				while(intItr.hasNext()){
					int interval = intItr.next();

					m_log.debug("remove Interval : " + interval);

					m_tableHolderMap.remove(interval);
				}
			}
			return (m_tableHolderMap.size() == 0);
		}
	}

	/**
	 * 収集間隔と最終参照日時を保持するクラス
	 */
	private class IntervalAndLastReference {
		private final int m_interval;
		private long m_lastReference;

		public IntervalAndLastReference(int interval){
			m_interval = interval;
			m_lastReference = System.currentTimeMillis();
		}

		public int getInterval(){
			return m_interval;
		}

		public long getLastReference(){
			return m_lastReference;
		}

		public void setLastReference(){
			m_lastReference = System.currentTimeMillis();
		}
	}
}
