/*

Copyright (C) 2008 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

 */

package com.clustercontrol.poller;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.Iterator;
import java.util.concurrent.ConcurrentHashMap;

import com.clustercontrol.sharedtable.DataTableNotFoundException;

public class PollerManager {
	private static Log m_log = LogFactory.getLog( PollerManager.class );

	private String m_managerName;

	/**
	 * Key   : ポーラ識別情報
	 * Value : ポーラ
	 */
	private ConcurrentHashMap<PollerInfo, PollingController> m_pollerTable;

	public PollerManager(String managerName){
		m_managerName = managerName;
		m_pollerTable = new ConcurrentHashMap<PollerInfo, PollingController>();
	}

	public PollingController createPoller(
			String group,
			String facilityId,
			boolean indexCheckFlg,
			String tableGroup,
			String tableName)
	throws NotInitializedException, DataTableNotFoundException {
		if(m_log.isDebugEnabled()){
			m_log.debug(
					"create poller    " +
					"PollerGroup : " + group + ", " +
					"FacilityId : " + facilityId + ", " +
					"IndexCheckFlg : " + indexCheckFlg + ", " +
					"TableGroup : " + tableGroup + ", " +
					"TableName : " + tableName
			);
		}

		// 収集情報、情報書き込み対象テーブル情報を格納したポーラを生成する
		// この時点ではインスタンスを生成するだけでポーリングはスケジューリングしないため、
		// ポーリングは実行されない
		PollingController poller = new PollingController(
				group,
				facilityId,
				facilityId,
				indexCheckFlg,
				tableGroup,
				tableName);

		// マップに登録する
		m_pollerTable.put(new PollerInfo(group, facilityId), poller);
		return poller;
	}

	/**
	 * 
	 * @param pollerGroup
	 * @param pollerName
	 */
	public PollingController getPoller(String pollerGroup, String pollerName){
		return m_pollerTable.get(new PollerInfo(pollerGroup, pollerName));
	}


	/**
	 * 管理しているポーラの情報を文字列で返します。
	 * @return 管理しているテーブルの情報
	 */
	public String getPollerDebugInfo(){
		String debugStr = "";

		Iterator<PollerInfo> itr = m_pollerTable.keySet().iterator();

		while(itr.hasNext()){
			PollerInfo info = itr.next();
			int minInterval = m_pollerTable.get(info).getPollingConfig().getMinPollingInterval();
			debugStr = debugStr + info.getPollerGroup() + ", " + info.getPollerName() + ", " + minInterval+ "\n";
			debugStr = debugStr + m_pollerTable.get(info).getPollingConfig().getDebugInfo();
		}

		return debugStr;
	}

	/**
	 * 管理しているポーラを一意に識別するための情報を保持するクラス
	 * HashMapのキーとして利用できるように equalsメソッドと hashCodeメソッドを実装
	 */
	private class PollerInfo{
		private String m_pollerGroup;
		private String m_pollerName;

		public PollerInfo(String pollerGroup, String pollerName){
			m_pollerGroup = pollerGroup;
			m_pollerName = pollerName;
		}

		public String getPollerGroup() {
			return m_pollerGroup;
		}

		public String getPollerName() {
			return m_pollerName;
		}

		@Override
		public boolean equals(Object other) {
			if (other instanceof PollerInfo) {
				PollerInfo info = (PollerInfo)other;

				if (this.m_pollerGroup == null && this.m_pollerName == null){
					if (info.m_pollerGroup == null && info.m_pollerName == null){
						return true;
					}
				} else if (this.m_pollerGroup == null && this.m_pollerName != null){
					if (info.m_pollerGroup == null && this.m_pollerName.equals(info.m_pollerName)){
						return true;
					}
				} else if (this.m_pollerGroup != null && this.m_pollerName == null){
					if (this.m_pollerGroup.equals(info.m_pollerGroup) && info.m_pollerName == null){
						return true;
					}
				} else {
					if (this.m_pollerGroup.equals(info.m_pollerGroup)){
						return this.m_pollerName.equals(info.m_pollerName);
					}
				}
				return false;
			} else {
				return false;
			}
		}

		@Override
		public int hashCode() {
			int result = 17;

			result = 37 * result + ((this.m_pollerGroup != null) ? this.m_pollerGroup.hashCode() : 0);

			result = 37 * result + ((this.m_pollerName != null) ? this.m_pollerName.hashCode() : 0);

			return result;
		}
	}
}
