/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

 */

package com.clustercontrol.performance.ejb.session;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.SessionContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.MonitorNotFound;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.monitor.ejb.session.MonitorSettingControllerLocal;
import com.clustercontrol.monitor.ejb.session.MonitorSettingControllerUtil;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.MonitorTypeConstant;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoLocal;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoUtil;
import com.clustercontrol.monitor.run.factory.SelectMonitor;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorItemCodeMstLocal;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorItemCodeMstUtil;
import com.clustercontrol.performance.util.ExportCollectedDataFile;
import com.clustercontrol.performance.bean.CollectedDataInfo;
import com.clustercontrol.performance.bean.CollectedDataSet;
import com.clustercontrol.performance.bean.CollectorItemInfo;
import com.clustercontrol.performance.bean.CollectorItemParentInfo;
import com.clustercontrol.performance.bean.PerformanceDataSettings;
import com.clustercontrol.performance.bean.PerformanceFilterInfo;
import com.clustercontrol.performance.bean.PerformanceListInfo;
import com.clustercontrol.performance.dao.CalculatedDataDAO;
import com.clustercontrol.performance.util.code.CollectorItemCodeTable;
import com.clustercontrol.performance.util.code.CollectorItemTreeItem;
import com.clustercontrol.repository.ejb.session.RepositoryControllerBean;
import com.clustercontrol.repository.factory.FacilitySelector;
import com.clustercontrol.repository.bean.FacilityTreeItem;

/**
 *　性能管理機能の管理を行うコントローラクラス
 * クライアントからの Entity Bean へのアクセスは、このSession Bean を介して行います。
 * 
 * @version 4.0.0
 * @since 1.0.0
 *
 * <!-- begin-xdoclet-definition -->
 * @ejb.bean name="CollectorController"
 *           jndi-name="CollectorController"
 *           type="Stateless"
 *           transaction-type="Container"
 *           view-type="local"
 * 
 * <!-- @jboss.clustered -->
 * 
 * @ejb.transaction type="Required"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorCalcMethodMstLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorCategoryCollectMstLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorCategoryMstLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorDeviceInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorItemCalcMethodMstLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorItemCodeMstLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorPollingMstLocal"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 * 
 * <!-- end-xdoclet-definition -->
 * @generated
 */
public abstract class CollectorControllerBean implements javax.ejb.SessionBean {

	//	ログ出力
	private static Log m_log = LogFactory.getLog(CollectorControllerBean.class);

	// コンテキスト情報。
	@SuppressWarnings("unused")
	private SessionContext m_context;

	// 実績収集で1行あたりに必要な物理サイズ容量（Byte）
	// サイズ見積り時に使われる（ヘッダサイズ ＋ timestamp + double precision）
	private static long _fixRowSize = 24 + 8 + 8;

	/**
	 * コンテキスト情報を設定します。<BR>
	 * Session Bean の生成時に行う処理を実装します。
	 * @see javax.ejb.SessionBean#setSessionContext(javax.ejb.SessionContext)
	 */
	@Override
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}

	/**
	 * 性能[一覧]に表示するためのリストを取得します。
	 * 以下の条件のうちいずれかを満たす監視設定を取得します。
	 * 
	 * ・収集が有効になっていること
	 * ・収集データが1件以上存在すること
	 * 
	 * @ejb.interface-method
	 * 
	 * @return
	 * @throws HinemosUnknown
	 */
	public ArrayList<PerformanceListInfo> getPerformanceList() throws HinemosUnknown {
		m_log.debug("getPerformanceList()");
		ArrayList<PerformanceListInfo> ret = new ArrayList<PerformanceListInfo>();

		try {
			ArrayList<MonitorInfo> monitorList = new SelectMonitor().getMonitorList();
			CalculatedDataDAO calcDao = new CalculatedDataDAO();
			for (MonitorInfo monitorInfo : monitorList){

				m_log.debug("getPerformanceList() target monitorId = " + monitorInfo.getMonitorId());

				if (monitorInfo.getMonitorType() == MonitorTypeConstant.TYPE_NUMERIC) {
					m_log.debug("getPerformanceList() add monitorId = " + monitorInfo.getMonitorId());

					PerformanceListInfo perfListInfo = new PerformanceListInfo();
					perfListInfo.setCollectorFlg(monitorInfo.getCollectorFlg());
					perfListInfo.setMonitorId(monitorInfo.getMonitorId());
					perfListInfo.setMonitorTypeId(monitorInfo.getMonitorTypeId());
					perfListInfo.setDescription(monitorInfo.getDescription());
					perfListInfo.setFacilityId(monitorInfo.getFacilityId());
					perfListInfo.setScopeText(monitorInfo.getScope());
					perfListInfo.setRunInterval(monitorInfo.getRunInterval());

					if(calcDao.getLatestDate(monitorInfo.getMonitorId()) == null){
						perfListInfo.setLatestDate(null);
					}
					else{
						perfListInfo.setLatestDate(calcDao.getLatestDate(monitorInfo.getMonitorId()).getTime());
					}
					if(calcDao.getOldestDate(monitorInfo.getMonitorId()) == null){
						perfListInfo.setOldestDate(null);
					}
					else{
						perfListInfo.setOldestDate(calcDao.getOldestDate(monitorInfo.getMonitorId()).getTime());
					}
					ret.add(perfListInfo);
				}
			}

		} catch (Exception e) {
			m_log.error("getPerformanceList() " + e.getMessage(),e);
			throw new HinemosUnknown(e.getMessage(),e);
		}


		return ret;

	}


	/**
	 * 性能[一覧]に表示するためのリストを取得します。
	 * 以下の条件のうちいずれかを満たす監視設定を取得します。
	 * 
	 * ・収集が有効になっていること
	 * ・収集データが1件以上存在すること
	 * ・フィルタ条件に合致すること
	 * 
	 * @ejb.interface-method
	 * 
	 * @param condition
	 * @return
	 * @throws HinemosUnknown
	 */
	public ArrayList<PerformanceListInfo> getPerformanceList(PerformanceFilterInfo condition) throws HinemosUnknown {
		m_log.debug("getPerformanceList() condition");
		if(m_log.isDebugEnabled()){
			if(condition != null){
				m_log.debug("getPerformanceList() " +
						"monitorId = " + condition.getMonitorId() +
						", monitorTypeId = " + condition.getMonitorTypeId() +
						", description = " + condition.getDescription() +
						", oldestFromDate = " + condition.getOldestFromDate() +
						", oldestToDate = " + condition.getOldestToDate() +
						", latestFromDate = " + condition.getLatestFromDate() +
						", latestToDate = " + condition.getLatestToDate());
			}
		}
		ArrayList<PerformanceListInfo> filterList = new ArrayList<PerformanceListInfo>();
		// 条件未設定の場合は空のリストを返却する
		if(condition == null){
			m_log.debug("getPerformanceList() condition is null");
			return filterList;
		}

		// DBではなくJavaAPにてフィルタリングを行う
		for(PerformanceListInfo info : getPerformanceList()){


			// monitorId
			if(condition.getMonitorId() != null && !"".equals(condition.getMonitorId())){
				if(info.getMonitorId() == null || (info.getMonitorId() != null && !info.getMonitorId().matches(".*" + condition.getMonitorId() + ".*"))){
					m_log.debug("getPerformanceList() continue : monitorId target = " + info.getMonitorId() + ", filter = " + condition.getMonitorId());
					continue;
				}
			}

			// monitorTypeId
			if(condition.getMonitorTypeId() != null && !"".equals(condition.getMonitorTypeId())){
				if(info.getMonitorTypeId() == null || (info.getMonitorTypeId() != null && !info.getMonitorTypeId().matches(".*" + condition.getMonitorTypeId() + ".*"))){
					m_log.debug("getPerformanceList() continue : monitorTypeId target = " + info.getMonitorTypeId() + ", filter = " + condition.getMonitorTypeId());
					continue;
				}
			}

			// description
			if(condition.getDescription() != null && !"".equals(condition.getDescription())){
				if(info.getDescription() == null || (info.getDescription() != null && !info.getDescription().matches(".*" + condition.getDescription() + ".*"))){
					m_log.debug("getPerformanceList() continue : description target = " + info.getDescription() + ", filter = " + condition.getDescription());
					continue;
				}
			}

			// oldestFromDate
			if(condition.getOldestFromDate() > 0){
				if(info.getOldestDate() == null || !(info.getOldestDate() >= condition.getOldestFromDate())){
					m_log.debug("getPerformanceList() continue : oldestFromDate target = " + info.getOldestDate() + ", filter = " + condition.getOldestFromDate());
					continue;
				}
			}
			// oldestToDate
			if(condition.getOldestToDate() > 0){
				if(info.getOldestDate() == null || !(info.getOldestDate() <= condition.getOldestToDate())){
					m_log.debug("getPerformanceList() continue : oldestToDate target = " + info.getOldestDate() + ", filter = " + condition.getOldestToDate());
					continue;
				}
			}

			// latestFromDate
			if(condition.getLatestFromDate() > 0){
				if(info.getLatestDate() == null || !(info.getLatestDate() >= condition.getLatestFromDate())){
					m_log.debug("getPerformanceList() continue : latestFromDate target = " + info.getLatestDate() + ", filter = " + condition.getLatestFromDate());
					continue;
				}
			}
			// latestToDate
			if(condition.getLatestToDate() > 0){
				if(info.getLatestDate() == null || !(info.getLatestDate() <= condition.getLatestToDate())){
					m_log.debug("getPerformanceList() continue : latestToDate target = " + info.getLatestDate() + ", filter = " + condition.getLatestToDate());
					continue;
				}
			}

			m_log.debug("getPerformanceList() add display list : target = " + info.getMonitorId());
			filterList.add(info);
		}

		return filterList;
	}


	/**
	 * 性能実績収集に必要な基本情報を取得します。
	 * 
	 * @param monitorId 監視項目ID
	 * @return グラフ描画に必要な基本情報
	 * 
	 * @ejb.interface-method
	 *
	 * @return
	 */
	public PerformanceDataSettings getPerformanceGraphInfo(String monitorId) {
		m_log.debug("getPerformanceGraphInfo() : monitorId = " + monitorId);

		PerformanceDataSettings perfDataSettings = null;
		try {

			// 対象の監視設定の取得
			MonitorInfoLocal monitorInfoLocal = MonitorInfoUtil.getLocalHome().findByPrimaryKey(monitorId);
			// ファシリティツリーの取得(facilityId is null の場合を考えて、監視設定より取得)
			FacilityTreeItem facilityTree = FacilitySelector.getFacilityTree(monitorInfoLocal.getFacilityId(), Locale.getDefault(), false, Boolean.TRUE);
			ArrayList<String> targetfacilityIdList = FacilitySelector.getNodeFacilityIdList(monitorInfoLocal.getFacilityId(), RepositoryControllerBean.ALL, false, true);

			// 戻り値の初期化
			perfDataSettings = new PerformanceDataSettings();

			// 監視ID
			perfDataSettings.setMonitorId(monitorId);
			// 監視IDに指定されたファシリティID
			perfDataSettings.setFacilityId(monitorInfoLocal.getFacilityId());
			// ファシリティツリー
			perfDataSettings.setFacilityTreeItem(facilityTree);
			// ファシリティIDリスト
			perfDataSettings.setTargetFacilityIdList(targetfacilityIdList);
			// 監視間隔=自動描画間隔
			perfDataSettings.setInterval(monitorInfoLocal.getRunInterval());
			// 表示する単位
			perfDataSettings.setMeasure(monitorInfoLocal.getMeasure());
			// 収集有効/無効フラグ
			perfDataSettings.setStatus(monitorInfoLocal.getCollectorFlg());
			// 監視種別
			perfDataSettings.setMonitorTypeId(monitorInfoLocal.getMonitorTypeId());

			// 収集項目ID一覧、収集項目IDに対して存在するデバイス表示名を設定
			CalculatedDataDAO calcDao = new CalculatedDataDAO();
			ArrayList<CollectorItemParentInfo> itemCodeList = calcDao.getCollectorItemParentList(monitorId, monitorInfoLocal.getMonitorTypeId());
			perfDataSettings.setItemCodeList(itemCodeList);

			// 収集項目IDに該当する収集項目名を設定（リソース監視以外は固定）
			if ((HinemosModuleConstant.MONITOR_PERFORMANCE).equals(
					monitorInfoLocal.getMonitorTypeId())){
				// リソース監視の場合
				for (CollectorItemParentInfo itemInfo : perfDataSettings.getItemCodeList()){
					CollectorItemCodeMstLocal mst = CollectorItemCodeMstUtil.getLocalHome().findByPrimaryKey(itemInfo.getItemCode());

					// 表示名
					perfDataSettings.setItemName(itemInfo.getItemCode(), mst.getItemName());

					m_log.debug("getPerformanceGraphInfo() : MON_PRF itemCode = " + itemInfo.getItemCode() + ", itemName = " + mst.getItemName());
				}

				MonitorSettingControllerLocal bean = MonitorSettingControllerUtil.getLocalHome().create();
				MonitorInfo monitorInfo = bean.getMonitor(monitorId, HinemosModuleConstant.MONITOR_PERFORMANCE);

				// 監視対象のItemCode
				perfDataSettings.setTargetItemCode(monitorInfo.getPerfCheckInfo().getItemCode());
				// 監視対象のデバイス
				perfDataSettings.setTargetDisplayName(monitorInfo.getPerfCheckInfo().getDeviceDisplayName());

			} else {
				// リソース監視以外は監視設定で取得された値を指定(リソース監視は1つのItemCodeを持つ)
				if(perfDataSettings.getItemCodeList().size() > 0){
					CollectorItemParentInfo itemInfo = perfDataSettings.getItemCodeList().get(0);

					// 表示名
					perfDataSettings.setItemName(itemInfo.getItemCode(), monitorInfoLocal.getItemName());
					// 監視対象のItemCode
					perfDataSettings.setTargetItemCode(itemInfo.getItemCode());
					// 監視対象のデバイス
					perfDataSettings.setTargetDisplayName(itemInfo.getDisplayName());

					m_log.debug("getPerformanceGraphInfo() : MON_XXX itemCode = " + itemInfo.getItemCode() + ", itemName = " + monitorInfoLocal.getItemName());
				}
				else{
					m_log.error("getPerformanceGraphInfo() itemCode size = 0");
				}
			}

			// 最新・最古の収集時刻を設定
			if(calcDao.getLatestDate(monitorId) == null){
				perfDataSettings.setLatestDate(null);
			}
			else{
				perfDataSettings.setLatestDate(calcDao.getLatestDate(monitorId).getTime());
			}
			if(calcDao.getOldestDate(monitorId) == null){
				perfDataSettings.setOldestDate(null);
			}
			else{
				perfDataSettings.setOldestDate(calcDao.getOldestDate(monitorId).getTime());
			}
			m_log.debug("getPerformanceGraphInfo() : LatestDate = " + perfDataSettings.getLatestDate());
			m_log.debug("getPerformanceGraphInfo() : OldestDate = " + perfDataSettings.getOldestDate());
		} catch (FinderException e) {
			// 指定の収集IDが存在しない場合はnullを返す
			m_log.error("getPerformanceGraphInfo()", e);
		} catch (NamingException e){
			m_log.error("getPerformanceGraphInfo()", e);
			throw new EJBException(e);
		} catch (CreateException e) {
			m_log.error("getPerformanceGraphInfo()", e);
			throw new EJBException(e);
		} catch (HinemosUnknown e) {
			m_log.error("getPerformanceGraphInfo()", e);
			throw new EJBException(e);
		} catch (MonitorNotFound e) {
			m_log.error("getPerformanceGraphInfo()", e);
			throw new EJBException(e);
		}
		return perfDataSettings;
	}

	/**
	 * 実績収集で収集されたデータを取得します。
	 * @param facilityIdList　ファシリティIDのリスト
	 * @param itemInfoList　収集項目のリスト
	 * @param startDate　取得したい始点の時刻
	 * @param endDate　　取得した終点の時刻
	 * @return　収集データのデータセット
	 * 
	 * @ejb.interface-method
	 *
	 */
	public CollectedDataSet getRecordCollectedData(
			ArrayList<String> facilityIdList,
			ArrayList<CollectorItemInfo> itemInfoList,
			Date startDate,
			Date endDate) {

		m_log.debug("getRecordCollectedData() facilityIdList size = " + facilityIdList.size() + ", itemInfoList size = " + itemInfoList.size());

		CollectedDataSet ret = new CollectedDataSet();

		for(String facilityId : facilityIdList){
			m_log.debug("getRecordCollectedData() facilityId = " + facilityId);

			for(CollectorItemInfo itemInfo : itemInfoList){
				m_log.debug("getRecordCollectedData() facilityId = " + facilityId + ", itemInfo = " + itemInfo.getItemCode());

				CalculatedDataDAO dao = new CalculatedDataDAO();

				// 収集項目をDBから取得します
				List<CollectedDataInfo> dataList = dao.select(
						itemInfo.getCollectorId(),
						itemInfo.getItemCode(),
						itemInfo.getDisplayName(),
						facilityId,
						startDate,
						endDate);

				m_log.debug("getRecordCollectedData() size = " + dataList.size());

				ret.setCollectedDataList(facilityId, itemInfo, dataList);
			}
		}

		return ret;
	}

	/**
	 * 収集項目コードの一覧を取得します
	 * 
	 * @ejb.interface-method
	 * @generated
	 * 
	 * @return 収集項目IDをキーとしCollectorItemTreeItemが格納されているHashMap
	 */
	public HashMap<String, CollectorItemTreeItem> getItemCodeMap(){
		m_log.debug("getItemCodeMap()");

		return CollectorItemCodeTable.getItemCodeMap();
	}

	/**
	 * 指定のファシリティで収集可能な項目のリストを返します
	 * デバイス別の収集項目があり、ノード毎に登録されているデバイス情報が異なるため、
	 * 取得可能な収集項目はファシリティ毎に異なる。
	 * 
	 * @ejb.interface-method
	 * 
	 * @param facilityId ファシリティID
	 * @return 指定のファシリティで収集可能な項目のリスト
	 */
	public List<CollectorItemInfo> getAvailableCollectorItemList(String facilityId) {
		m_log.debug("getAvailableCollectorItemList() facilityId = " + facilityId);

		return CollectorItemCodeTable.getAvailableCollectorItemList(facilityId);

	}

	/**
	 * 指定した監視項目ID(収集ID)とファシリティID(ノード、スコープ)の性能実績データファイル(csv形式)に出力するファイルパスのリストを返却する。
	 * このメソッドは作成するファイル名を返却し、CSV出力処理は別スレッドで動作する。
	 * 
	 * @ejb.interface-method
	 * 
	 * @param monitorId 監視項目ID
	 * @param facilityId ファシリティID(ノードorスコープ)
	 * @param header ヘッダをファイルに出力するか否か
	 * @param archive ファイルをアーカイブするか否か
	 * 
	 * @return Hinemos マネージャサーバ上に出力されたファイル名
	 */
	public List<String> createPerfFile(String monitorId, String facilityId, boolean header, boolean archive) throws HinemosUnknown{
		m_log.debug("createPerfFile() monitorId = " + monitorId + ", facilityId = " + facilityId + ", header = " + header + ", archive = " + archive);
		return ExportCollectedDataFile.create(monitorId, facilityId, header, archive);
	}

	/**
	 * 指定したリストのファイルパスを削除する
	 * 
	 * @ejb.interface-method
	 * 
	 * @param filepathList
	 * @throws HinemosUnknown
	 */
	public void deletePerfFile(ArrayList<String> fileNameList) throws HinemosUnknown{
		m_log.debug("deletePerformanceFile()");
		ExportCollectedDataFile.deleteFile(fileNameList);
	}

}