/*

 Copyright (C) 2006 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 */

package com.clustercontrol.performance.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;

import javax.ejb.EJBException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.commons.util.ConnectionManager;
import com.clustercontrol.commons.util.HinemosProperties;
import com.clustercontrol.performance.bean.CollectedDataErrorTypeConstant;
import com.clustercontrol.performance.bean.CollectedDataInfo;
import com.clustercontrol.performance.bean.CollectorItemParentInfo;
import com.clustercontrol.performance.bean.PerfData;
import com.clustercontrol.performance.bean.Sample;

/**
 * 収集した性能値をDBに保存、読み出しするDAOクラス
 * 
 * 
 * @version 4.0.0
 * @since 1.0
 */
public class CalculatedDataDAO {

	//	ログ出力
	private static Log m_log = LogFactory.getLog(CalculatedDataDAO.class);
	private static int GRAPH_MAX_PLOT = 10000;

	// SQL
	private static final String SQL_INSERT_00 = "INSERT INTO CC_CALCULATED_DATA " +
	"(COLLECTORID, " +
	"ITEM_CODE, " +
	"DISPLAY_NAME, " +
	"DATE_TIME, " +
	"FACILITYID, " +
	"VALUE) " +
	"values (?,?,?,?,?,?)";

	private static final String SQL_SELECT_00 = "SELECT DATE_TIME,VALUE FROM CC_CALCULATED_DATA WHERE " +
	"COLLECTORID=? AND " +
	"ITEM_CODE=? AND " +
	"DISPLAY_NAME=? AND " +
	"FACILITYID=? AND " +
	"DATE_TIME BETWEEN ? AND ?" +
	"ORDER BY DATE_TIME LIMIT " + GRAPH_MAX_PLOT;

	static {
		// グラフのitem当たりの最大プロット数を設定
		String graphMaxValueString = HinemosProperties.getProperty("performance.graph.max.plot");
		if(graphMaxValueString != null){
			int max = GRAPH_MAX_PLOT;
			try{
				max = Integer.parseInt(graphMaxValueString);
			} catch (NumberFormatException e) {
				m_log.warn("performance.graph.max.plot is not number . " + graphMaxValueString);
			}
			m_log.info("performance.graph.max.plot = " + max);
			GRAPH_MAX_PLOT = max;
		}
	}


	/**
	 * コンストラクター
	 */
	public CalculatedDataDAO(){
	}

	/**
	 * 収集された性能値を収集ID、SNMPのOID、ファシリティID、期間（開始の時刻、終了の時刻）で検索します。
	 *
	 * 
	 * @param collectorID 収集ID
	 * @param itemCode 収集項目コード
	 * @param displayName リポジトリ表示名
	 * @param facilityID ファシリティID
	 * @param startDate (検索の）開始時刻
	 * @param stopDate （検索の）終了時刻
	 * @return　収集された性能値のコレクション
	 * @throws EJBException
	 */
	public List<CollectedDataInfo> select(
			String collectorID,
			//			String collectMethod,
			String itemCode,
			String displayName,
			String facilityID,
			Date startDate,
			Date stopDate )
			throws EJBException{

		m_log.debug("select() start :" + collectorID + " " + itemCode + " " + displayName + " " + facilityID);
		ArrayList<CollectedDataInfo> ret = new ArrayList<CollectedDataInfo>();  // DBから取得したデータを格納するリスト

		Connection conn = null;
		PreparedStatement stmt =null;
		ResultSet res = null;
		try {
			// SQL文発行準備
			conn = ConnectionManager.getConnectionManager().getConnection();
			stmt = conn.prepareStatement(SQL_SELECT_00);

			Timestamp start = null;
			if (startDate != null) {
				start = new Timestamp(startDate.getTime());
			}
			Timestamp stop  = null;
			if (stopDate != null) {
				stop  = new Timestamp(stopDate.getTime());
			}

			stmt.setString(1, collectorID);
			stmt.setString(2, itemCode);
			stmt.setString(3, displayName);
			stmt.setString(4, facilityID);
			stmt.setTimestamp(5, start);
			stmt.setTimestamp(6, stop);

			m_log.debug("sql : " + SQL_SELECT_00);
			res = stmt.executeQuery();

			while(res.next()) {
				Date d = null;
				if (res.getTimestamp("DATE_TIME") != null) {
					d = new Date(res.getTimestamp("DATE_TIME").getTime());
				}
				CollectedDataInfo data;
				if(res.getObject("VALUE") != null){
					data = new CollectedDataInfo(facilityID, itemCode, displayName, d==null?null:d.getTime(), res.getDouble("VALUE"));
				} else {
					data = new CollectedDataInfo(facilityID, itemCode, displayName, d==null?null:d.getTime(), Double.NaN);
				}

				ret.add(data);
			}
		} catch (SQLException e) {
			m_log.error("select() error :" + facilityID + " EJBException ",e);
			throw new EJBException(e.getMessage());
		}  finally{

			//コネクション、結果セット、プリペアドステートメントのクロース
			try {
				if(res != null){
					res.close();
				}
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("select() error :" + facilityID + " EJBException ", e1);
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("select() end   :" + collectorID + " " + itemCode + " " + displayName + " " + facilityID);

		return ret;
	}

	/**
	 * 収集した性能値を収集IDで削除します。
	 * 
	 * 収集設定に紐付けられた性能値を一括で削除します。
	 * 
	 * @param collectorID　収集ID
	 * @throws SQLException
	 */
	public void delete(String collectorID)
	throws EJBException{

		m_log.debug("delete() start :" + collectorID );

		Connection conn = null;
		PreparedStatement stmt = null;

		try{
			conn = ConnectionManager.getConnectionManager().getConnection();
			//conn.setAutoCommit(false);
			//SQL文の定義
			String sql = "DELETE FROM CC_CALCULATED_DATA WHERE COLLECTORID = ? ";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);

			stmt.setString(1,collectorID);

			stmt.executeUpdate();
		} catch (SQLException e) {
			m_log.error("delete() error :" + collectorID + " EJBException ", e);
			throw new EJBException(e.getMessage());
		}  finally{

			//コネクション、結果セット、プリペアドステートメントのクロース
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("delete() error :" + collectorID + " EJBException ", e1);
				throw new EJBException(e1.getMessage());
			}
			m_log.debug("delete() end   :" + collectorID );
		}
	}


	private class PerfKey {
		private String collectorId = null;
		private String itemCode = null;
		private String displayName = null;
		private String facilityId = null;
		private PerfKey (String collectorId, String itemCode, String displayName, String facilityId) {
			this.collectorId = collectorId;
			this.itemCode = itemCode;
			this.displayName = displayName;
			this.facilityId = facilityId;
		}

		@Override
		public boolean equals(Object other) {
			PerfKey otherKey = (PerfKey) other;
			if (!collectorId.equals(otherKey.collectorId)){
				return false;
			}
			if (!itemCode.equals(otherKey.itemCode)){
				return false;
			}
			if (!displayName.equals(otherKey.displayName)){
				return false;
			}
			if (!facilityId.equals(otherKey.facilityId)){
				return false;
			}
			return true;
		}

		@Override
		public String toString() {
			return collectorId + "," + itemCode + "," + displayName + "," + facilityId;
		}

		@Override
		public int hashCode() {
			int result = collectorId.hashCode();
			result = 37 * result + itemCode.hashCode();
			result = 37 * result + displayName.hashCode();
			result = 37 * result + facilityId.hashCode();
			return result;
		}
	}
	private class PerfValue {
		private Date time = null;
		private double value = 0;
		private PerfValue (Date time, double value) {
			this.time = time;
			this.value = value;
		}
		private Date getTime () {
			return time;
		}
		private double getValue() {
			return value;
		}
	}

	// NaNのときに昔のデータを利用するためのキャッシュ。
	private static ConcurrentHashMap<PerfKey, PerfValue> perfCache =
		new ConcurrentHashMap<PerfKey, PerfValue>();
	private static int useOld = -1;

	/**
	 * 収集した性能値をインサートします。
	 * @param sample
	 * @throws EJBException
	 */
	public void insert(Sample sample)
	throws EJBException {
		if(m_log.isDebugEnabled()){
			m_log.debug("insert() start : collectorId = " + sample.getCollectorId() + ", dateTime = " + sample.getDateTime());
		}

		Connection conn = null;
		PreparedStatement stmt = null;
		Timestamp time = null;
		if (sample.getDateTime() != null) {
			time = new Timestamp(sample.getDateTime().getTime());
		}

		try{
			//SQL文のセット
			conn = ConnectionManager.getConnectionManager().getConnection();
			stmt = conn.prepareStatement(SQL_INSERT_00);

			//共通の値
			stmt.setString(1, sample.getCollectorId());
			stmt.setTimestamp(4, time);

			ArrayList<PerfData> list = sample.getPerfDataList();
			for (PerfData data : list){

				stmt.setString(2, data.getItemCode());
				stmt.setString(3, data.getDisplayName());
				stmt.setString(5, data.getFacilityId());

				PerfKey perfKey = new PerfKey(sample.getCollectorId(),
						data.getItemCode(),data.getDisplayName(),data.getFacilityId());

				// 性能値
				if(data.getErrorType() == CollectedDataErrorTypeConstant.NOT_ENOUGH_COLLECT_COUNT
						|| data.getErrorType() == CollectedDataErrorTypeConstant.POLLING_TIMEOUT
						|| data.getErrorType() == CollectedDataErrorTypeConstant.FACILITY_NOT_FOUND
						|| data.getErrorType() == CollectedDataErrorTypeConstant.UNKNOWN
						|| Double.isNaN(data.getValue())){
					/*
					 * 性能値が算出不能だった場合は前の値を利用する。
					 * 660秒以上前のデータは利用しない。
					 */
					try {
						if (useOld < 0) {
							useOld = Integer.parseInt(
									HinemosProperties.getProperty("performance.use.old", "660"));
						}
					} catch (Exception e) {
						m_log.warn("insert() : " + e.getMessage());
					}
					PerfValue perfValue = perfCache.get(perfKey);
					if (sample.getDateTime() != null &&
							perfValue != null &&
							perfValue.getTime() != null &&
							sample.getDateTime().getTime() - perfValue.getTime().getTime()
							< useOld * 1000) {
						m_log.info("insert() : use old perf-data " + perfKey);
						stmt.setDouble(6, perfValue.getValue());
					} else {
						stmt.setNull(6, Types.DOUBLE);
					}
				} else {
					// 性能値が算出可能だった場合はそのまま設定する
					stmt.setDouble(6, data.getValue());
					perfCache.put(
							perfKey,
							new PerfValue(sample.getDateTime(), data.getValue())
					);
				}

				int row = stmt.executeUpdate();

				if (row != 1) {
					String msg = "result row is not 1";
					throw new SQLException(msg);
				}
			}


		} catch (SQLException e) {
			m_log.error("insert() error : collectorId = " + sample.getCollectorId() + ", dateTime = " + sample.getDateTime(), e);
			throw new EJBException(e.getMessage());
		}  finally{

			//コネクション、結果セット、プリペアドステートメントのクロース
			try {
				if(stmt != null){
					stmt.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("insert() error : collectorId = " + sample.getCollectorId() + ", dateTime = " + sample.getDateTime(), e1);
				throw new EJBException(e1.getMessage());
			}
		}

		if(m_log.isDebugEnabled()){
			m_log.debug("insert() end : collectorId = " + sample.getCollectorId() + ", dateTime = " + sample.getDateTime());
		}
	}

	/**
	 * 指定された監視項目IDの収集データの中で、最新のものの日付時刻を返します
	 * @param monitorId 監視項目ID
	 * @return 収集データの中で最新の日付時刻
	 */
	public Date getLatestDate(String monitorId) {
		Date ret = null;

		m_log.debug("getLatestDate() start :" + monitorId );

		Connection conn = null;
		PreparedStatement stmt =null;
		ResultSet res = null;

		try {
			conn = ConnectionManager.getConnectionManager().getConnection();

			String preQuery = "SELECT date_time FROM cc_calculated_data WHERE collectorid=? ORDER BY date_time DESC LIMIT 1";

			stmt = conn.prepareStatement(preQuery);
			stmt.setString(1, monitorId);

			res = stmt.executeQuery();
			if (res.next()) {
				if (res.getTimestamp("date_time") != null) {
					ret = new Date(res.getTimestamp("date_time").getTime());
				}
			}


		} catch (SQLException e) {
			m_log.error("getLatestDate() error :" + monitorId + " EJBException ",e);
			throw new EJBException(e.getMessage());

		}  finally{

			//コネクション、結果セット、プリペアドステートメントのクロース
			try {
				if(res != null){
					res.close();
				}
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("getLatestDate() error :" + monitorId + " EJBException ", e1);
				throw new EJBException(e1.getMessage());
			}
		}

		return ret;
	}


	/**
	 * 指定された監視項目IDの収集データの中で、最古のものの日付時刻を返します
	 * @param monitorId 監視項目ID
	 * @return 収集データの中で最古の日付時刻
	 */
	public Date getOldestDate(String monitorId) {
		Date ret = null;

		m_log.debug("getOldestDate() start :" + monitorId );

		Connection conn = null;
		PreparedStatement stmt =null;
		ResultSet res = null;

		try {
			conn = ConnectionManager.getConnectionManager().getConnection();

			String preQuery = "SELECT date_time FROM cc_calculated_data WHERE collectorid=? ORDER BY date_time ASC LIMIT 1";

			stmt = conn.prepareStatement(preQuery);
			stmt.setString(1, monitorId);

			res = stmt.executeQuery();
			if (res.next()) {
				if (res.getTimestamp("date_time") != null) {
					ret = new Date(res.getTimestamp("date_time").getTime());
				}
			}


		} catch (SQLException e) {
			m_log.error("getOldestDate() error :" + monitorId + " EJBException ",e);
			throw new EJBException(e.getMessage());
		} finally {

			//コネクション、結果セット、プリペアドステートメントのクロース
			try {
				if(res != null){
					res.close();
				}
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("getOldestDate() error :" + monitorId + " EJBException ", e1);
				throw new EJBException(e1.getMessage());
			}
		}

		return ret;
	}


	/**
	 * 指定された監視項目IDおよびファシリティの収集データの中で、最新のものの日付時刻を返します
	 * @param monitorId 監視項目ID
	 * @param facilityId ファシリティID
	 * @return 収集データの中で最新の日付時刻
	 */
	public Date getLatestDate(String monitorId, String facilityId) {
		Date ret = null;

		m_log.debug("getLatestDate() start :" + monitorId + ", " + facilityId);

		Connection conn = null;
		PreparedStatement stmt =null;
		ResultSet res = null;

		try {
			conn = ConnectionManager.getConnectionManager().getConnection();

			String preQuery = "SELECT date_time FROM cc_calculated_data WHERE collectorid=? AND facilityid=? ORDER BY date_time DESC LIMIT 1";

			stmt = conn.prepareStatement(preQuery);
			stmt.setString(1, monitorId);
			stmt.setString(2, facilityId);

			res = stmt.executeQuery();
			if (res.next()) {
				if (res.getTimestamp("date_time") != null) {
					ret = new Date(res.getTimestamp("date_time").getTime());
				}
			}


		} catch (SQLException e) {
			m_log.error("getLatestDate() error :" + monitorId + ", " + facilityId + " EJBException ",e);
			throw new EJBException(e.getMessage());

		}  finally{

			//コネクション、結果セット、プリペアドステートメントのクロース
			try {
				if(res != null){
					res.close();
				}
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("getLatestDate() error :" + monitorId + ", " + facilityId + " EJBException ", e1);
				throw new EJBException(e1.getMessage());
			}
		}

		return ret;
	}


	/**
	 * 指定された監視項目IDおよびファシリティIDの収集データの中で、最古のものの日付時刻を返します
	 * @param monitorId 監視項目ID
	 * @param facilityId ファシリティID
	 * @return 収集データの中で最古の日付時刻
	 */
	public Date getOldestDate(String monitorId, String facilityId) {
		Date ret = null;

		m_log.debug("getOldestDate() start :" + monitorId + ", " + facilityId );

		Connection conn = null;
		PreparedStatement stmt =null;
		ResultSet res = null;

		try {
			conn = ConnectionManager.getConnectionManager().getConnection();

			String preQuery = "SELECT date_time FROM cc_calculated_data WHERE collectorid=? AND facilityid=? ORDER BY date_time ASC LIMIT 1";

			stmt = conn.prepareStatement(preQuery);
			stmt.setString(1, monitorId);
			stmt.setString(2, facilityId);

			res = stmt.executeQuery();
			if (res.next()) {
				if (res.getTimestamp("date_time") != null) {
					ret = new Date(res.getTimestamp("date_time").getTime());
				}
			}


		} catch (SQLException e) {
			m_log.error("getOldestDate() error :" + monitorId + ", " + facilityId + " EJBException ",e);
			throw new EJBException(e.getMessage());
		} finally {

			//コネクション、結果セット、プリペアドステートメントのクロース
			try {
				if(res != null){
					res.close();
				}
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("getOldestDate() error :" + monitorId + ", " + facilityId + " EJBException ", e1);
				throw new EJBException(e1.getMessage());
			}
		}

		return ret;
	}

	/**
	 * 指定された監視項目IDの収集データが存在するかどうかを返します。
	 * @param monitorId
	 * @return 存在している場合：true, 存在しない場合：false
	 */
	public boolean existCalculatedData(String monitorId) {
		boolean ret = false;

		m_log.debug("existCalculatedData() start :" + monitorId);

		Connection conn = null;
		PreparedStatement stmt =null;
		ResultSet res = null;

		try {
			conn = ConnectionManager.getConnectionManager().getConnection();

			String preQuery = "SELECT collectorid FROM cc_calculated_data WHERE collectorid=? LIMIT 1";

			stmt = conn.prepareStatement(preQuery);
			stmt.setString(1, monitorId);

			res = stmt.executeQuery();

			if (res.next()) {
				ret = true;
			}

		} catch (SQLException e) {
			m_log.error("existCalculatedData() error :" + monitorId + " EJBException " + e);
			throw new EJBException(e.getMessage());
		} finally {

			//コネクション、結果セット、プリペアドステートメントのクロース
			try {
				if(res != null){
					res.close();
				}
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("existCalculatedData() error :" + monitorId + " EJBException " + e1);
				throw new EJBException(e1.getMessage());
			}
		}

		return ret;
	}


	/**
	 * 指定した監視IDの収集データに対して、存在するItemCodeとその親のItemCode、デバイス名のリストを取得する
	 * 
	 * @param monitorId
	 * @return
	 */
	public ArrayList<CollectorItemParentInfo> getCollectorItemParentList(String monitorId, String monitorTypeId) {
		m_log.debug("getCollectorItemParentList() start : monitorId = " + monitorId + ", monitorTypeId = " + monitorTypeId);

		ArrayList<CollectorItemParentInfo> itemList = new ArrayList<CollectorItemParentInfo>();

		Connection conn = null;
		PreparedStatement stmt =null;
		ResultSet res = null;

		try {
			conn = ConnectionManager.getConnectionManager().getConnection();

			String preQuery = null;
			if ((HinemosModuleConstant.MONITOR_PERFORMANCE).equals(monitorTypeId)){
				preQuery =
					"SELECT distinct c.parent_item_code, d.item_code, d.display_name " +
					"FROM cc_calculated_data AS d, cc_collector_item_code_mst AS c " +
					"WHERE d.collectorid = ? AND d.item_code = c.item_code ORDER BY 2,3";
			}
			else{
				preQuery =
					"SELECT distinct null as parent_item_code, d.item_code, d.display_name " +
					"FROM cc_calculated_data AS d, cc_monitor_info AS m " +
					"WHERE d.collectorid = ? AND d.item_code = m.monitor_id ORDER BY 2,3";
			}
			m_log.debug("getCollectorItemParentList() sql : " + preQuery);

			stmt = conn.prepareStatement(preQuery);
			stmt.setString(1, monitorId);
			res = stmt.executeQuery();

			// 結果のセット
			while (res.next()) {
				String parentItemCode = res.getString(1);
				String itemCode = res.getString(2);
				String displayName = res.getString(3);
				m_log.debug("getCollectorItemParentList() : monitorId = " + monitorId + ", " +
						"parentItemCode = " + parentItemCode + ", " +
						"itemCode = " + itemCode + ", " +
						"displayName = " + displayName);
				CollectorItemParentInfo info = new CollectorItemParentInfo();
				info.setCollectorId(monitorId);
				info.setParentItemCode(parentItemCode);
				info.setItemCode(itemCode);
				info.setDisplayName(displayName);

				itemList.add(info);
			}
		} catch (SQLException e) {
			m_log.error("getCollectorItemParentList() error :" + monitorId, e);
			throw new EJBException(e.getMessage());
		} finally {

			//コネクション、結果セット、プリペアドステートメントのクロース
			try {
				if(res != null){
					res.close();
				}
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("getCollectorItemParentList() error :" + monitorId, e1);
				throw new EJBException(e1.getMessage());
			}
		}

		return itemList;
	}


	public static void main (String args[]) {
		CalculatedDataDAO dao = new CalculatedDataDAO();
		PerfValue value = null;

		PerfKey key1 = dao.new PerfKey("collectorId", "itemcode", "displayName", "facilityId");
		value = dao.new PerfValue(new Date(), 100);
		perfCache.put(key1, value);
		PerfKey key2 = dao.new PerfKey("collectorId", "itemcode", "displayName", "facilityId2");
		value = dao.new PerfValue(new Date(), 110);
		perfCache.put(key2, value);
		PerfKey key3 = dao.new PerfKey("collectorId", "itemcode", "displayName", "facilityId");
		value = dao.new PerfValue(new Date(), 120);
		perfCache.put(key3, value);

		System.out.println("perfCache.size=" + perfCache.keySet().size());
		for (PerfKey k : perfCache.keySet()) {
			System.out.println("key=" + k + ",hashCode=" + k.hashCode() + ", " + perfCache.get(key3).value);
		}
		System.out.println("equal=" + key1.equals(key3));
	}
}
