/*

Copyright (C) since 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.notify.util;

import java.util.StringTokenizer;

import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.bean.DataRangeConstant;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.commons.util.CommonValidator;
import com.clustercontrol.util.Messages;
import com.clustercontrol.notify.bean.NotifyCommandInfo;
import com.clustercontrol.notify.bean.NotifyEventInfo;
import com.clustercontrol.notify.bean.NotifyInfo;
import com.clustercontrol.notify.bean.NotifyJobInfo;
import com.clustercontrol.notify.bean.NotifyLogEscalateInfo;
import com.clustercontrol.notify.bean.NotifyMailInfo;
import com.clustercontrol.notify.bean.NotifyStatusInfo;
import com.clustercontrol.notify.bean.NotifyTypeConstant;
import com.clustercontrol.notify.mail.bean.MailTemplateInfo;

/**
 * 通知の入力チェッククラス
 * 
 * @since 4.0
 */
public class NotifyValidator {

	public static void validateNotifyInfo(NotifyInfo notifyInfo) throws InvalidSetting{
		// notifyId
		CommonValidator.validateId(Messages.getString("message.notify.10"),notifyInfo.getNotifyId(), 64);

		if (notifyInfo.getNotifyType() == NotifyTypeConstant.NOTIFY_ID_TYPE_SELFCHECK) {
			// たぶんメッセージがおかしい。けど、このルートは通らないはず。
			if (notifyInfo.getNotifyId() == null && ! notifyInfo.getNotifyId().matches("^" + HinemosModuleConstant.SYSYTEM_SELFCHECK + ".*")){
				// String[] replace = { HinemosModuleConstant.SYSYTEM_SELFCHECK };
				// throw new InvalidSetting(Messages.getString("message.notify.34", replace));
				throw new InvalidSetting("this notify id is not for selfcheck");
			}
		}

		// description
		CommonValidator.validateString(Messages.getString("description"), notifyInfo.getDescription(),
				false, 1, 256);
		// 再通知抑制期間
		if (notifyInfo.getRenotifyPeriod() != null) {
			CommonValidator.validateInt(Messages.getString("suppress.by.time.interval"), notifyInfo.getRenotifyPeriod(),
					1, DataRangeConstant.SMALLINT_HIGH);
		}
		// 初回通知するまでのカウント
		CommonValidator.validateInt(Messages.getString("notify.initial"), notifyInfo.getInitialCount(),
				1, MonitorResultStatusUpdater.MAX_INITIAL_NOTIFY_COUNT-1);

		// コマンド通知
		if (notifyInfo.getNotifyType() == NotifyTypeConstant.TYPE_COMMAND) {
			boolean flag = true;
			for (NotifyCommandInfo info : notifyInfo.getNotifyCommandInfo()) {
				if (info.getValidFlg() == ValidConstant.TYPE_VALID) {
					// 実効ユーザ
					if (info.getEffectiveUser() == null || info.getEffectiveUser().length() == 0) {
						throw new InvalidSetting(Messages.getString("message.notify.31"));
					}
					CommonValidator.validateString("effective.user",
							info.getEffectiveUser(), true, 1, 64);
					// 実行コマンド
					if (info.getCommand() == null || info.getCommand().length() == 0) {
						throw new InvalidSetting(Messages.getString("message.notify.32"));
					}
					flag = false;
					CommonValidator.validateString("command",
							info.getEffectiveUser(), true, 1, 1024);
				}
			}
			// 各重要度で１つも通知が選択されていない場合はエラー
			if (flag) {
				throw new InvalidSetting(Messages.getString("message.notify.13"));
			}
		}

		// イベント通知
		if (notifyInfo.getNotifyType() == NotifyTypeConstant.TYPE_EVENT) {
			boolean flag = true;
			for (NotifyEventInfo info : notifyInfo.getNotifyEventInfo()) {
				if (info.getValidFlg() == ValidConstant.TYPE_VALID) {
					flag = false;
				}
			}
			// 各重要度で１つも通知が選択されていない場合はエラー
			if (flag) {
				throw new InvalidSetting(Messages.getString("message.notify.13"));
			}
		}

		// ジョブ通知
		if (notifyInfo.getNotifyType() == NotifyTypeConstant.TYPE_JOB) {
			boolean flag = true;
			for (NotifyJobInfo info : notifyInfo.getNotifyJobInfo()) {
				if (info.getValidFlg() == ValidConstant.TYPE_VALID) {
					if (info.getJobId() == null || info.getJobId().length() == 0) {
						throw new InvalidSetting(Messages.getString("message.notify.20"));
					}
					flag = false;
				}
			}
			// 各重要度で１つも通知が選択されていない場合はエラー
			if (flag) {
				throw new InvalidSetting(Messages.getString("message.notify.13"));
			}
		}

		// ログエスカレーション通知
		if (notifyInfo.getNotifyType() == NotifyTypeConstant.TYPE_LOG_ESCALATE) {
			boolean flag = true;
			for (NotifyLogEscalateInfo info : notifyInfo.getNotifyLogEscalateInfo()){
				if (info.getValidFlg() == ValidConstant.TYPE_VALID){
					if (info.getSyslogFacility() == null) {
						String[] args = { "(" + PriorityConstant.typeToString(info.getPriority()) + ")"};
						throw new InvalidSetting(Messages.getString("message.notify.21", args));
					}
					if (info.getSyslogPriority() == null) {
						String[] args = { "(" + PriorityConstant.typeToString(info.getPriority()) + ")"};
						throw new InvalidSetting(Messages.getString("message.notify.22", args));
					}
					if (info.getEscalateMessage() == null || info.getEscalateMessage().length() == 0) {
						String[] args = { "(" + PriorityConstant.typeToString(info.getPriority()) + ")"};
						throw new InvalidSetting(Messages.getString("message.notify.23", args));
					}
					CommonValidator.validateString(Messages.getString("message"),
							info.getEscalateMessage(), true, 1, 1024);
					if (info.getEscalatePort() == null) {
						throw new InvalidSetting(Messages.getString("message.notify.24"));
					}
					CommonValidator.validateInt(Messages.getString("port.number"),
							info.getEscalatePort(), 1, DataRangeConstant.PORT_NUMBER_MAX);
					flag = false;
				}
			}
			// 各重要度で１つも通知が選択されていない場合はエラー
			if (flag) {
				throw new InvalidSetting(Messages.getString("message.notify.13"));
			}
		}

		// メール通知
		if (notifyInfo.getNotifyType() == NotifyTypeConstant.TYPE_MAIL) {
			boolean flag = true;
			for (NotifyMailInfo info : notifyInfo.getNotifyMailInfo()) {
				if (info.getValidFlg() == ValidConstant.TYPE_VALID) {
					if (info.getMailAddress() == null) {
						throw new InvalidSetting(Messages.getString("message.notify.18"));
					}
					CommonValidator.validateString(Messages.getString("email.address.ssv"),
							info.getMailAddress(), true, 1, 1024);
					//セミコロンで分ける
					StringTokenizer t = new StringTokenizer(info.getMailAddress(), ";");
					while (t.hasMoreTokens()) {
						String mailAddress = t.nextToken();
						//@マークで分ける
						String attr[] = mailAddress.split("@");
						if (attr.length != 2) {
							throw new InvalidSetting(Messages.getString("message.notify.18"));
						}
						flag = false;
					}
				}
			}
			// 各重要度で１つも通知が選択されていない場合はエラー
			if (flag) {
				throw new InvalidSetting(Messages.getString("message.notify.13"));
			}
		}

		// ステータス通知
		if (notifyInfo.getNotifyType() == NotifyTypeConstant.TYPE_STATUS) {
			boolean flag = true;
			for (NotifyStatusInfo info : notifyInfo.getNotifyStatusInfo()) {
				if (info.getValidFlg() == ValidConstant.TYPE_VALID) {
					if (info.getStatusUpdatePriority() == null) {
						throw new InvalidSetting(Messages.getString("message.notify.12"));
					}
					flag = false;
				}
			}
			// 各重要度で１つも通知が選択されていない場合はエラー
			if (flag) {
				throw new InvalidSetting(Messages.getString("message.notify.13"));
			}
		}
	}

	/**
	 * メールテンプレート情報の妥当性チェック
	 * 
	 * @param mailTemplateInfo
	 * @throws InvalidSetting
	 */
	public static void validateMailTemplateInfo(MailTemplateInfo mailTemplateInfo) throws InvalidSetting{
		// mailTemplateId
		CommonValidator.validateId(Messages.getString("mail.template.id"), mailTemplateInfo.getMailTemplateId(), 64);

		CommonValidator.validateString(Messages.getString("description"),
				mailTemplateInfo.getDescription(), false, 1, 256);
	}
}
