/*

 Copyright (C) 2006 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.notify.ejb.mdb;

import javax.jms.JMSException;
import javax.jms.ObjectMessage;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.monitor.bean.EventConfirmConstant;
import com.clustercontrol.notify.bean.NotifyRequestMessage;
import com.clustercontrol.notify.bean.OutputBasicInfo;
import com.clustercontrol.notify.util.OutputEvent;

/**
 * イベント出力を行う Message-Driven Bean <BR>
 * <p>監視管理画面のイベントビューに表示します。<BR>
 *
 * <!-- begin-xdoclet-definition -->
 * @ejb.bean name="NotifyEventBean"
 *     acknowledge-mode="Auto-acknowledge"
 *     destination-type="javax.jms.Queue"
 * 
 *     transaction-type="Container"
 *     destination-jndi-name="queue/clustercontrolNotifyEvent"
 * 
 * @jboss.depends name="user:service=HinemosService,name=HinemosService"
 * 
 * @jboss.container-configuration
 *  name="Hinemos Notification Message Driven Bean"
 * 
 *--
 * Server Runtime Specific Tags
 * If you are not using a specific runtime, you can safely remove the tags below.
 * @jonas.message-driven-destination jndi-name="queue/clustercontrolNotifyEvent"
 * @jboss.destination-jndi-name name="queue/clustercontrolNotifyEvent"
 *
 *--
 * <!-- end-xdoclet-definition -->
 **/
public class NotifyEventBean implements javax.ejb.MessageDrivenBean, javax.jms.MessageListener {

	private static final long serialVersionUID = -9053271265707732531L;

	/** ログ出力のインスタンス。 */
	private static Log m_log = LogFactory.getLog( NotifyEventBean.class );

	/** イベント出力 */
	private static OutputEvent m_output = new OutputEvent();

	/** コンテキスト情報 */
	private javax.ejb.MessageDrivenContext messageContext = null;

	/**
	 * コンテキスト情報を設定します。<BR>
	 * Message-Driven Bean がインスタンスプールに格納される際に行う処理を実装します。
	 * @see javax.ejb.MessageDrivenBean#setMessageDrivenContext(javax.ejb.MessageDrivenContext)
	 */
	@Override
	public void setMessageDrivenContext(
			javax.ejb.MessageDrivenContext messageContext)
	throws javax.ejb.EJBException {
		this.messageContext = messageContext;
	}

	/**
	 * Message-Driven Bean が生成される際に行う処理を実装します。
	 * @ejb.create-method
	 */
	public void ejbCreate() {
		//no specific action required for message-driven beans
	}

	/**
	 * Message-Driven Bean が削除される際に行う処理を実装します。
	 * @see javax.ejb.MessageDrivenBean#ejbRemove()
	 */
	@Override
	public void ejbRemove() {
		messageContext = null;
	}

	/**
	 * 引数で指定された受信メッセージを元に、イベント通知をおこないます。<BR>
	 * Message-Driven Bean で行うビジネスロジックを実装します。
	 * 特定のメッセージを受信した場合のみ処理を行います。受信メッセージは下記の通りです。
	 * 
	 * <p><li>{@link com.clustercontrol.notify.message.OutputNotifyGroupInfo} :
	 * 含まれている通知出力情報を基にイベントを表示します。
	 * 
	 * @see javax.jms.MessageListener#onMessage(javax.jms.Message)
	 * 
	 * @see com.clustercontrol.notify.util.OutputEvent#outputEvent(OutputNotifyGroupInfo)
	 * @see com.clustercontrol.notify.message.OutputNotifyGroupInfo
	 */
	@Override
	@SuppressWarnings("unchecked")
	public void onMessage(javax.jms.Message message) {
		m_log.debug("Message Driven Bean got message " + message);

		if(message instanceof ObjectMessage) {
			ObjectMessage msg = (ObjectMessage)message;

			try
			{
				Object objMsg = msg.getObject();
				if(objMsg instanceof NotifyRequestMessage) {
					NotifyRequestMessage requestMessage = (NotifyRequestMessage)objMsg;
					m_output.notify(requestMessage);
				} else if(objMsg instanceof OutputBasicInfo){
					// 未確認状態でイベントに出力する
					m_output.insertEventLog((OutputBasicInfo)objMsg, EventConfirmConstant.TYPE_UNCONFIRMED);
				} else {
					m_log.error("onMessage(): ObjectMessage is not an expected instance. " + objMsg.getClass().getCanonicalName());
				}
			}
			catch(JMSException e) {
				m_log.error("onMessage():" + messageContext.toString() + e.getMessage(),e);
			}
			catch(Exception e) {
				m_log.error("onMessage():" + e.getMessage(),e);
			}
		}
	}
}