/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.monitor.run.factory;

import java.sql.Timestamp;
import java.util.Collection;
import java.util.Date;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.InvalidTransactionException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.SystemException;
import javax.transaction.Transaction;
import javax.transaction.TransactionManager;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.MonitorNotFound;
import com.clustercontrol.fault.NotifyNotFound;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.commons.bean.SettingUpdateInfo;
import com.clustercontrol.commons.scheduler.TriggerSchedulerException;
import com.clustercontrol.custom.util.CustomManagerUtil;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoLocal;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoUtil;
import com.clustercontrol.notify.ejb.entity.MonitorStatusLocal;
import com.clustercontrol.notify.ejb.entity.MonitorStatusPK;
import com.clustercontrol.notify.ejb.entity.MonitorStatusUtil;
import com.clustercontrol.notify.ejb.entity.NotifyHistoryLocal;
import com.clustercontrol.notify.ejb.entity.NotifyHistoryUtil;
import com.clustercontrol.notify.ejb.session.NotifyControllerLocal;
import com.clustercontrol.notify.ejb.session.NotifyControllerUtil;
import com.clustercontrol.notify.util.MonitorResultStatusUpdater;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * 監視情報を変更する抽象クラスです。
 * <p>
 * 監視種別（真偽値，数値，文字列）の各クラスで継承してください。
 *
 * @version 4.0.0
 * @since 2.0.0
 */
abstract public class ModifyMonitor {
	/** ログ出力のインスタンス。 */
	private static Log m_log = LogFactory.getLog( ModifyMonitor.class );

	/** 監視情報のローカルコンポーネント。 */
	protected MonitorInfoLocal m_monitor;

	/** 監視情報。 */
	protected MonitorInfo m_monitorInfo;

	/** ファシリティ変更フラグ。 */
	protected boolean m_isModifyFacilityId = false;

	/** 実行間隔変更フラグ。 */
	protected boolean m_isModifyRunInterval = false;

	/** 有効への変更フラグ。 */
	protected boolean m_isModifyEnableFlg = false;

	/**
	 * スケジュール実行種別を返します。
	 */
	protected abstract String getTriggerType();

	/**
	 * スケジュール実行の遅延時間を返します。
	 */
	protected abstract int getDelayTime();

	/**
	 * トランザクションを開始し、引数で指定された監視情報を変更します。
	 * 
	 * @param info 監視情報
	 * @param user 最終変更ユーザ
	 * @return 変更に成功した場合、</code> true </code>
	 * @throws InvalidSetting
	 */
	public boolean modify(MonitorInfo info, String user) throws CreateException, RemoveException, NamingException, NotSupportedException, HeuristicMixedException, HeuristicRollbackException, RollbackException, InvalidTransactionException, IllegalStateException, SystemException, NotifyNotFound, HinemosUnknown, MonitorNotFound, TriggerSchedulerException {

		m_monitorInfo = info;

		TransactionManager tm = null;
		Transaction oldTx = null;
		boolean result = false;

		try
		{
			// TransactionManagerを取得
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");

			// 実行中のスレッドがトランザクションに関連付けられている場合は、トランザクションを退避
			if(tm.getTransaction() != null){
				oldTx = tm.suspend();
			}

			try{
				// トランザクション開始
				if(tm.getTransaction() == null){
					tm.begin();
				}

				// 監視情報を変更
				result = modifyMonitorInfo(user);
				if(result){
					// コミット
					tm.commit();
				}

				if (info != null) {
					if (HinemosModuleConstant.MONITOR_CUSTOM.equals(info.getMonitorTypeId())) {
						// read-committedのため、commit後に外部コンポーネントに通知する
						CustomManagerUtil.broadcastConfigured();
					}
					if (HinemosModuleConstant.MONITOR_SYSTEMLOG.equals(info.getMonitorTypeId())) {
						SettingUpdateInfo.getInstance().setSystemLogMonitorUpdateTime(System.currentTimeMillis());
					}
					if (HinemosModuleConstant.MONITOR_SNMPTRAP.equals(info.getMonitorTypeId())) {
						SettingUpdateInfo.getInstance().setSnmptrapMonitorUpdateTime(System.currentTimeMillis());
					}
				}

			} catch(NotSupportedException e){
				outputLog(e, "modify()");
				throw e;
			} catch (RollbackException e) {
				outputLog(e, "modify()");
				throw e;
			} catch (HeuristicMixedException e) {
				outputLog(e, "modify()");
				throw e;
			} catch(HeuristicRollbackException e){
				outputLog(e, "modify()");
				throw e;
			} catch(SystemException e){
				outputLog(e, "modify()");
				throw e;
			}
			finally{
				// トランザクション関連の例外が発生した場合は、ロールバック
				if(tm.getTransaction() != null){
					if(!result){
						tm.rollback();
					}
				}
			}
		}
		finally{

			// 一時停止していたトランザクションを再開
			if(oldTx != null){
				try{
					tm.resume(oldTx);

				} catch(InvalidTransactionException e){
					outputLog(e, "modify()");
					throw e;
				} catch(IllegalStateException e){
					outputLog(e, "modify()");
					throw e;
				} catch(SystemException e){
					outputLog(e, "modify()");
					throw e;
				}
			}
		}
		return result;
	}

	/**
	 * 判定情報を変更し、監視情報に設定します。
	 * <p>
	 * 各監視種別（真偽値，数値，文字列）のサブクラスで実装します。
	 * 
	 * @return 変更に成功した場合、</code> true </code>
	 * @throws CreateException
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws HinemosUnknown
	 * @throws NotifyNotFound
	 */
	protected abstract boolean modifyJudgementInfo() throws CreateException, RemoveException, NamingException, HinemosUnknown, NotifyNotFound;

	/**
	 * チェック条件情報を変更し、監視情報に設定します。
	 * <p>
	 * 各監視管理のサブクラスで実装します。
	 * 
	 * @return 変更に成功した場合、</code> true </code>
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 */
	protected abstract boolean modifyCheckInfo() throws CreateException, FinderException, NamingException;

	/**
	 * 監視情報を作成します。
	 * <p>
	 * <ol>
	 * <li>監視対象IDと監視項目IDより、監視情報を取得します。</li>
	 * <li>監視情報を、引数で指定されたユーザで変更します。</li>
	 * <li>判定情報を変更し、監視情報に設定します。各監視種別（真偽値，数値，文字列）のサブクラスで実装します（{@link #modifyJudgementInfo()}）。</li>
	 * <li>チェック条件情報を変更し、監視情報に設定します。各監視管理のサブクラスで実装します（{@link #modifyCheckInfo()}）。</li>
	 * <li>実行間隔 もしくは 有効/無効が変更されている場合は、Quartzの登録を変更します。</li>
	 * </ol>
	 * 
	 * @param user ユーザ
	 * @return 更新に成功した場合、true
	 * @throws CreateException
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws HinemosUnknown
	 * @throws NotifyNotFound
	 * @throws MonitorNotFound
	 * @throws TriggerSchedulerException
	 */
	protected boolean modifyMonitorInfo(String user) throws CreateException, RemoveException, NamingException, NotifyNotFound, HinemosUnknown, MonitorNotFound, TriggerSchedulerException{
		Timestamp now = new Timestamp(new Date().getTime());

		try{
			// 監視情報を設定
			m_monitor = MonitorInfoUtil.getLocalHome().findByPrimaryKey(m_monitorInfo.getMonitorId());

			// ファシリティIDが変更されているか
			if(!m_monitorInfo.getFacilityId().equals(m_monitor.getFacilityId())){
				m_isModifyFacilityId = true;
			}
			// 監視/収集間隔が変更されているか
			if(m_monitorInfo.getRunInterval() != m_monitor.getRunInterval().intValue()){
				m_isModifyRunInterval = true;
			}
			// 監視/収集無効から有効(監視or収集のいずれか1つでも)に変更されているか
			if(m_monitor.getMonitorFlg().intValue() == YesNoConstant.TYPE_NO &&
					m_monitor.getCollectorFlg().intValue() == YesNoConstant.TYPE_NO &&
					(m_monitorInfo.getMonitorFlg() == YesNoConstant.TYPE_YES || m_monitorInfo.getCollectorFlg() == YesNoConstant.TYPE_YES)){
				m_isModifyEnableFlg = true;
			}
			m_log.debug("modifyMonitorInfo() m_isModifyFacilityId = " + m_isModifyFacilityId
					+ ", m_isModifyRunInterval = " + m_isModifyRunInterval
					+ ", m_isModifyEnableFlg = " + m_isModifyEnableFlg);

			m_monitor.setDescription(m_monitorInfo.getDescription());
			if(m_isModifyFacilityId)
				m_monitor.setFacilityId(m_monitorInfo.getFacilityId());
			if(m_isModifyRunInterval)
				m_monitor.setRunInterval(m_monitorInfo.getRunInterval());
			m_monitor.setDelayTime(getDelayTime());
			m_monitor.setTriggerType(getTriggerType());
			m_monitor.setCalendarId(m_monitorInfo.getCalendarId());
			m_monitor.setFailurePriority(m_monitorInfo.getFailurePriority());
			m_monitor.setApplication(m_monitorInfo.getApplication());
			m_monitor.setNotifyGroupId(m_monitorInfo.getNotifyGroupId());
			m_monitor.setMonitorFlg(m_monitorInfo.getMonitorFlg());
			m_monitor.setCollectorFlg(m_monitorInfo.getCollectorFlg());
			m_monitor.setItemName(m_monitorInfo.getItemName());
			m_monitor.setMeasure(m_monitorInfo.getMeasure());
			m_monitor.setUpdateDate(now);
			m_monitor.setUpdateUser(user);

			NotifyControllerLocal nc = NotifyControllerUtil.getLocalHome().create();
			nc.modifyNotifyRelation(m_monitorInfo.getNotifyId(), m_monitorInfo.getNotifyGroupId());

			// 判定情報を設定
			if(modifyJudgementInfo()){

				// チェック条件情報を設定
				if(modifyCheckInfo()){

					// Quartzの登録情報を変更
					new ModifySchedule().updateSchedule(m_monitorInfo.getMonitorId());

					// この監視設定の監視結果状態を削除する
					Collection<MonitorStatusLocal> statusList =
						MonitorStatusUtil.getLocalHome().findByPluginIdAndMonitorId(m_monitor.getMonitorTypeId(), m_monitor.getMonitorId());
					for(MonitorStatusLocal status : statusList){
						MonitorStatusPK pk = (MonitorStatusPK)status.getPrimaryKey();
						status.remove();
						MonitorResultStatusUpdater.removeCache(pk);
					}

					// この監視設定の結果として通知された通知履歴を削除する
					Collection<NotifyHistoryLocal> historyList = NotifyHistoryUtil.getLocalHome().findByPluginIdAndMonitorId(m_monitor.getMonitorTypeId(), m_monitor.getMonitorId());
					for(NotifyHistoryLocal history : historyList){
						history.remove();
					}

					return true;
				}
			}
			return false;

		} catch (CreateException e) {
			outputLog(e, "modifyMonitorInfo()");
			throw e;
		} catch (FinderException e) {
			outputLog(e, "modifyMonitorInfo()");
			throw new MonitorNotFound(e.getMessage(), e);
		} catch (RemoveException e) {
			outputLog(e, "modifyMonitorInfo()");
			throw e;
		} catch (TriggerSchedulerException e) {
			outputLog(e, "modifyMonitorInfo()");
			throw e;
		} catch (NamingException e) {
			outputLog(e, "modifyMonitorInfo()");
			throw e;
		}
	}

	/**
	 * アプリケーションログにログを出力します。
	 * 
	 * @param e 例外
	 * @param method メソッド名
	 */
	private void outputLog(Exception e, String method) {
		AplLogger apllog = new AplLogger("MON", "mon");
		String[] args = {m_monitorInfo.getMonitorTypeId(), m_monitorInfo.getMonitorId() };
		apllog.put("SYS", "008", args);
		m_log.debug(method + ":" + e.getMessage());
	}
}
