/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.monitor.run.factory;

import java.sql.Timestamp;
import java.util.Calendar;
import java.util.Date;
import java.util.Random;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.FinderException;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.InvalidTransactionException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.SystemException;
import javax.transaction.Transaction;
import javax.transaction.TransactionManager;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.MonitorDuplicate;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.commons.bean.SettingUpdateInfo;
import com.clustercontrol.commons.scheduler.TriggerSchedulerException;
import com.clustercontrol.custom.util.CustomManagerUtil;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoUtil;
import com.clustercontrol.notify.ejb.session.NotifyControllerLocal;
import com.clustercontrol.notify.ejb.session.NotifyControllerUtil;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * 監視情報を作成する抽象クラス<BR>
 * <p>
 * 監視種別（真偽値，数値，文字列）の各クラスで継承してください。
 *
 * @version 4.0.0
 * @since 2.0.0
 */
abstract public class AddMonitor {

	/** ログ出力のインスタンス。 */
	private static Log m_log = LogFactory.getLog( AddMonitor.class );

	/** 監視情報。 */
	protected MonitorInfo m_monitorInfo;

	/**
	 * トランザクションを開始し、引数で指定された監視情報を作成します。
	 * 
	 * @param info 監視情報
	 * @param user 新規作成ユーザ
	 * @return 作成に成功した場合、</code> true </code>
	 * @throws CreateException
	 * @throws NamingException
	 * @throws NotSupportedException
	 * @throws HeuristicMixedException
	 * @throws HeuristicRollbackException
	 * @throws RollbackException
	 * @throws InvalidTransactionException
	 * @throws IllegalStateException
	 * @throws SystemException
	 * @throws HinemosUnknown
	 * @throws MonitorDuplicate
	 * @throws TriggerSchedulerException
	 * @throws InvalidSetting
	 * @see #addMonitorInfo(String)
	 */
	public boolean add(MonitorInfo info, String user) throws CreateException, NamingException, NotSupportedException, HeuristicMixedException, HeuristicRollbackException, RollbackException, InvalidTransactionException, IllegalStateException, SystemException, HinemosUnknown, MonitorDuplicate, TriggerSchedulerException {

		m_monitorInfo = info;

		TransactionManager tm = null;
		Transaction oldTx = null;
		boolean result = false;

		try
		{
			// TransactionManagerを取得
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");

			// 実行中のスレッドがトランザクションに関連付けられている場合は、トランザクションを退避
			if(tm.getTransaction() != null){
				oldTx = tm.suspend();
			}

			try{
				// トランザクション開始
				if(tm.getTransaction() == null){
					tm.begin();
				}

				// 監視情報を登録
				result = addMonitorInfo(user);

				if(result){
					// コミット
					tm.commit();

					if (info != null) {
						if (HinemosModuleConstant.MONITOR_CUSTOM.equals(info.getMonitorTypeId())) {
							// read-committedのため、commit後に外部コンポーネントに通知する
							CustomManagerUtil.broadcastConfigured();
						}
						if (HinemosModuleConstant.MONITOR_SYSTEMLOG.equals(info.getMonitorTypeId())) {
							SettingUpdateInfo.getInstance().setSystemLogMonitorUpdateTime(System.currentTimeMillis());
						}
						if (HinemosModuleConstant.MONITOR_SNMPTRAP.equals(info.getMonitorTypeId())) {
							SettingUpdateInfo.getInstance().setSnmptrapMonitorUpdateTime(System.currentTimeMillis());
						}
					}
				}

			} catch(NotSupportedException e){
				outputLog(e, "add()");
				throw e;
			} catch (RollbackException e) {
				outputLog(e, "add()");
				throw e;
			} catch (HeuristicMixedException e) {
				outputLog(e, "add()");
				throw e;
			} catch(HeuristicRollbackException e){
				outputLog(e, "add()");
				throw e;
			} catch(SystemException e){
				outputLog(e, "add()");
				throw e;
			} catch (CreateException e) {
				if (e instanceof DuplicateKeyException) {
					m_log.warn("addAgent " + e.getMessage());
					throw new MonitorDuplicate(e.getMessage(),e);
				} else {
					throw e;
				}
			}
			finally{
				// トランザクション関連の例外が発生した場合は、ロールバック
				if(tm.getTransaction() != null){
					if(!result){
						tm.rollback();
					}
				}
			}
		}
		finally{

			// 一時停止していたトランザクションを再開
			if(oldTx != null){
				try{
					tm.resume(oldTx);

				} catch(InvalidTransactionException e){
					outputLog(e, "add()");
					throw e;
				} catch(IllegalStateException e){
					outputLog(e, "add()");
					throw e;
				} catch(SystemException e){
					outputLog(e, "add()");
					throw e;
				}
			}
		}
		return result;
	}

	/**
	 * 判定情報を作成し、監視情報に設定します。
	 * <p>
	 * 各監視種別（真偽値，数値，文字列）のサブクラスで実装します。
	 * 
	 * @return 作成に成功した場合、</code> true </code>
	 * @throws CreateException
	 * @throws NamingException
	 * @throws HinemosUnknown
	 */
	protected abstract boolean addJudgementInfo() throws CreateException, NamingException, HinemosUnknown;

	/**
	 * チェック条件情報を作成し、監視情報に設定します。
	 * <p>
	 * 各監視管理のサブクラスで実装します。
	 * 
	 * @return 作成に成功した場合、</code> true </code>
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 */
	protected abstract boolean addCheckInfo() throws CreateException, NamingException;

	/**
	 * スケジュール実行種別を返します。
	 */
	protected abstract String getTriggerType();

	/**
	 * スケジュール実行の遅延時間を返します。
	 */
	protected abstract int getDelayTime();

	/**
	 * 監視項目IDをベースにスケジュール実行の遅延時間を生成して返します。
	 */
	public static int getDelayTimeBasic(MonitorInfo monitorInfo){
		// 再起動時も常に同じタイミングでQuartzのTriggerが起動されるように収集種別と収集項目IDから、DelayTimeを生成する

		// 収集種別と収集項目IDを結合した文字列のhashを求める
		int hashCode = (monitorInfo.getMonitorId() + monitorInfo.getMonitorType()).hashCode();

		// hashをシードとして乱数を作成する。このとき乱数の範囲は、0～(monitorInfo-1)とする
		int offsetSecond = new Random(hashCode).nextInt(monitorInfo.getRunInterval());
		m_log.debug("MonitorID : " + monitorInfo.getMonitorId()
				+ ", MonitorType : " + monitorInfo.getMonitorType()
				+ ", offset : " + offsetSecond);

		return offsetSecond;
	}


	/**
	 * 監視情報を作成します。
	 * <p>
	 * <ol>
	 * <li>監視情報を、引数で指定されたユーザで作成します。</li>
	 * <li>判定情報を作成し、監視情報に設定します。各監視種別（真偽値，数値，文字列）のサブクラスで実装します（{@link #addJudgementInfo()}）。</li>
	 * <li>チェック条件情報を作成し、監視情報に設定します。各監視管理のサブクラスで実装します（{@link #addCheckInfo()}）。</li>
	 * <li>Quartzに、スケージュールと監視情報の有効/無効を登録します。</li>
	 * </ol>
	 * 
	 * @param user 新規作成ユーザ
	 * @return 作成に成功した場合、</code> true </code>
	 * @throws CreateException
	 * @throws NamingException
	 * @throws HinemosUnknown
	 * @throws TriggerSchedulerException
	 * 
	 * @see com.clustercontrol.monitor.run.ejb.entity.MonitorInfoBean
	 * @see #addJudgementInfo()
	 * @see #addCheckInfo()
	 * @see com.clustercontrol.monitor.run.factory.ModifySchedule#addSchedule(MonitorInfo, String, Calendar)
	 */
	protected boolean addMonitorInfo(String user) throws CreateException, NamingException, HinemosUnknown, TriggerSchedulerException {
		Timestamp now = new Timestamp(new Date().getTime());

		try{
			// 監視情報を挿入
			MonitorInfoUtil.getLocalHome().create(
					m_monitorInfo.getMonitorId(),
					m_monitorInfo.getMonitorTypeId(),
					m_monitorInfo.getMonitorType(),
					m_monitorInfo.getDescription(),
					m_monitorInfo.getFacilityId(),
					m_monitorInfo.getRunInterval(),
					getDelayTime(),
					getTriggerType(),
					m_monitorInfo.getCalendarId(),
					m_monitorInfo.getFailurePriority(),
					m_monitorInfo.getApplication(),
					m_monitorInfo.getNotifyGroupId(),
					m_monitorInfo.getMonitorFlg(),
					m_monitorInfo.getCollectorFlg(),
					m_monitorInfo.getItemName(),
					m_monitorInfo.getMeasure(),
					now,
					now,
					user,
					user
			);

			// 通知情報を投入
			if(m_monitorInfo.getNotifyGroupId() != null){
				NotifyControllerLocal nc = NotifyControllerUtil.getLocalHome().create();
				nc.addNotifyRelation(m_monitorInfo.getNotifyId());
			}

			// 判定情報を設定
			if(addJudgementInfo()){
				// チェック条件情報を設定
				if(addCheckInfo()){
					// Quartzに登録(runInterval = 0 -> スケジュール起動を行わない監視)
					if(m_monitorInfo.getRunInterval() > 0){
						ModifySchedule quartz = new ModifySchedule();
						quartz.updateSchedule(m_monitorInfo.getMonitorId());
					}
					return true;
				}
			}
			return false;

		} catch (CreateException e) {
			outputLog(e, "addMonitorInfo()");
			throw e;
		} catch (TriggerSchedulerException e) {
			outputLog(e, "addMonitorInfo()");
			throw e;
		} catch (NamingException e) {
			outputLog(e, "addMonitorInfo()");
			throw e;
		}
	}

	/**
	 * アプリケーションログにログを出力します。
	 * 
	 * @param e 例外
	 * @param method メソッド名
	 */
	private void outputLog(Exception e, String method){
		AplLogger apllog = new AplLogger("MON", "mon");
		String[] args = {m_monitorInfo.getMonitorTypeId(), m_monitorInfo.getMonitorId() };
		apllog.put("SYS", "007", args);
		m_log.debug(method + ":" + e.getMessage());
	}
}
