/*

Copyright (C) since 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.jobmanagement.util;


import java.util.Collection;

import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.commons.util.CommonValidator;
import com.clustercontrol.jobmanagement.bean.JobCommandInfo;
import com.clustercontrol.jobmanagement.bean.JobConstant;
import com.clustercontrol.jobmanagement.bean.JobFileInfo;
import com.clustercontrol.jobmanagement.bean.JobInfo;
import com.clustercontrol.jobmanagement.bean.JobSchedule;
import com.clustercontrol.jobmanagement.bean.JobTreeItem;
import com.clustercontrol.jobmanagement.bean.OperationConstant;
import com.clustercontrol.jobmanagement.bean.SystemParameterConstant;
import com.clustercontrol.jobmanagement.ejb.entity.JobMasterPK;
import com.clustercontrol.jobmanagement.ejb.entity.JobMasterUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobScheduleLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobScheduleUtil;
import com.clustercontrol.repository.ejb.entity.FacilityLocal;
import com.clustercontrol.repository.ejb.entity.FacilityUtil;
import com.clustercontrol.util.Messages;

/**
 * ジョブ管理の入力チェッククラス
 * 
 * @since 4.0
 */
public class JobValidator {
	private static Log m_log = LogFactory.getLog( JobValidator.class );

	/**
	 * ジョブスケジュールのvalidate
	 * 
	 * @param jobSchedule
	 * @throws InvalidSetting
	 * @throws HinemosUnknown
	 */
	public static void validateJobSchedule(JobSchedule jobSchedule) throws InvalidSetting, HinemosUnknown{

		String id = jobSchedule.getId();
		// scheduleId
		CommonValidator.validateId(Messages.getString("schedule.id"), id, 64);

		// schedule
		CommonValidator.validateSchedule(jobSchedule.getSchedule());

		// scheduleName
		CommonValidator.validateString(Messages.getString("schedule.name"), jobSchedule.getName(), true, 1, 64);

		// jobid
		validateJobId(jobSchedule.getJobunitId(),jobSchedule.getJobId());

		// calenderId
		CommonValidator.validateCalenderId(jobSchedule.getCalendarId(), false);
	}


	/**
	 * ジョブIDの存在チェック
	 * @param jobunitId
	 * @param jobId
	 * @throws InvalidSetting
	 */
	public static void validateJobId (String jobunitId, String jobId) throws InvalidSetting {
		JobMasterPK pk = new JobMasterPK(jobunitId, jobId);
		try {
			JobMasterUtil.getLocalHome().findByPrimaryKey(pk);
		} catch (Exception e) {
			throw new InvalidSetting(Messages.getString("message.job.1") +
					" Target job is not exist! jobunitId = " + jobunitId +
					", jobId = " + jobId);
		}
	}

	/**
	 * ジョブを登録、変更、削除した際に各種参照に問題がないか
	 * 
	 * @throws InvalidSetting
	 * @throws HinemosUnknown
	 */
	public static void validateJobMaster() throws InvalidSetting, HinemosUnknown{
		m_log.debug("validateJobMaster()");

		// ジョブスケジュールの参照
		m_log.debug("validateJobMaster() jobschedule check start");
		try{
			Collection<JobScheduleLocal> jobScheduleList = JobScheduleUtil.getLocalHome().findAll();
			for(JobScheduleLocal jobSchedule : jobScheduleList){
				String jobunitId = jobSchedule.getJobunit_id();
				String jobId = jobSchedule.getJob_id();

				m_log.debug("validateJobMaster() target jobschedule " + jobSchedule.getSchedule_id() + ", jobunitId = " + jobunitId + ", jobId = " + jobId);
				try{
					// jobunitId,jobidの村税チェック
					validateJobId(jobunitId,jobId);

					// 削除対象のジョブツリーの中にジョブスケジュールからの参照がある
					String[] args = {jobSchedule.getSchedule_id()};
					m_log.debug(Messages.getString("message.job.81", args));
				} catch (InvalidSetting e) {
					String[] args = {jobSchedule.getSchedule_id(), jobunitId, jobId};
					throw new InvalidSetting(Messages.getString("message.job.82", args));
				}
			}

		} catch (NamingException e) {
			throw new HinemosUnknown(e);
		} catch (FinderException e) {
			throw new HinemosUnknown(e);
		}

	}

	/**
	 * ジョブ定義のvalidate
	 * @throws InvalidSetting
	 * @throws HinemosUnknown
	 */
	public static void validateJobUnit(JobTreeItem item) throws InvalidSetting, HinemosUnknown{

		if(item == null){
			return ;
		}

		if(item.getData() != null && item.getData() instanceof JobInfo){
			JobInfo info = item.getData();

			// ジョブID
			String jobId = info.getId();
			CommonValidator.validateId(Messages.getString("job.id"), jobId, 64);

			// ジョブユニットID
			String jobunitId = info.getJobunitId();
			CommonValidator.validateId(Messages.getString("jobunit.id"), jobunitId, 64);

			// ジョブ名
			String jobName = info.getName();
			CommonValidator.validateString(Messages.getString("job.name"), jobName, true, 1, 64);

			// 説明
			String description = info.getDescription();
			CommonValidator.validateString(Messages.getString("description"), description, true, 0, 256);

			// ジョブユニットの場合は、jobIdとjobunitIdは一緒。
			if (info.getType() == JobConstant.TYPE_JOBUNIT) {
				if (!jobId.equals(jobunitId)) {
					throw new InvalidSetting("jobType is TYPE_JOBUNIT, but jobId != jobunitId");
				}
			}

			// ジョブの場合は、ファシリティIDの存在チェック
			if (info.getType() == JobConstant.TYPE_JOB) {
				JobCommandInfo command = info.getCommand();

				// 実行するファシリティIDのチェック
				if(command.getFacilityID() == null || "".equals(command.getFacilityID())){
					throw new InvalidSetting(Messages.getString("message.hinemos.3"));
				}else{

					// ジョブ変数でない場合は、ファシリティIDのチェックを行う
					if(!SystemParameterConstant.isParam(
							command.getFacilityID(),
							SystemParameterConstant.FACILITY_ID)){
						try{
							FacilityUtil.getLocalHome().findByPrimaryKey(command.getFacilityID());
						} catch (NamingException e) {
							throw new HinemosUnknown("add job unknown error. FacilityId = " + command.getFacilityID() + ", jobunitId = " + jobunitId + ", jobId = " + jobId, e);
						} catch (FinderException e) {
							throw new InvalidSetting("FacilityId is not exist in repository. FacilityId = " + command.getFacilityID() + ", jobunitId = " + jobunitId + ", jobId = " + jobId);
						}
					}

				}
			} else if (info.getType() == JobConstant.TYPE_FILEJOB) {
				JobFileInfo file = info.getFile();

				// 送信元ファシリティID(ノード)
				if(file.getSrcFacilityID() == null || "".equals(file.getSrcFacilityID())){
					throw new InvalidSetting(Messages.getString("message.hinemos.2"));
				}else{
					try{
						FacilityLocal facility = FacilityUtil.getLocalHome().findByPrimaryKey(file.getSrcFacilityID());
						if(!facility.isNode()){
							throw new InvalidSetting("Src FacilityId is not node. Src FacilityId = " + file.getSrcFacilityID() + ", jobunitId = " + jobunitId + ", jobId = " + jobId);
						}
					} catch (NamingException e) {
						throw new HinemosUnknown("add file transfer job unknown error. Src FacilityId = " + file.getSrcFacilityID() + ", jobunitId = " + jobunitId + ", jobId = " + jobId, e);
					} catch (FinderException e) {
						throw new InvalidSetting("FacilityId is not exist in repository. Src FacilityId = " + file.getSrcFacilityID() + ", jobunitId = " + jobunitId + ", jobId = " + jobId);
					}
				}

				// 受信先ファシリティID(ノード/スコープ)
				if(file.getDestFacilityID() == null || "".equals(file.getDestFacilityID())){
					throw new InvalidSetting(Messages.getString("message.hinemos.3"));
				}else{
					try{
						FacilityUtil.getLocalHome().findByPrimaryKey(file.getDestFacilityID());
					} catch (NamingException e) {
						throw new HinemosUnknown("add file transfer job unknown error. Dest FacilityId = " + file.getDestFacilityID() + ", jobunitId = " + jobunitId + ", jobId = " + jobId, e);
					} catch (FinderException e) {
						throw new InvalidSetting("FacilityId is not exist in repository. Dest FacilityId = " + file.getDestFacilityID() + ", jobunitId = " + jobunitId + ", jobId = " + jobId);
					}
				}
				
				// 停止[コマンド]が選択されていないか
				if(info.getWaitRule().getEnd_delay_operation() == YesNoConstant.TYPE_YES && info.getWaitRule().getEnd_delay_operation_type() == OperationConstant.TYPE_STOP_AT_ONCE){
					throw new InvalidSetting(Messages.getString("message.job.85"));
				}
			}
		}

		//子JobTreeItemを取得
		JobTreeItem[] childrens = item.getChildrenArray();
		for(int i = 0; i < childrens.length; i++){
			validateJobUnit(childrens[i]);
		}
	}
}
