/*

 Copyright (C) 2006 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.jobmanagement.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Hashtable;
import java.util.Iterator;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.commons.util.ConnectionManager;
import com.clustercontrol.jobmanagement.ejb.entity.JobStartTimeInfoBean;
import com.clustercontrol.jobmanagement.ejb.entity.JobStartTimeInfoPK;

/**
 * ジョブ待ち条件時刻情報のDAOインターフェースを実装するクラス<BR>
 * 
 * @version 2.1.0
 * @since 2.1.0
 * 
 * @see com.clustercontrol.jobmanagement.ejb.entity.JobStartTimeInfoBean
 * @see com.clustercontrol.jobmanagement.dao.JobStartTimeInfoDAO
 */
public class JobStartTimeInfoDAOImpl implements JobStartTimeInfoDAO {

	/** ログ出力のインスタンス。 */
	private static Log m_log = LogFactory.getLog(JobStartTimeInfoDAOImpl.class);

	@Override
	public void init() {
	}

	/**
	 * SELECT文を発行します。
	 * 引数で指定されたプライマリキーで検索し、取得したデータベースの内容をEntity Beanに反映します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobStartTimeInfoDAO#load(com.clustercontrol.jobmanagement.ejb.entity.JobStartTimeInfoPK, com.clustercontrol.jobmanagement.ejb.entity.JobStartTimeInfoBean)
	 */
	@Override
	public void load(JobStartTimeInfoPK pk, JobStartTimeInfoBean ejb) throws EJBException {
		m_log.debug("load() start : " + pk.toString());

		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;

		try {
			conn = ConnectionManager.getConnectionManager().getConnection();

			//SQL文の定義
			String sql = "SELECT * FROM cc_job_start_time_info WHERE session_id = ? AND job_id = ? AND jobunit_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);

			stmt.setString(1, pk.getSession_id());
			stmt.setString(2, pk.getJob_id());
			stmt.setString(3, pk.getJobunit_id());

			res = stmt.executeQuery();

			if (res.next()) {
				//取得した値をBeanにセット
				ejb.setJob_id(res.getString("job_id"));
				ejb.setJobunit_id(res.getString("jobunit_id"));
				ejb.setSession_id(res.getString("session_id"));
				ejb.setStart_time(res.getTime("start_time"));
			} else {
				String msg = "JobStartTimeInfo data is not found.";
				m_log.error("load() error : " + pk.toString() + " SQLException " + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("load() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		} finally {
			//コネクション、結果セット、プリペアドステートメントのクローズ
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (res != null) {
					res.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("load() error : " + pk.toString() + " SQLException " + e1.getMessage());
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("load() end : " + pk.toString());
	}

	/**
	 * UPDATE文を発行します。
	 * 引数で指定されたEntity Beanの内容でデータベースを更新します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobStartTimeInfoDAO#store(com.clustercontrol.jobmanagement.ejb.entity.JobStartTimeInfoBean)
	 */
	@Override
	public void store(JobStartTimeInfoBean ejb) throws EJBException {
		m_log.debug("store() start : sessionId=" + ejb.getSession_id()
				+ ", jobunitId=" + ejb.getJobunit_id()
				+ ", jobId=" + ejb.getJob_id());

		Connection conn = null;
		PreparedStatement stmt = null;

		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("UPDATE cc_job_start_info SET ");
			sql.append("start_time = ? ");
			sql.append("WHERE session_id = ? ");
			sql.append("AND job_id = ? ");
			sql.append("AND jobunit_id = ? ");

			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());

			stmt.setTime(1, ejb.getStart_time());
			stmt.setString(2, ejb.getSession_id());
			stmt.setString(3, ejb.getJob_id());
			stmt.setString(4, ejb.getJobunit_id());

			int row = stmt.executeUpdate();

			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("store() error : " + ejb.getSession_id() + ", "
						+ ejb.getJob_id() + " SQLException" + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("store() error : sessionId=" + ejb.getSession_id()
					+ ", jobunitId=" + ejb.getJobunit_id()
					+ ", jobId=" + ejb.getJob_id() + " SQLException");
			throw new EJBException(e.getMessage());
		} finally {
			// コネクション、結果セット、プリペアドステートメントのクローズ
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("store() error : sessionId=" + ejb.getSession_id()
						+ ", jobunitId=" + ejb.getJobunit_id()
						+ ", jobId=" + ejb.getJob_id() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("store() end : sessionId=" + ejb.getSession_id()
				+ ", jobunitId=" + ejb.getJobunit_id()
				+ ", jobId=" + ejb.getJob_id());
	}

	/**
	 * DELETE文を発行します。
	 * 引数で指定されたプライマリキーでデータベースから削除します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobStartTimeInfoDAO#remove(com.clustercontrol.jobmanagement.ejb.entity.JobStartTimeInfoPK)
	 */
	@Override
	public void remove(JobStartTimeInfoPK pk) throws RemoveException, EJBException {
		m_log.debug("remove() start : " + pk.toString());

		Connection conn = null;
		PreparedStatement stmt = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();

			//SQL文の定義
			String sql = "DELETE FROM cc_job_start_time_info WHERE session_id = ? AND job_id = ? AND jobunit_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);

			stmt.setString(1, pk.getSession_id());
			stmt.setString(2, pk.getJob_id());
			stmt.setString(3, pk.getJobunit_id());

			int row = stmt.executeUpdate();

			if (row != 1 ) {
				String msg = "result row is not 1";
				m_log.error("remove() error : " + pk.toString() + " SQLException" + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("remove() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());

		} finally {
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("remove() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("remove() end : " + pk.toString());
	}

	/**
	 * INSERT文を発行します。
	 * 引数で指定されたEntity Beanの内容をデータベースに挿入し、プライマリキーを返します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobStartTimeInfoDAO#create(com.clustercontrol.jobmanagement.ejb.entity.JobStartTimeInfoBean)
	 */
	@Override
	public JobStartTimeInfoPK create(JobStartTimeInfoBean ejb) throws CreateException, EJBException {
		m_log.debug("create() start : sessionId=" + ejb.getSession_id()
				+ ", jobunitId=" + ejb.getJobunit_id()
				+ ", jobId=" + ejb.getJob_id());

		Connection conn = null;
		PreparedStatement stmt = null;
		JobStartTimeInfoPK pk;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("INSERT INTO cc_job_start_time_info (");
			sql.append("session_id, ");
			sql.append("job_id, ");
			sql.append("start_time, ");
			sql.append("jobunit_id) ");
			sql.append("VALUES (?,?,?,?)");

			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());

			stmt.setString(1, ejb.getSession_id());
			stmt.setString(2, ejb.getJob_id());
			stmt.setTime(3, ejb.getStart_time());
			stmt.setString(4, ejb.getJobunit_id());

			int row =	stmt.executeUpdate();
			pk = new JobStartTimeInfoPK(
					ejb.getSession_id(),
					ejb.getJobunit_id(),
					ejb.getJob_id());

			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("create() error : sessionId=" + ejb.getSession_id()
						+ ", jobunitId=" + ejb.getJobunit_id()
						+ ", jobId=" + ejb.getJob_id() + " SQLException");
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			if(e.getSQLState().equals("23505")){
				m_log.error("create() error : sessionId=" + ejb.getSession_id()
						+ ", jobunitId=" + ejb.getJobunit_id()
						+ ", jobId=" + ejb.getJob_id() + " SQLException" + "DuplicateKeyException ");
				throw new DuplicateKeyException(e.getMessage());
			}
			throw new CreateException(e.getMessage());
		}finally{
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("create() error : sessionId=" + ejb.getSession_id()
						+ ", jobunitId=" + ejb.getJobunit_id()
						+ ", jobId=" + ejb.getJob_id() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("create() end : sessionId=" + ejb.getSession_id()
				+ ", jobunitId=" + ejb.getJobunit_id()
				+ ", jobId=" + ejb.getJob_id());
		return pk;
	}

	/**
	 * 全件取得するSELECT文を発行します。
	 * 取得したデータのプライマリキーをコレクションに格納し返します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobStartTimeInfoBean#ejbFindAll()
	 * @see com.clustercontrol.jobmanagement.dao.JobStartTimeInfoDAO#findAll()
	 */
	@Override
	public Collection findAll() throws FinderException {
		m_log.debug("findAll() start : ");

		ArrayList<JobStartTimeInfoPK> ret = new ArrayList<JobStartTimeInfoPK>();

		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			String sql = "SELECT session_id, jobunit_id, job_id FROM cc_job_start_time_info";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			res = stmt.executeQuery();

			while(res.next()) {
				JobStartTimeInfoPK pk= new JobStartTimeInfoPK(
						res.getString("session_id"),
						res.getString("jobunit_id"),
						res.getString("job_id"));
				ret.add(pk);
			}

		} catch (SQLException e) {
			m_log.error("findAll() error :  SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findAll() error :  SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findAll() end : ");
		return ret;
	}

	/**
	 * 1件取得するSELECT文を発行します。
	 * 引数で指定されたプライマリキーで、データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobStartTimeInfoBean#ejbFindByPrimaryKey(com.clustercontrol.jobmanagement.ejb.entity.JobStartTimeInfoPK)
	 * @see com.clustercontrol.jobmanagement.dao.JobStartTimeInfoDAO#findByPrimaryKey(com.clustercontrol.jobmanagement.ejb.entity.JobStartTimeInfoPK)
	 */
	@Override
	public JobStartTimeInfoPK findByPrimaryKey(JobStartTimeInfoPK pk) throws FinderException {
		m_log.debug("findByPrimaryKey() start : " + pk.toString());

		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();

			//SQL文の定義
			String sql = "SELECT session_id, jobunit_id, job_id FROM cc_job_start_time_info WHERE session_id = ? AND job_id = ? AND jobunit_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);

			stmt.setString(1, pk.getSession_id());
			stmt.setString(2, pk.getJob_id());
			stmt.setString(3, pk.getJobunit_id());

			res = stmt.executeQuery();

			if (res.next()) {
				return pk;
			} else {
				String msg = "id " + pk.toString() + " not found.";
				throw new FinderException(msg);
			}
		} catch (SQLException e) {
			m_log.error("find() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("find() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
			m_log.debug("findByPrimaryKey() end : " + pk.toString());
		}
	}

	/**
	 * ジョブ待ち条件時刻情報をジョブ待ち条件時刻マスタから作成します。
	 * 
	 * @param sessionId セッションID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobs ジョブIDのコレクション
	 * @throws CreateException
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobStartTimeInfoBean#ejbHomeMakeSession(String, Collection)
	 */
	public void makeSession(String sessionId, String jobunitId, Collection jobs) throws CreateException{
		m_log.debug("makeSession() start : sessionId=" + sessionId + ", jobunitId=" + jobunitId);

		Connection conn=null;
		Connection connMaster = null;
		PreparedStatement stmt=null;
		PreparedStatement stmtMaster= null;
		String jobId = null;

		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			connMaster = ConnectionManager.getConnectionManager().getConnection();

			//SQL文の定義(INSERT)
			StringBuilder sql = new StringBuilder();
			sql.append("INSERT INTO cc_job_start_time_info (");
			sql.append("session_id, ");
			sql.append("job_id, ");
			sql.append("start_time, ");
			sql.append("jobunit_id) ");
			sql.append("VALUES (?,?,?,?)");

			//SQL文のセット(INSERT)
			stmt = conn.prepareStatement(sql.toString());
			stmt.setString(1,sessionId);
			stmt.setString(4, jobunitId);

			//SQL文の定義(マスタ検索)
			String sql_mst = "SELECT * FROM cc_job_start_time_mst WHERE job_id = ? AND jobunit_id = ?";

			//SQL文のセット(マスタ検索)
			stmtMaster = connMaster.prepareStatement(sql_mst);

			Iterator it = jobs.iterator();
			while(it.hasNext()){
				//ジョブID取得
				Hashtable ht = (Hashtable)it.next();
				jobId = (String)ht.get("jobId");

				//SQL文のセット(マスタ検索)
				stmtMaster.setString(1, jobId);
				stmtMaster.setString(2, jobunitId);

				//SQL実行(マスタ検索)
				ResultSet resMaster = stmtMaster.executeQuery();

				if (resMaster.next()) {
					//SQL文のセット(INSERT)
					stmt.setString(2, resMaster.getString("job_id"));
					stmt.setTime(3, resMaster.getTime("start_time"));
					//INSERTを実行
					int row =	stmt.executeUpdate();
					if (row != 1) {
						String msg = "result row is not 1";
						m_log.error("makeSession() error : sessionId=" + sessionId
								+ ", jobunitId=" + jobunitId
								+ ", jobId" + resMaster.getString("job_id")
								+ " SQLException");
						throw new EJBException(msg);
					}
				}
				resMaster.close();
			}
		} catch (SQLException e) {
			if(e.getSQLState().equals("23505")){
				m_log.error("makeSession() error : sessionId=" + sessionId
						+ ", jobunitId=" + jobunitId
						+ ", jobId" + jobId+ " SQLException" + "DuplicateKeyException ");
				throw new DuplicateKeyException(e.getMessage());
			}
			throw new CreateException(e.getMessage());
		} finally {
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
				if(stmtMaster != null){
					stmtMaster.close();
				}
				if(connMaster != null){
					connMaster.close();
				}
			} catch (SQLException e1) {
				m_log.error("makeSession() error : sessionId=" + sessionId
						+ ", jobunitId=" + jobunitId
						+ ", jobId" + jobId + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("makeSession() end : sessionId=" + sessionId
				+ ", jobunitId=" + jobunitId);
	}
}
