/*

Copyright (C) 2011 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */


package com.clustercontrol.calendar.util;

import java.sql.Time;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.calendar.bean.CalendarAddInfo;
import com.clustercontrol.calendar.bean.CalendarInfo;
import com.clustercontrol.calendar.bean.WeekdayInfo;
import com.clustercontrol.commons.util.CommonValidator;
import com.clustercontrol.util.Messages;

/**
 * カレンダの入力チェッククラス
 * 
 * @since 4.0
 */
public class CalendarValidator {

	private static Log m_log = LogFactory.getLog(CalendarValidator.class);
	/**
	 * カレンダ情報(CalendarInfo)の基本設定の妥当性チェック
	 * 
	 * @param calendarInfo
	 * @throws InvalidSetting
	 */
	public static void validateCalendarInfo(CalendarInfo calendarInfo) throws InvalidSetting{
		// calendarId
		CommonValidator.validateId(Messages.getString("calendar.id"), calendarInfo.getId(), 64);

		// calendarName
		CommonValidator.validateString(Messages.getString("calendar.name"), calendarInfo.getName(), true, 1, 256);

		if (calendarInfo.getValidTimeFrom() == null || calendarInfo.getValidTimeFrom() == 0) {
			String[] args = { "(" + Messages.getString("start") + ")" };
			throw new InvalidSetting(Messages.getString("message.calendar.24", args));
		}
		if (calendarInfo.getValidTimeTo() == null || calendarInfo.getValidTimeTo() == 0) {
			String[] args = { "(" + Messages.getString("end") + ")" };
			throw new InvalidSetting(Messages.getString("message.calendar.24", args));
		}

		if (calendarInfo.getValidTimeFrom() != null && calendarInfo.getValidTimeTo() != null) {
			if (calendarInfo.getValidTimeFrom() >= calendarInfo.getValidTimeTo()) {
				String[] args = { Messages.getString("time") + "(" + Messages.getString("end") + ")",
						Messages.getString("time") + "(" + Messages.getString("start") + ")" };
				throw new InvalidSetting(Messages.getString("message.calendar.30", args));
			}
		}
	}

	/**
	 * カレンダ曜日別情報(WeekdayInfo)の基本設定の妥当性チェック
	 * 
	 * @param calendarId
	 * @param weekdayInfo
	 * @throws HinemosUnknown
	 * @throws InvalidSetting
	 */
	public static void validateWeekdayInfo(String calendarId, WeekdayInfo weekdayInfo) throws HinemosUnknown, InvalidSetting{
		// calendarId
		CommonValidator.validateCalenderId(calendarId, true);

		Long from = weekdayInfo.getTimeFrom();
		Long to = weekdayInfo.getTimeTo();
		if (from == null) {
			String[] args = { "(" + Messages.getString("start") + ")" };
			throw new InvalidSetting(Messages.getString("message.calendar.25", args));
		}
		if (to == null){
			String[] args = { "(" + Messages.getString("end") + ")" };
			throw new InvalidSetting(Messages.getString("message.calendar.25", args));
		}

		Time tFrom = new Time(from);
		Time tTo = new Time(to);
		m_log.debug("validateWeekdayInfo : " + tFrom + "(" + from + "), " + tTo + "(" + to + ")");

		// from > toだけでは網羅できないので、追加する。
		// from 4:00, to 25:00と登録すると、登録に成功するが、
		// 内部では25:00は1:00に変換されて、from 4:00, to 1:00というおかしな状態になってしまう。
		// それをカバーするために、追加の条件を入れている。
		if (from >= to ||
				tFrom.getHours() * 24 * 60 + tFrom.getMinutes() * 60 + tFrom.getSeconds() >=
					tTo.getHours() * 24 * 60 + tTo.getMinutes() * 60 + tTo.getSeconds()) {

			String[] args = { Messages.getString("time") + "(" + Messages.getString("end") + ")",
					Messages.getString("time") + "(" + Messages.getString("start") + ")" };
			throw new InvalidSetting(Messages.getString("message.calendar.31", args));
		}
	}

	/**
	 * カレンダ追加情報(CalendarAddInfo)の基本設定の妥当性チェック
	 * 
	 * @param calendarId
	 * @param calendarAddInfo
	 * @throws HinemosUnknown
	 * @throws InvalidSetting
	 */
	public static void validateCalendarAddInfo(String calendarId, CalendarAddInfo calendarAddInfo) throws HinemosUnknown, InvalidSetting{
		// calendarId
		CommonValidator.validateCalenderId(calendarId, true);

		if (calendarAddInfo.getTimeFrom() == null || calendarAddInfo.getTimeFrom() == 0) {
			String[] args = { "(" + Messages.getString("start") + ")" };
			throw new InvalidSetting(Messages.getString("message.calendar.24", args));
		}
		if (calendarAddInfo.getTimeTo() == null || calendarAddInfo.getTimeTo() == 0) {
			String[] args = { "(" + Messages.getString("end") + ")" };
			throw new InvalidSetting(Messages.getString("message.calendar.24", args));
		}

		if (calendarAddInfo.getTimeFrom() != null && calendarAddInfo.getTimeTo() != null) {
			if (calendarAddInfo.getTimeFrom() >= calendarAddInfo.getTimeTo()) {
				String[] args = { Messages.getString("time") + "(" + Messages.getString("end") + ")",
						Messages.getString("time") + "(" + Messages.getString("start") + ")" };
				throw new InvalidSetting(Messages.getString("message.calendar.30", args));
			}
		}
	}

}
