/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.calendar.factory;

import java.sql.Time;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Date;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.CalendarDuplicate;
import com.clustercontrol.fault.CalendarNotFound;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.calendar.bean.CalendarAddInfo;
import com.clustercontrol.calendar.bean.CalendarInfo;
import com.clustercontrol.calendar.bean.WeekdayInfo;
import com.clustercontrol.calendar.ejb.entity.CalendarAddInfoLocal;
import com.clustercontrol.calendar.ejb.entity.CalendarAddInfoPK;
import com.clustercontrol.calendar.ejb.entity.CalendarAddInfoUtil;
import com.clustercontrol.calendar.ejb.entity.CalendarInfoLocal;
import com.clustercontrol.calendar.ejb.entity.CalendarInfoUtil;
import com.clustercontrol.calendar.ejb.entity.CalendarWeekdayInfoLocal;
import com.clustercontrol.calendar.ejb.entity.CalendarWeekdayInfoPK;
import com.clustercontrol.calendar.ejb.entity.CalendarWeekdayInfoUtil;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * カレンダ更新を行うファクトリークラス<BR>
 *
 * @version 2.0.0
 * @since 2.0.0
 */
public class ModifyCalendar {
	private static Log m_log = LogFactory.getLog(ModifyCalendar.class);
	
	private static final long END_OF_DAY = 53999000;
	private static final long ADDITION_TIME = 999;

	/**
	 * カレンダ追加
	 * 
	 * @param info
	 * @param userName
	 * @return
	 * @throws CreateException
	 * @throws NamingException
	 * @throws CalendarDuplicate
	 * @throws InvalidSetting
	 */
	public void addCalendar(CalendarInfo info, String userName) throws CreateException, NamingException, CalendarDuplicate {
		String id = null;
		try {
			//現在日時を取得
			Date now = new Date();

			//ID取得
			id = info.getId();
			//名前を取得
			String name = info.getName();
			//説明を取得
			String description = info.getDescription();
			//有効期間(From)を取得
			Date validTimeFrom = null;
			if (info.getValidTimeFrom() != null) {
				validTimeFrom = new Date(info.getValidTimeFrom());
			}
			//有効期間(To)を取得
			Date validTimeTo = null;
			if (info.getValidTimeTo() != null) {
				validTimeTo = new Date(info.getValidTimeTo());
			}

			//FROMとTOをチェック
			if(validTimeFrom.after(validTimeTo) || validTimeFrom.equals(validTimeTo)){
				throw new java.lang.IllegalArgumentException();
			}

			//カレンダを作成
			CalendarInfoUtil.getLocalHome().create(
					id,
					name,
					description,
					null,
					validTimeFrom,
					validTimeTo,
					now,
					now,
					userName,
					userName);

		} catch (CreateException e) {
			if (e instanceof DuplicateKeyException) {
				AplLogger apllog = new AplLogger("CAL", "cal");
				String[] args = {id};
				apllog.put("SYS", "001", args);
				m_log.debug("addCalendar() : " + e.getMessage());
				throw new CalendarDuplicate(e.getMessage(), e);
			}
			else{
				throw e;
			}
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("CAL", "cal");
			String[] args = {id};
			apllog.put("SYS", "001", args);

			m_log.debug("addCalendar() : " + e.getMessage());
			throw e;
		}
	}

	/**
	 * カレンダ（基本）情報を変更します。
	 * 
	 * @param info
	 * @param userName
	 * @return
	 * @throws NamingException
	 * @throws CalendarNotFound
	 * @throws InvalidSetting
	 */
	public void modifyCalendar(CalendarInfo info, String userName) throws NamingException, CalendarNotFound {
		String id = null;
		try {
			Date now = new Date();

			//UID取得
			id = info.getId();

			//calを検索し取得
			CalendarInfoLocal cal = CalendarInfoUtil.getLocalHome().findByPrimaryKey(id);
			//名前を取得し、calに設定
			cal.setCalendar_name(info.getName());
			//説明を取得し、calに設定
			cal.setDescription(info.getDescription());
			//有効期間(From)を取得、calに設定
			if (info.getValidTimeFrom() != null) {
				cal.setValid_time_from(new Date(info.getValidTimeFrom()));
			}
			//有効期間(To)を取得、calに設定
			if (info.getValidTimeTo() != null) {
				cal.setValid_time_to(new Date(info.getValidTimeTo()));
			}
			//更新者を取得し、calに設定
			cal.setUpdate_user(userName);
			//更新日時を取得し、calに設定
			cal.setUpdate_date(now);
		}catch (FinderException e) {
			AplLogger apllog = new AplLogger("CAL", "cal");
			String[] args = {id};
			apllog.put("SYS", "002", args);

			m_log.debug("modifyCalendar() : " + e.getMessage());
			throw new CalendarNotFound("modifyCalendar() find error : calendarId=" + id);
		} catch (EJBException e) {
			AplLogger apllog = new AplLogger("CAL", "cal");
			String[] args = {id};
			apllog.put("SYS", "002", args);

			m_log.debug("modifyCalendar() : " + e.getMessage());
			throw e;
		}
	}

	/**
	 * カレンダ（基本）情報を削除します。<BR>
	 * 
	 * @param id
	 * @return
	 * @throws NamingException
	 * @throws CalendarNotFound
	 */
	public void deleteCalendar(String id) throws NamingException, RemoveException, CalendarNotFound {
		try {
			//カレンダ情報を検索し取得
			CalendarInfoLocal cal = CalendarInfoUtil.getLocalHome().findByPrimaryKey(id);

			//カレンダ情報を削除
			cal.remove();

		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("CAL", "cal");
			String[] args = {id};
			apllog.put("SYS", "003", args);

			m_log.debug("deleteCalendar() : " + e.getMessage());
			throw new CalendarNotFound("deleteCalendar() find error : calendarId=" + id);
		} catch (RemoveException e) {
			AplLogger apllog = new AplLogger("CAL", "cal");
			String[] args = {id};
			apllog.put("SYS", "003", args);

			m_log.debug("deleteCalendar() : " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("CAL", "cal");
			String[] args = {id};
			apllog.put("SYS", "003", args);

			m_log.debug("deleteCalendar() : " + e.getMessage());
			throw e;
		}
	}

	/**
	 * カレンダ曜日別情報を登録します。<BR>
	 * 
	 * @param id
	 * @param info
	 * @param userName
	 * @return
	 * @throws NamingException
	 * @throws CreateException
	 * @throws CalendarDuplicate
	 * @throws CalendarNotFound
	 * @throws InvalidSetting
	 * @throws HinemosUnknown
	 */
	public void addWeekday(String id, WeekdayInfo info, String userName) throws NamingException, CreateException, CalendarDuplicate, CalendarNotFound, HinemosUnknown {
		Integer weekday = null;
		Time from = null;
		Time to = null;
		//現在日時を取得
		Date now = new Date();

		try {
			if(END_OF_DAY == info.getTimeTo()){
				info.setTimeTo(info.getTimeTo() + ADDITION_TIME);
			}
			
			//曜日取得
			weekday = info.getWeekday();
			//説明を取得
			String description = info.getDescription();
			//時間帯(From)を取得
			from = new Time(info.getTimeFrom());
			//時間帯(To)を取得
			to = new Time(info.getTimeTo());

			//FROMとTOをチェック
			if(from.after(to) || from.equals(to)){
				throw new java.lang.IllegalArgumentException();
			}

			//時間帯の重複をチェック
			Collection<CalendarWeekdayInfoLocal> ct =
				CalendarWeekdayInfoUtil.getLocalHome().findByTimeFromTo(
						id, weekday, from, to);
			if(ct.size() > 0){
				throw new DuplicateKeyException();
			}

			//カレンダ曜日別情報を作成
			CalendarWeekdayInfoUtil.getLocalHome().create(
					id,
					Integer.valueOf(weekday),
					from,
					to,
					description,
					now,
					now,
					userName,
					userName);

		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("CAL", "cal");
			SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, String.valueOf(weekday), formatter.format(from), formatter.format(to)};
			apllog.put("SYS", "004", args);

			m_log.debug("addWeekday() : " + e.getMessage());
			throw e;
		} catch (CreateException e) {
			if (e instanceof DuplicateKeyException) {
				AplLogger apllog = new AplLogger("CAL", "cal");
				SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
				String[] args = {id, String.valueOf(weekday), formatter.format(from), formatter.format(to)};
				apllog.put("SYS", "004", args);

				m_log.debug("addWeekday() : " + e.getMessage());
				throw new CalendarDuplicate(e.getMessage(), e);
			}
			else{
				throw e;
			}

		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("CAL", "cal");
			SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, String.valueOf(weekday), formatter.format(from), formatter.format(to)};
			apllog.put("SYS", "004", args);

			m_log.debug("addWeekday() : " + e.getMessage());
			throw new CalendarNotFound("addWeekday() find error : calendarId=" + id);
		}
	}

	/**
	 * カレンダ曜日別情報を変更します。<BR>
	 * 
	 * @param id
	 * @param info
	 * @param userName
	 * @return
	 * @throws NamingException
	 * @throws RemoveException
	 * @throws CalendarDuplicate
	 * @throws CalendarNotFound
	 * @throws InvalidSetting
	 * @throws HinemosUnknown
	 */
	public void modifyWeekday(String id, Integer weekday, Long from, Long to, WeekdayInfo info, String userName) throws NamingException, RemoveException, CreateException, CalendarDuplicate, CalendarNotFound, HinemosUnknown {
		try {
			//現在日時を取得
			Date now = new Date();

			//カレンダ曜日別情報を削除
			CalendarWeekdayInfoLocal infoLocal =
				CalendarWeekdayInfoUtil.getLocalHome().findByPrimaryKey(
						new CalendarWeekdayInfoPK(id, Integer.valueOf(weekday), new Time(from), new Time(to)));
			Date createTimeStamp = infoLocal.getReg_date();
			String createUser = infoLocal.getReg_user();
			infoLocal.remove();

			//曜日取得
			Integer newWeekday = info.getWeekday();
			//説明を取得
			String newDescription = info.getDescription();
			//時間帯(From)を取得
			Time newFrom = null;
			if (info.getTimeFrom() != null) {
				newFrom = new Time(info.getTimeFrom());
			}
			//時間帯(To)を取得
			Time newTo = null;
			if (info.getTimeTo() != null) {
				if(END_OF_DAY == info.getTimeTo()){
					info.setTimeTo(info.getTimeTo() + ADDITION_TIME);
				}
				newTo = new Time(info.getTimeTo());
			}

			//時間帯の重複をチェック
			Collection<CalendarWeekdayInfoLocal> ct =
				CalendarWeekdayInfoUtil.getLocalHome().findByTimeFromTo(
						id, newWeekday, newFrom, newTo);
			if(ct.size() > 0){
				throw new DuplicateKeyException();
			}

			//カレンダ曜日別情報を作成
			CalendarWeekdayInfoUtil.getLocalHome().create(
					id,
					Integer.valueOf(newWeekday),
					newFrom==null?null:new Time(newFrom.getTime()),
							newTo==null?null:new Time(newTo.getTime()),
									newDescription,
									createTimeStamp,
									now,
									createUser,
									userName);

		} catch (DuplicateKeyException e) {
			AplLogger apllog = new AplLogger("CAL", "cal");
			SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, weekday.toString(), formatter.format(from), formatter.format(to)};
			apllog.put("SYS", "006", args);

			m_log.debug("modifyWeekday() : " + e.getMessage());
			throw new CalendarDuplicate(e.getMessage(), e);
		} catch (RemoveException e) {
			AplLogger apllog = new AplLogger("CAL", "cal");
			SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, weekday.toString(), formatter.format(from), formatter.format(to)};
			apllog.put("SYS", "006", args);

			m_log.debug("modifyWeekday() : " + e.getMessage());
			throw e;
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("CAL", "cal");
			SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, weekday.toString(), formatter.format(from), formatter.format(to)};
			apllog.put("SYS", "006", args);

			m_log.debug("modifyWeekday() : " + e.getMessage());
			throw new CalendarNotFound("modifyWeekday() find error : calendarId=" + id);
		} catch (CreateException e) {
			AplLogger apllog = new AplLogger("CAL", "cal");
			SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, weekday.toString(), formatter.format(from), formatter.format(to)};
			apllog.put("SYS", "006", args);

			m_log.debug("modifyWeekday() : " + e.getMessage());
			throw e;
		}
	}

	/**
	 * カレンダ曜日別情報を削除します。<BR>
	 * 
	 * @param id
	 * @param weekday
	 * @param from
	 * @param to
	 * @return
	 * @throws NamingException
	 * @throws NamingException
	 * @throws CalendarNotFound
	 */
	public void deleteWeekday(String id, Integer weekday, Long from, Long to) throws NamingException, RemoveException, CalendarNotFound {
		try {
			//カレンダ曜日別情報を削除
			CalendarWeekdayInfoLocal info =
				CalendarWeekdayInfoUtil.getLocalHome().findByPrimaryKey(
						new CalendarWeekdayInfoPK(id, weekday, new Time(from), new Time(to)));
			info.remove();

		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("CAL", "cal");
			SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, String.valueOf(weekday), formatter.format(from), formatter.format(to)};
			apllog.put("SYS", "005", args);

			m_log.debug("deleteWeekday() : " + e.getMessage());
			throw new CalendarNotFound("deleteWeekday() find error : calendarId=" + id);
		} catch (RemoveException e) {
			AplLogger apllog = new AplLogger("CAL", "cal");
			SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, String.valueOf(weekday), formatter.format(from), formatter.format(to)};
			apllog.put("SYS", "005", args);

			m_log.debug("deleteWeekday() : " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("CAL", "cal");
			SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, String.valueOf(weekday), formatter.format(from), formatter.format(to)};
			apllog.put("SYS", "005", args);

			m_log.debug("deleteWeekday() : " + e.getMessage());
			throw e;
		}
	}

	/**
	 * カレンダ追加情報(例外実行情報)を追加します。<BR>
	 * 
	 * @param id
	 * @param info
	 * @param userName
	 * @return
	 * @throws NamingException
	 * @throws CreateException
	 * @throws CalendarDuplicate
	 * @throws CalendarNotFound
	 * @throws InvalidSetting
	 * @throws HinemosUnknown
	 */
	public void addCalendarAdd(String id, CalendarAddInfo info, String userName) throws NamingException, CreateException, CalendarDuplicate, CalendarNotFound, HinemosUnknown {
		Date from = null;
		Date to = null;
		try {
			//現在日時を取得
			Date now = new Date();

			//説明を取得
			String description = info.getDescription();
			//日時(From)を取得
			if (info.getTimeFrom() != null) {
				from = new Date(info.getTimeFrom());
			}
			//日時(To)を取得
			if (info.getTimeTo() != null) {
				to = new Date(info.getTimeTo());
			}
			//実行取得
			int execute = ValidConstant.booleanToType(info.isExecuteFlag());

			//FROMとTOをチェック
			if(from.after(to) || from.equals(to)){
				throw new java.lang.IllegalArgumentException();
			}

			//時間帯の重複をチェック
			Collection<CalendarAddInfoLocal> ct =
				CalendarAddInfoUtil.getLocalHome().findByTimeFromTo(
						id, from, to);
			if(ct.size() > 0){
				throw new DuplicateKeyException();
			}

			//カレンダ追加情報を作成
			CalendarAddInfoUtil.getLocalHome().create(
					id,
					from,
					to,
					Integer.valueOf(execute),
					description,
					now,
					now,
					userName,
					userName);

		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("CAL", "cal");
			SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, formatter.format(from), formatter.format(to)};
			apllog.put("SYS", "007", args);

			m_log.debug("addAddInfo() : " + e.getMessage());
			throw e;
		} catch (CreateException e) {
			if (e instanceof DuplicateKeyException) {
				AplLogger apllog = new AplLogger("CAL", "cal");
				SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
				String[] args = {id, formatter.format(from), formatter.format(to)};
				apllog.put("SYS", "007", args);

				m_log.debug("addAddInfo() : " + e.getMessage());
				throw new CalendarDuplicate(e.getMessage(), e);
			}
			else{
				throw e;
			}
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("CAL", "cal");
			SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, formatter.format(from), formatter.format(to)};
			apllog.put("SYS", "007", args);

			m_log.debug("addAddInfo() : " + e.getMessage());
			throw new CalendarNotFound("addCalendarAdd() find error : calendarId=" + id);
		}
	}

	/**
	 * カレンダ追加情報(例外実行情報)を変更します。<BR>
	 * 
	 * @param id
	 * @param from
	 * @param to
	 * @param info
	 * @param userName
	 * @return
	 * @throws NamingException
	 * @throws CreateException
	 * @throws CalendarNotFound
	 * @throws CalendarDuplicate
	 * @throws InvalidSetting
	 * @throws HinemosUnknown
	 */
	public void modifyCalendarAdd(String id, Long from, Long to, CalendarAddInfo info, String userName) throws RemoveException, CreateException, CalendarNotFound, NamingException, CalendarDuplicate, HinemosUnknown {
		try {
			//現在日時を取得
			Date now = new Date();

			//カレンダ追加情報を削除
			CalendarAddInfoLocal infoLocal =
				CalendarAddInfoUtil.getLocalHome().findByPrimaryKey(
						new CalendarAddInfoPK(
								id,
								from==null?null:new Date(from),
										to==null?null:new Date(to)));
			Date createTimeStamp = infoLocal.getReg_date();
			String createUser = infoLocal.getReg_user();
			infoLocal.remove();

			//説明を取得
			String newDescription = info.getDescription();
			//日時(From)を取得
			Date newFrom = null;
			if (info.getTimeFrom() != null) {
				newFrom = new Date(info.getTimeFrom());
			}
			//日時(To)を取得
			Date newTo = null;
			if (info.getTimeTo() != null) {
				newTo = new Date(info.getTimeTo());
			}
			//実行取得
			int newExecute = ValidConstant.booleanToType(info.isExecuteFlag());

			//時間帯の重複をチェック
			Collection<CalendarAddInfoLocal> ct =
				CalendarAddInfoUtil.getLocalHome().findByTimeFromTo(
						id, newFrom, newTo);
			if(ct.size() > 0){
				throw new DuplicateKeyException();
			}

			//カレンダ追加情報を作成
			CalendarAddInfoUtil.getLocalHome().create(
					id,
					newFrom,
					newTo,
					Integer.valueOf(newExecute),
					newDescription,
					createTimeStamp,
					now,
					createUser,
					userName);
		} catch (DuplicateKeyException e) {
			AplLogger apllog = new AplLogger("CAL", "cal");
			SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, formatter.format(from), formatter.format(to)};
			apllog.put("SYS", "009", args);

			m_log.debug("modifyAddInfo() : " + e.getMessage());
			throw new CalendarDuplicate(e.getMessage(), e);
		} catch (RemoveException e) {
			AplLogger apllog = new AplLogger("CAL", "cal");
			SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, formatter.format(from), formatter.format(to)};
			apllog.put("SYS", "009", args);

			m_log.debug("modifyAddInfo() : " + e.getMessage());
			throw e;
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("CAL", "cal");
			SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, formatter.format(from), formatter.format(to)};
			apllog.put("SYS", "009", args);

			m_log.debug("modifyAddInfo() : " + e.getMessage());
			throw new CalendarNotFound("modifyCalendarAdd() find error : calendarId=" + id);
		} catch (CreateException e) {
			AplLogger apllog = new AplLogger("CAL", "cal");
			SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, formatter.format(from), formatter.format(to)};
			apllog.put("SYS", "009", args);

			m_log.debug("modifyAddInfo() : " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("CAL", "cal");
			SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, formatter.format(from), formatter.format(to)};
			apllog.put("SYS", "009", args);

			m_log.debug("modifyAddInfo() : " + e.getMessage());
			throw e;
		}
	}


	/**
	 * カレンダ追加情報(例外実行情報)を削除します。<BR>
	 * 
	 * @param id
	 * @param from
	 * @param to
	 * @return
	 * @throws NamingException
	 * @throws CalendarNotFound
	 */
	public void deleteCalendarAdd(String id, Long from, Long to) throws NamingException, RemoveException, CalendarNotFound {
		try {
			//カレンダ追加情報を削除
			CalendarAddInfoLocal info =
				CalendarAddInfoUtil.getLocalHome().findByPrimaryKey(
						new CalendarAddInfoPK(
								id,
								from==null?null:new Date(from),
										to==null?null:new Date(to)));
			info.remove();

		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("CAL", "cal");
			SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, formatter.format(from), formatter.format(to)};
			apllog.put("SYS", "008", args);

			m_log.debug("deleteAddInfo() : " + e.getMessage());
			throw new CalendarNotFound("deleteCalendarAdd() find error : calendarId=" + id);
		} catch (RemoveException e) {
			AplLogger apllog = new AplLogger("CAL", "cal");
			SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, formatter.format(from), formatter.format(to)};
			apllog.put("SYS", "008", args);

			m_log.debug("deleteAddInfo() : " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("CAL", "cal");
			SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
			String[] args = {id, formatter.format(from), formatter.format(to)};
			apllog.put("SYS", "008", args);

			m_log.debug("deleteAddInfo() : " + e.getMessage());
			throw e;
		}
	}
}
