/*

Copyright (C) 2011 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.calendar.bean;

import java.util.Calendar;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.CalendarInvalid;

/**
 * カレンダの実行規則を格納するDTOクラス
 * 
 * @since 4.0
 * @author Tomoya Takahata
 */
public class CalendarWeekdayDTO {

	private static Log log = LogFactory.getLog(CalendarWeekdayDTO.class);

	private Integer weekday = new Integer(0);
	private Long beginTime;
	private Long endTime;
	private String description;

	private Long registerDate;
	private String registerUser;
	private Long updateDate;
	private String updateUser;

	/**
	 * コンストラクタ(no-argument for JAXB)
	 */
	public CalendarWeekdayDTO() {

	}

	/**
	 * コンストラクタ
	 * @param weekday 曜日
	 * @param beginTime 開始時刻
	 * @param endTime 終了時刻
	 * @param description 実行規則の説明
	 * @param registerDate 登録日時
	 * @param registerUser 登録ユーザ名
	 * @param updateDate 更新日時
	 * @param updateUser 更新ユーザ名
	 * @throws CalendarInvalid 妥当でない値が存在した場合
	 */
	public CalendarWeekdayDTO(Integer weekday, Long beginTime, Long endTime, String description,
			Long registerDate, String registerUser, Long updateDate, String updateUser) throws CalendarInvalid {
		this.weekday = weekday;
		this.beginTime = beginTime;
		this.endTime = endTime;
		this.description = description;
		this.registerDate = registerDate;
		this.registerUser = registerUser;
		this.updateDate = updateDate;
		this.updateUser = updateUser;

		validate();
	}

	/**
	 * 曜日を格納する。<br/>
	 * java.util.Calendarが提供する曜日値(Calendar.SUNDAY, Calendar.MONDAY...)を指定する。<br/>
	 * @param weekday 曜日
	 */
	public void setWeekday(Integer weekday) {
		this.weekday = weekday;
	}

	/**
	 * 曜日を返す。<br/>
	 * java.util.Calendarが提供する曜日値(Calendar.SUNDAY, Calendar.MONDAY...)で返される。<br/>
	 * @return 曜日
	 */
	public Integer getWeekday() {
		return weekday;
	}

	/**
	 * 開始時刻を格納する。<br/>
	 * 1970-1-1 xx:xx:xxにて時刻を指定すること。<br/>
	 * @param beginTime 開始時刻
	 */
	public void setBeginTime(Long beginTime) {
		this.beginTime = beginTime;
	}

	/**
	 * 開始時刻を返す。<br/>
	 * @return 開始時刻
	 */
	public Long getBeginTime() {
		return beginTime;
	}

	/**
	 * 終了時刻を格納する。<br/>
	 * 1970-1-1 xx:xx:xxにて時刻を指定すること。<br/>
	 * @param endTime 終了時刻
	 */
	public void setEndTime(Long endTime) {
		this.endTime = endTime;
	}

	/**
	 * 終了時刻を返す。<br/>
	 * @return 終了時刻
	 */
	public Long getEndTime() {
		return endTime;
	}

	/**
	 * 実行規則の説明を格納する。<br/>
	 * @param description 実行規則の説明
	 */
	public void setDescription(String description) {
		this.description = description;
	}

	/**
	 * 実行規則の説明を返す。<br/>
	 * @return 実行規則の説明
	 */
	public String getDescription() {
		return description;
	}

	/**
	 * 登録日時を格納する。<br/>
	 * @param registerDate 登録日時
	 */
	public void setRegisterDate(Long registerDate) {
		this.registerDate = registerDate;
	}

	/**
	 * 登録日時を返す。<br/>
	 * @return 登録日時
	 */
	public Long getRegisterDate() {
		return registerDate;
	}

	/**
	 * 登録ユーザ名を格納する。<br/>
	 * @param registerUser 登録ユーザ名
	 */
	public void setRegisterUser(String registerUser) {
		this.registerUser = registerUser;
	}

	/**
	 * 登録ユーザ名を返す。<br/>
	 * @return 登録ユーザ名
	 */
	public String getRegisterUser() {
		return registerUser;
	}

	/**
	 * 更新日時を格納する。<br/>
	 * @param updateDate 更新日時
	 */
	public void setUpdateDate(Long updateDate) {
		this.updateDate = updateDate;
	}

	/**
	 * 更新日時を返す。<br/>
	 * @return 更新日時
	 */
	public Long getUpdateDate() {
		return updateDate;
	}

	/**
	 * 更新ユーザ名を格納する。<br/>
	 * @param updateUser 更新ユーザ名
	 */
	public void setUpdateUser(String updateUser) {
		this.updateUser = updateUser;
	}

	/**
	 * 更新ユーザ名を返す。<br/>
	 * @return 更新ユーザ名
	 */
	public String getUpdateUser() {
		return updateUser;
	}

	/**
	 * コピーされた実行規則インスタンスを返す。<br/>
	 */
	@Override
	public CalendarWeekdayDTO clone() {
		CalendarWeekdayDTO dto = null;
		try {
			dto = new CalendarWeekdayDTO(getWeekday(), getBeginTime(), getEndTime(), getDescription(),
					getRegisterDate(), getRegisterUser(), getUpdateDate(), getUpdateUser());
		} catch (CalendarInvalid e) {
			log.warn("cloning CalendarWeekdayDTO is failed. (" + this + ")", e);
		}
		return dto;
	}

	/**
	 * メンバ変数の妥当性を確認する。<br/>
	 * @throws CalendarInvalid 妥当でない値が存在した場合
	 */
	public void validate() throws CalendarInvalid {
		// null check
		if (beginTime == null) {
			throw new CalendarInvalid("begin time must be defined(not null). (" + this + ")");
		}
		if (endTime == null) {
			throw new CalendarInvalid("end time must be defined(not null). (" + this + ")");
		}

		// variable consistency
		if (weekday != Calendar.SUNDAY && weekday != Calendar.MONDAY && weekday != Calendar.TUESDAY
				&& weekday != Calendar.WEDNESDAY && weekday != Calendar.THURSDAY && weekday != Calendar.FRIDAY
				&& weekday != Calendar.SATURDAY) {
			throw new CalendarInvalid("weekday must be defined in java.util.Calendar class. (" + this + ")");
		}
		if (beginTime > endTime) {
			throw new CalendarInvalid("end time must be after begin time. (" +
					this + ", " + beginTime + "," + endTime + ")");
		}
	}

	@Override
	public String toString() {
		return "CalendarWeekdayDTO (weekday = " + weekday
		+ ", beginDate = " + beginTime + ", endDate = " + endTime + ")";
	}
}