/*

Copyright (C) 2011 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.util;

import java.sql.Date;
import java.sql.Time;
import java.util.Calendar;

import com.clustercontrol.ws.agent.CalendarDTO;
import com.clustercontrol.ws.agent.CalendarIrregularDTO;
import com.clustercontrol.ws.agent.CalendarIrregularType;
import com.clustercontrol.ws.agent.CalendarWeekdayDTO;

/**
 * カレンダ設定を用いた汎用処理をまとめたクラス(staticメソッドのみを保持)
 * 
 * @since 4.0
 * @author Tomoya Takahata
 */
public class CalendarWSUtil {

	/**
	 * 特定のカレンダ設定に対して、現在日時が実行期間に含まれるかどうかを判定するメソッド
	 * @param config カレンダ設定
	 * @return 実行期間に含まれる場合はtrue, それ以外はfalse
	 */
	public static boolean isRunnable(CalendarDTO config) {
		return isRunnable(config, new Date(System.currentTimeMillis()));
	}

	/**
	 * 特定のカレンダ設定に対して、指定日時が実行期間に含まれるかどうかを判定するメソッド
	 * @param config カレンダ設定
	 * @param date 指定日時
	 * @return 実行期間に含まれる場合はtrue, それ以外はfalse
	 */
	public static boolean isRunnable(CalendarDTO config, Date date) {
		// ローカル変数
		Calendar weekdayCal = null;
		Calendar checkCal = null;

		// 有効期間外の場合はfalse
		if (date.getTime() < config.getBeginDate() || config.getEndDate() < date.getTime()) {
			return false;
		}

		// 実行例外の時間帯に含まれている場合は、その実行種別を返す
		for (CalendarIrregularDTO dto : config.getIrregulars()) {
			if (dto.getBeginDate() <= date.getTime() && date.getTime() <= dto.getEndDate()) {
				return dto.getType() == CalendarIrregularType.ON ? true : false;
			}
		}

		// 実行規則の曜日に含まれているかどうか
		for (CalendarWeekdayDTO dto : config.getWeekdays()) {
			weekdayCal = Calendar.getInstance();
			weekdayCal.setTime(date);
			int weekday = weekdayCal.get(Calendar.DAY_OF_WEEK);

			// 実行規則で定義されている曜日の場合
			if (dto.getWeekday() == weekday) {
				checkCal = Calendar.getInstance();
				checkCal.set(Calendar.YEAR, 1970);
				checkCal.set(Calendar.MONTH, 0);
				checkCal.set(Calendar.DAY_OF_MONTH, 1);
				checkCal.set(Calendar.HOUR_OF_DAY, weekdayCal.get(Calendar.HOUR_OF_DAY));
				checkCal.set(Calendar.MINUTE, weekdayCal.get(Calendar.MINUTE));
				checkCal.set(Calendar.SECOND, weekdayCal.get(Calendar.SECOND));
				Time checkTime = new Time(checkCal.getTime().getTime());

				// 実行規則で定義されている時間帯内の場合
				if (dto.getBeginTime() <= checkTime.getTime() && checkTime.getTime() <= dto.getEndTime()) {
					return true;
				}
			}
		}

		// 有効期間内だが、実行例外・実行規則の時間帯に含まれない場合
		return false;
	}

	public static String toStringCalendarDTO(CalendarDTO dto) {
		// Local Variables
		String ret = null;
		String weekdayStr = null;
		String irregularStr = null;

		// MAIN
		if (dto == null)
			return null;


		if (dto.getWeekdays() != null) {
			for (CalendarWeekdayDTO weekday : dto.getWeekdays()) {
				weekdayStr += weekdayStr == null ? toStringCalendarWeekdayDTO(weekday) : ", " + toStringCalendarWeekdayDTO(weekday);
			}
		}

		if (dto.getIrregulars() != null) {
			for (CalendarIrregularDTO irregular : dto.getIrregulars()) {
				irregularStr += irregularStr == null ? toStringCalendarIrregularDTO(irregular) : ", " + toStringCalendarIrregularDTO(irregular);
			}
		}

		ret = "CalendarDTO [calendarId = " + dto.getCalendarId()
		+ ", calendarName" + dto.getCalendarName()
		+ ", beginDate = " + dto.getBeginDate()
		+ ", endDate = " + dto.getEndDate()
		+ ", weekday = (" + weekdayStr + ")"
		+ ", irregular = (" + irregularStr + ")"
		+ "]";

		return ret;
	}

	public static String toStringCalendarWeekdayDTO(CalendarWeekdayDTO dto) {
		if (dto == null)
			return null;

		return "CalendarWeekdayDTO [weekday = " + dto.getWeekday()
		+ ", beginDate = " + dto.getBeginTime() + ", endDate = " + dto.getEndTime() + "]";
	}

	public static String toStringCalendarIrregularDTO(CalendarIrregularDTO dto) {
		if (dto == null)
			return null;

		return "CalendarIrregularDTO (beginDate = " + dto.getBeginDate()
		+ ", endDate = " + dto.getEndDate() + ", type = " + dto.getType() + ")";
	}

}
