#!/bin/bash

# Copyright (C) since 2006 NTT DATA Corporation
# 
# This program is free software; you can redistribute it and/or
# Modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation, version 2.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details

# bash configuration
SCRIPT_DIR=$(cd $(dirname $0);pwd)
. ${SCRIPT_DIR}/../../hinemos.cfg
. ${SCRIPT_DIR}/../hinemos_utility.sh

DIR=`dirname $0`
USER=`/usr/bin/whoami`
HOST=`hostname`

########################################
#  Local Variable
########################################
DBPORT=24001
DBUSER=hinemos
DBNAME=hinemos
DBPASSWORD=hinemos

########################################
#  Local Message
########################################

# INFO
MSG_I001="removing events confirmed..."
MSG_I002="removing all event..."
MSG_I003="removing job histories which is already done..."
MSG_I004="removing all job history..."
MSG_I005="removing collective-run histories which is already done..."
MSG_I006="removing all collective-run history..."
MSG_I007="removing vm operation histories which is already done..."
MSG_I008="removing all vm operation history..."
MSG_I009="removing all status..."
MSG_I010="removing performance data..."
MSG_I101="successful in removing data."

# WARN
#MSG_W001="xxx"

# ERROR
MSG_E001="failed to remove events or histories."
MSG_E002="no argument specified."

########################################
# Function
########################################

#
# show usage
#
function usage() {
	echo "usage : ${PROG} [options] [-c|-C|-e|-E|-j|-J|-S|-v|-V|-p CollectorId|-P]"
	echo "options:"
	echo "  -r day     specify retention period (performance data only)"
	echo "  -w passwd  set password (for Hinemos RDBM Server)"
	echo "command:"
	echo "  -c              remove collective-run histories which is already done."
	echo "  -C              remove all collective-run history."
	echo "  -e              remove events confirmed."
	echo "  -E              remove all event."
	echo "  -j              remove job histories which is already done."
	echo "  -J              remove all job history."
	echo "  -S              remove all status."
	echo "  -v              remove vm operation histories which is already done."
	echo "  -V              remove all vm operation history."
	echo "  -p CollectorId  remove performance data."
	echo "  -P              remove all performance data."
}

########################################
# SHELL
########################################

# help chekc
for OPT in $@
do
	case $OPT in
		 --help)
			usage
			exit 0
			;;
	esac
done


# option check
while getopts cCeEjJp:PSvVr:w: OPT
do
	case $OPT in
		e)
			ARGS="${ARGS} e"
			;;
		E)
			ARGS="${ARGS} E"
			;;
		c)
			ARGS="${ARGS} c"
			;;
		C)
			ARGS="${ARGS} C"
			;;
		j)
			ARGS="${ARGS} j"
			;;
		J)
			ARGS="${ARGS} J"
			;;
		p)
			ARGS="${ARGS} p"
			COLLECTOR_ID=$OPTARG
			;;
		P)
			ARGS="${ARGS} P"
			;;
		S)
			ARGS="${ARGS} S"
			;;
		v)
			ARGS="${ARGS} v"
			;;
		V)
			ARGS="${ARGS} V"
			;;
		r)
			START_DATE=`date +%Y-%m-%d_%H%M%S --date "${OPTARG} day ago"`
			;;
		w)
			PASSWD=$OPTARG
			;;
		*)
			ExitIllegalOptionErrorWithoutLogger
			;;
	esac
done

shift $(( $OPTIND - 1 ))

if [ ! $# = 0 -o "x${ARGS}" = "x" ]
then
	ExitIllegalArgumentError
fi


# check process
CheckJBossProcessStopped
CheckPostgreSQLProcessRunning


# main
ReadPostgreSQLPasswd

for ARG in ${ARGS}
do
	case $ARG in
		e)
			SQL_EXEC_MSG="${MSG_I001}"
			SQL=`cat <<DML
BEGIN;
DELETE FROM cc_event_log WHERE confirm_flg = '1';
COMMIT;
DML`
			;;
		E)
			SQL_EXEC_MSG="${MSG_I002}"
			SQL=`cat <<DML
BEGIN;
TRUNCATE TABLE cc_event_log CASCADE;
COMMIT;
DML`
			;;
		j)	
			SQL_EXEC_MSG="${MSG_I003}"
			SQL=`cat <<DML
BEGIN;
SELECT session_id INTO TEMPORARY cc_job_completed_sessions FROM cc_job_session_job 
   WHERE ( status = '300' OR status = '301' ) AND ( job_id, session_id ) IN ( SELECT job_id, session_id FROM cc_job_session );
DELETE FROM cc_job_session WHERE session_id IN ( SELECT session_id FROM cc_job_completed_sessions );
DELETE FROM cc_notify_relation_info
   WHERE ( notify_group_id ) LIKE ANY ( SELECT DISTINCT 'JOB_SES-' || session_id || '%' FROM cc_job_completed_sessions );
DELETE FROM cc_monitor_status WHERE plugin_id='JOB' AND monitor_id IN (SELECT session_id FROM cc_job_completed_sessions);
DELETE FROM cc_notify_history WHERE plugin_id='JOB' AND monitor_id IN (SELECT session_id FROM cc_job_completed_sessions);
COMMIT;
DML`
			;;
		J)
			SQL_EXEC_MSG="${MSG_I004}"
			SQL=`cat <<DML
BEGIN;
TRUNCATE TABLE cc_job_session CASCADE;
DELETE FROM cc_notify_relation_info WHERE notify_group_id LIKE 'JOB_SES%';
DELETE FROM cc_monitor_status WHERE plugin_id='JOB';
DELETE FROM cc_notify_history WHERE plugin_id='JOB';
COMMIT;
DML`
			;;
		c)
			SQL_EXEC_MSG="${MSG_I005}"
			SQL=`cat <<DML
BEGIN;
DELETE FROM cc_crun_session WHERE status = '300';
COMMIT;
DML`
			;;
		C)
			SQL_EXEC_MSG="${MSG_I006}"
			SQL=`cat <<DML
BEGIN;
TRUNCATE TABLE cc_crun_session CASCADE;
COMMIT;
DML`
			;;
		v)
			SQL_EXEC_MSG="${MSG_I007}"
			SQL=`cat <<DML
BEGIN;
DELETE FROM cc_vm_operation_detail WHERE status = 2 OR status = 3;
COMMIT;
DML`
			;;
		V)
			SQL_EXEC_MSG="${MSG_I008}"
			SQL=`cat <<DML
BEGIN;
TRUNCATE TABLE cc_vm_operation_detail CASCADE;
COMMIT;
DML`
			;;

		S)
			SQL_EXEC_MSG="${MSG_I009}"
			SQL=`cat <<DML
BEGIN;
TRUNCATE TABLE cc_status_info CASCADE;
COMMIT;
DML`
			;;
		p)
			SQL_EXEC_MSG="${MSG_I010} CollectorId=${COLLECTOR_ID}"
			if [ "x"${START_DATE} = "x" ]
			then
				SQL=`cat <<DML
BEGIN;
UPDATE cc_collector_profile SET start_date = (SELECT max(date_time) FROM cc_calculated_data WHERE collectorid = '${COLLECTOR_ID}') WHERE collectorid = '${COLLECTOR_ID}' AND EXISTS (SELECT * FROM cc_calculated_data WHERE collectorid = '${COLLECTOR_ID}');
DELETE FROM cc_calculated_data WHERE collectorid = '${COLLECTOR_ID}';
UPDATE cc_collector_status SET count = (SELECT count(*) FROM (SELECT date_time FROM cc_calculated_data WHERE collectorid = '${COLLECTOR_ID}' GROUP BY date_time) A) WHERE collectorid = '${COLLECTOR_ID}';
COMMIT;
DML`
			else
				SQL=`cat <<DML
BEGIN;
DELETE FROM cc_calculated_data WHERE collectorid = '${COLLECTOR_ID}' and date_time < '${START_DATE}';
UPDATE cc_collector_profile SET start_date = (SELECT min(date_time) AS older_date_time FROM cc_calculated_data WHERE collectorid = '${COLLECTOR_ID}') WHERE collectorid = '${COLLECTOR_ID}' AND EXISTS (SELECT * FROM cc_calculated_data WHERE collectorid = '${COLLECTOR_ID}');
UPDATE cc_collector_status SET count = (SELECT count(*) FROM (SELECT date_time FROM cc_calculated_data WHERE collectorid = '${COLLECTOR_ID}' GROUP BY date_time) A) WHERE collectorid = '${COLLECTOR_ID}';
COMMIT;
DML`
			fi
			;;
		P)
			SQL_EXEC_MSG="${MSG_I010}"
			if [ "x"${START_DATE} = "x" ]
			then
				SQL=`cat <<DML
BEGIN;
TRUNCATE TABLE cc_calculated_data;
UPDATE cc_collector_status SET count = 0;
COMMIT;
DML`
			else
				SQL=`cat <<DML
BEGIN;
DELETE FROM cc_calculated_data WHERE date_time < '${START_DATE}';

CREATE TEMPORARY TABLE tmp_calcdata_summay AS SELECT collectorid, count(collectorid), min(date_time) AS older_date_time FROM (SELECT collectorid, date_time FROM cc_calculated_data GROUP BY collectorid, date_time ORDER BY collectorid) A GROUP BY collectorid;
UPDATE cc_collector_status B SET count = A.count FROM tmp_calcdata_summay AS A WHERE A.collectorid = B.collectorid;
UPDATE cc_collector_profile B SET start_date = A.older_date_time FROM tmp_calcdata_summay AS A WHERE A.collectorid = B.collectorid;
COMMIT;
DML`
			fi
			;;
	esac

	Logging "${SQL_EXEC_MSG}"
	MSG=`${PG_HOME}/bin/psql -p ${DBPORT} -U ${DBUSER} -d ${DBNAME} -c "${SQL}" 2>&1`
	RET=$?
	if [ ${RET} -ne 0 ]
	then
		Logging "${MSG}"
		Logging "${MSG_E001}"
		RET=$((16 + ${RET}))
		exit ${RET}
	fi
done


########################################
# Termination Processing
########################################

Logging "${MSG_I101}"
exit 0 

