/*
Copyright (C) 2010 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
*/
package com.clustercontrol.ws.endpoint;

import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Date;

import javax.ejb.CreateException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.FacilityDuplicateException;
import com.clustercontrol.FacilityNotFoundException;
import com.clustercontrol.HinemosUnknownException;
import com.clustercontrol.UsedFacilityException;
import com.clustercontrol.repository.bean.NodeInfo;
import com.clustercontrol.repository.bean.RepositoryTableInfo;
import com.clustercontrol.repository.bean.ScopeInfo;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;

/**
 * リポジトリ操作用のWebAPIエンドポイント
 */
@javax.jws.WebService
public class RepositoryEndpoint {
	protected static Log m_log = LogFactory.getLog( RepositoryEndpoint.class );
	
	public String echo(String str) {
		return str + ", " + str;
	}
	
	private RepositoryControllerLocal getRepositoryControllerLocal() throws HinemosUnknownException {
		RepositoryControllerLocal repository = null;
		try {
			repository = RepositoryControllerUtil.getLocalHome().create();
		} catch (CreateException e) {
			m_log.error("getRepositoryControllerLocal CreateException : " + e,e);
			throw new HinemosUnknownException(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error("getRepositoryControllerLocal NamingException : " + e,e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return repository;
	}
	
	/**********************
	 * ノードのメソッド群(getter)
	 **********************/
	/**
	 * ノード一覧を取得します。<BR>
	 * 
	 * クライアントなどで検索した場合に呼ばれ、該当するノード一覧を取得します。<BR>
	 * 引数はNodeInfoであり、"ファシリティID"、"ファシリティ名"、"説明"、
	 * "IPアドレス"、"OS名"、"OSリリース"、"管理者"、"連絡先"が１つ以上含まれており、
	 * その条件を元に該当するノードを戻り値とします。<BR>
	 * 戻り値はNodeInfoのArrayListで、NodeInfoには
	 * ノードの"ファシリティID"、"ファシリティ名"、"説明"のみ格納されています。<BR>
	 * 
	 * @param property　検索条件のプロパティ
	 * @return NodeInfoの配列
	 * @throws HinemosUnknownException 
	 */
	public ArrayList<NodeInfo> getFilterNodeList(NodeInfo property) throws HinemosUnknownException {
		m_log.debug("getFilterNodeList : nodeInfo=" + property);
		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getFilterNodeList(property);
	}
	
	/**
	 * 
	 * 監視・ジョブ等の処理を実行する対象となる、ファシリティIDのリストを取得します。
	 * 引数で指定されたファシリティIDが、ノードかスコープによって、以下のようなリストを取得します。
	 * 
	 * ノードの場合
	 *   引数で指定されたfacilityIdが格納されたArrauList
	 *   ただし、管理対象（有効/無効フラグが真）の場合のみ
	 * 
	 * スコープの場合
	 *   配下に含まれるノードのファシリティIDが格納されたArrayList
	 *   ただし、管理対象（有効/無効フラグが真）のみ
	 * 
	 * @param facilityId 処理を実行する対象となるファシリティID
	 * @return 有効なノードのリスト（有効なノードがひとつも含まれない場合は空のリスト）
	 * @throws HinemosUnknownException 
	 */
	public ArrayList<String> getExecTargetFacilityIdList(String facilityId) throws HinemosUnknownException {
		m_log.debug("getExecTargetFacilityIdList : facilityId=" + facilityId);
		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getExecTargetFacilityIdList(facilityId);
	}
	
	
	/**
	 * ノードの詳細プロパティを取得します。<BR>
	 * 
	 * faciliyIDで指定されるノードの詳細プロパティを取得します。<BR>
	 * 
	 * @param facilityId ファシリティID
	 * @param locale クライアントのロケール
	 * @return ノード情報プロパティ
	 * @throws FacilityNotFoundException 
	 * @throws HinemosUnknownException 
	 */
	public NodeInfo getNode(String facilityId) throws FacilityNotFoundException, HinemosUnknownException {
		m_log.debug("getNode : facilityId=" + facilityId);
		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getNode(facilityId);
	}
	
	
	/**
	 * ファシリティパスを取得します。<BR>
	 * 
	 * 第一引数がノードの場合は、パスではなく、ファシリティ名。<BR>
	 * 第一引数がスコープの場合は、第二引数との相対的なファシリティパスを取得します。<BR>
	 * (例　○○スコープ>××システム>DBサーバ)<BR>
	 * 
	 * @param facilityId ファシリティID
	 * @param parentFacilityId 上位のファシリティID
	 * @return String ファシリティパス
	 * @throws HinemosUnknownException 
	 */
	public String getFacilityPath(String facilityId, String parentFacilityId) throws HinemosUnknownException {
		m_log.debug("getFacilityPath : facilityId=" + facilityId +
				", parentFacilityId=" + parentFacilityId);
		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getFacilityPath(facilityId, parentFacilityId);
	}
	
	
	/**
	 * SNMPを利用してノードの情報を取得します。<BR>
	 * 
	 * 
	 * クライアントからSNMPで検出を行った際に呼び出されるメソッドです。<BR>
	 * SNMPポーリングにより、ノード詳細プロパティをセットし、クライアントに返す。
	 * 戻り値はNodeInfo
	 * 
	 * @param pollingData ポーリング対象のIPアドレス、コミュニティ名、バージョン、ポート
	 * @param locale クライアントのロケール
	 * @return ノード情報
	 * @throws FacilityNotFoundException 
	 * @throws HinemosUnknownException 
	 */
	public NodeInfo getNodePropertyBySNMP(String ipAddress, int port, String community, String version) throws UnknownHostException, FacilityNotFoundException, HinemosUnknownException {
		m_log.debug("getNodePropertyBySNMP : ipAddress=" + ipAddress + ", port=" + port + 
				", community=" + community + ", version=" + version);
		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getNodePropertyBySNMP(ipAddress, port, community, version);
	}
	
	
	/**********************
	 * TODO
	 * ノードのメソッド群(getter以外)
	 **********************/
	/**
	 * ノードを新規に追加します。<BR>
	 * またこのメソッドで組み込みスコープである"登録済みノード"スコープにも
	 * 割り当てが行われます。
	 * 
	 * @param nodeinfo 追加するノード情報のプロパティ
	 * @throws FacilityDuplicateException 
	 * @throws HinemosUnknownException 
	 */
	public void addNode(NodeInfo nodeInfo) throws FacilityDuplicateException, HinemosUnknownException {
		m_log.debug("addNode : nodeInfo=" + nodeInfo);
		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		repository.addNode(nodeInfo);
	}
	
	
	/**
	 * ノードを変更します。<BR>
	 * 引数のpropertyには変更する属性のみを設定してください。<BR>
	 * 
	 * @version 2.0.0
	 * @since 1.0.0
	 * 
	 * @param property　変更するノード情報のプロパティ
	 * @throws HinemosUnknownException 
	 */
	public void modifyNode(NodeInfo property) throws HinemosUnknownException {
		m_log.debug("modifyNode : nodeInfo=" + property);
		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		repository.modifyNode(property);
	}
	
	/**
	 * ノード情報を削除します。<BR>
	 * 
	 * faciityIDで指定されたノードをリポジトリから削除します。
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId ファシリティID
	 * @throws UsedFacilityException 
	 * @throws HinemosUnknownException 
	 */
	public void deleteNode(String facilityId) throws UsedFacilityException, HinemosUnknownException {
		m_log.debug("deleteNode : facilityId=" + facilityId);
		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		repository.deleteNode(facilityId);
	}
	
	/**********************
	 * TODO
	 * スコープのメソッド群
	 **********************/
	/**
	 * スコープ一覧を取得します。<BR>
	 * あるスコープを指定してその直下にあるスコープを取得します。<BR>
	 * このメソッドは引数としてそのスコープのファシリティIDを要求します。<BR>
	 * 戻り値はScopeInfoのArrayListで中のScopeInfoには子スコープの
	 * "ファシリティID"、"ファシリティ名"、"説明"のみ格納されています。<BR>
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param parentFacilityId　スコープのファシリティID
	 * @return ScopeInfoの配列
	 * @throws HinemosUnknownException 
	 */
	/*
	public ArrayList<ScopeInfo> getScopeList(String parentFacilityId) throws HinemosUnknownException {
		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getScopeList(parentFacilityId);
	}
	*/
	
	/**
	 * スコープ用プロパティ情報を取得します。<BR>
	 * 
	 * @param facilityId ファシリティID
	 * @param locale クライアントのロケール
	 * @return スコープのプロパティ情報（ファシリティID、ファシリティ名、説明）
	 * @throws FacilityNotFoundException 
	 * @throws HinemosUnknownException
	 */
	public ScopeInfo getScopeProperty(String facilityId) throws FacilityNotFoundException, HinemosUnknownException {
		m_log.debug("getScopeProperty : facilityId=" + facilityId);
		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getScopeProperty(facilityId);
	}
	
	/**
	 * スコープを新規に追加します。<BR>
	 * 
	 * parentFacilityIdで指定されるスコープの下にpropertyで指定されるスコープを
	 * 追加します。<BR>
	 * 引数propertyには、"ファシリティID"、"ファシリティ名"、"説明"（任意）を含める必要があります。
	 * 
	 * @param parentFacilityId
	 * @param property
	 * @throws FacilityDuplicateException 
	 * @throws HinemosUnknownException 
	 */
	public void addScope(String parentFacilityId, ScopeInfo property) throws FacilityDuplicateException, HinemosUnknownException {
		m_log.debug("addScope : parentFacilityId=" + parentFacilityId +
		", scopeInfo=" + property);
		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		repository.addScope(parentFacilityId, property);
	}
	
	
	/**
	 * スコープの情報を変更します。<BR>
	 * 
	 * 引数propertyで指定した内容でスコープ情報を更新します。<BR>
	 * 引数propertyには、"ファシリティID"、"ファシリティ名"、"説明"（任意）を含める必要があります。
	 * propertyに含まれるファシリティIDに対応するスコープの情報が変更されます。<BR>
	 * 
	 * @param property
	 * @throws HinemosUnknownException 
	 */
	public void modifyScope(ScopeInfo property) throws HinemosUnknownException {
		m_log.debug("modifyScope : scopeInfo=" + property);
		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		repository.modifyScope(property);
	}
	
	/**
	 * スコープ情報を削除します。<BR>
	 * 
	 * faciityIDで指定されたスコープをリポジトリから削除します。
	 * 
	 * @param facilityId ファシリティID
	 * @throws UsedFacilityException 
	 * @throws HinemosUnknownException 
	 */
	public void deleteScope(String facilityId) throws UsedFacilityException, HinemosUnknownException {
		m_log.debug("deleteScope : facilityId=" + facilityId);
		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		repository.deleteScope(facilityId);
	}
	
	
	
	/**********************
	 * ノード割り当てのメソッド群
	 **********************/
	/**
	 * 割当ノード一覧を取得します。<BR>
	 * 
	 * あるファシリティIDの配下または直下のノード一覧を取得します。<BR>
	 * このメソッドでは、引数levelで直下または配下を制御します。<BR>
	 * 戻り値はNodeInfoのArrayListで、NodeInfoには
	 * ノードの"ファシリティID"、"ファシリティ名"、"説明"のみ格納されています。<BR>
	 * 
	 * @param parentFacilityId
	 * @param level　取得レベル 0:ALL(配下) 1:ONE_LEVEL（直下）
	 * @return NodeInfoの配列
	 * @throws HinemosUnknownException 
	 */
	public ArrayList<NodeInfo> getNodeList(String parentFacilityId, int level) throws HinemosUnknownException {
		m_log.debug("getNodeList : parentFacilityId=" + parentFacilityId + ", level=" + level);
		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getNodeList(parentFacilityId, level);
	}
	
	
	/**
	 * 割当スコープ一覧を取得します。<BR>
	 * 割り当てスコープ一覧とは、あるノードが属しているスコープすべてを
	 * 一覧表示したものです。
	 * クライアントの割り当てスコープビューの表示データとなります。
	 * 戻り値はArrayListのArrayListで中のArrayListには"スコープ"が最上位からの
	 * スコープパス表記で（Stringで）格納されています。
	 * 外のArrayListには、そのレコードが順に格納されています。
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId ノードのファシリティID
	 * @return Stringの配列
	 * @throws HinemosUnknownException 
	 */
	public ArrayList<String> getNodeScopeList(String facilityId) throws HinemosUnknownException {
		m_log.debug("getNodeScopeList : facilityId=" + facilityId);
		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getNodeScopeList(facilityId);
	}
	
	/**
	 * ファシリティIDリストを取得します。<BR>
	 * 
	 * 引数に指定した親ファシリティIDの配下または直下のファシリティ（スコープ、ノード）の
	 * リストを取得します。<BR>
	 * 戻り値は ファシリティID（String）のArrayList
	 * 
	 * @param parentFacilityId ファシリティID
	 * @param level　取得レベル 0:ALL(配下) 1:ONE_LEVEL（直下）
	 * @return ファシリティIDの配列
	 * @throws HinemosUnknownException 
	 */
	public ArrayList<String> getFacilityIdList(String parentFacilityId, int level) throws HinemosUnknownException {
		m_log.debug("getFacilityIdList : parentFacilityId=" + parentFacilityId + ", level=" + level);
		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getFacilityIdList(parentFacilityId, level);
	}
	
	
	/**
	 * ノードのファシリティIDリストを取得します。<BR>
	 * 
	 * 引数に指定した親ファシリティIDの配下または直下のファシリティ（ノード）の
	 * リストを取得します。<BR>
	 * 戻り値は ファシリティID（String）のArrayList
	 * 
	 * @param parentFacilityId ファシリティID
	 * @param level  取得レベル 0:ALL(配下) 1:ONE_LEVEL（直下）
	 * @return ファシリティIDの配列
	 * @throws HinemosUnknownException 
	 */
	public ArrayList<String> getNodeFacilityIdList(String parentFacilityId, int level) throws HinemosUnknownException {
		m_log.debug("getNodeFacilityIdList : parentFacilityId=" + parentFacilityId + ", level=" + level);
		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getNodeFacilityIdList(parentFacilityId, level);
	}
	
	/**
	 * スコープへのノードの割り当てを行います。<BR>
	 * 
	 * parentFacilityIdで指定されるスコープにfacilityIdsで指定されるノード群を
	 * 割り当てます。
	 * 
	 * @param parentFacilityId　ノードを割り当てるスコープ
	 * @param facilityIds 割り当てさせるノード(群)
	 * @throws HinemosUnknownException 
	 */
	public void assignNodeScope(String parentFacilityId, String[] facilityIds) throws HinemosUnknownException {
		m_log.debug("assignNodeScope : parentFacilityId=" + parentFacilityId + ", facilityIds=" + facilityIds);
		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		repository.assignNodeScope(parentFacilityId, facilityIds);
	}
	
	/**
	 * ノードをスコープから削除します。（割り当てを解除します。）<BR>
	 * parentFacilityIdで指定されるスコープからfacilityIdsで指定されるノード群を
	 * 削除（割り当て解除）します。
	 * 
	 * @param parentFacilityId ノードを取り除くスコープ
	 * @param facilityIds 取り除かれるノード（群）
	 * @throws HinemosUnknownException 
	 */
	public void releaseNodeScope(String parentFacilityId, String[] facilityIds) throws HinemosUnknownException {
		m_log.debug("releaseNodeScope : parentFacilityId=" + parentFacilityId + ", facilityIds=" + facilityIds);
		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		repository.releaseNodeScope(parentFacilityId, facilityIds);
	}
	
	
	/**********************
	 * TODO
	 * その他のメソッド群
	 **********************/
	/**
	 * ファシリティがノードかどうかをチェックします。<BR>
	 * 
	 * ファシリティIDに対応するものがノードかチェックし、結果をbooleanで返します。
	 * 
	 * @param facilityId　ファシリティID
	 * @return true：ノード　false:ノードではない（スコープ）
	 * @throws HinemosUnknownException 
	 * @throws FacilityNotFoundException 
	 */
	public boolean isNode(String facilityId) throws HinemosUnknownException, FacilityNotFoundException {
		m_log.debug("isNode : facilityId=" + facilityId);
		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.isNode(facilityId);
	}
	
	/**
	 * ノード作成変更時に、利用可能プラットフォームを表示するためのメソッド。
	 * 
	 * @return ArrayList
	 * @throws HinemosUnknownException 
	 */
	public ArrayList<RepositoryTableInfo> getPlatformList() throws HinemosUnknownException{
		m_log.debug("getPlatformList : ");
		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getPlatformList();
	}
	
	/**
	 * ノード作成変更時に、仮想化種別hostのファシリティIDを表示するためのメソッド。
	 * 
	 * @return ArrayList
	 * @throws HinemosUnknownException 
	 */
	public ArrayList<RepositoryTableInfo> getHostList() throws HinemosUnknownException{
		m_log.debug("getHostList : ");
		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getHostList();
	}
	
	/**
	 * ノード作成変更時に、利用可能な仮想化ソリューションを表示するためのメソッド。
	 * 
	 * @version 3.2.0
	 * @since 3.2.0
	 * @return ArrayList
	 * @throws HinemosUnknownException 
	 */
	public ArrayList<RepositoryTableInfo> getVmSolutionMstList() throws HinemosUnknownException{
		m_log.debug("getVmSolutionMstList : ");
		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getVmSolutionMstList();
	}
	
	/**
	 * ノード作成変更時に、利用可能な仮想化プロトコルを表示するためのメソッド。
	 * 
	 * @return ArrayList
	 * @throws HinemosUnknownException 
	 */
	public ArrayList<String> getVmProtocolMstList() throws HinemosUnknownException{
		m_log.debug("getVmProtocolMstList : ");
		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getVmProtocolMstList();
	}
	/**
	 * リポジトリの最終更新時刻を取得
	 * 
	 *     
	 * @return
	 * @throws HinemosUnknownException 
	 */
	public Date getLastUpdate() throws HinemosUnknownException{
		m_log.debug("getLastUpdate : ");
		RepositoryControllerLocal repository = getRepositoryControllerLocal();
		return repository.getLastUpdate();
	}
}
