/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.sql.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.sql.bean.JdbcDriverInfo;

/**
 * JDBCドライバリソース取得クラス<BR>
 * 
 * @version 2.0.0
 * @since 2.0.0
 */
public class JdbcDriverUtil { 
	
	protected static Log m_log = LogFactory.getLog( JdbcDriverUtil.class );
			
	private static final String PROPERTIES_FILE = "jdbc.properties";
	private static final String HINEMOS_ETC_DIR = "hinemos.manager.etc.dir";
	
	private static final String JDBC_DRIVER = "jdbc.driver";
	private static final String JDBC_DRIVER_NAME = "jdbc.driver.name.";
	private static final String JDBC_DRIVER_CLASSNAME = "jdbc.driver.classname.";
	private static final String JDBC_DRIVER_LOGINTIMEOUT = "jdbc.driver.logintimeout.";
	private static final String JDBC_DRIVER_PROPERTIES = "jdbc.driver.properties.";
	
	
	private static HashMap<String, JdbcDriverProperties> jdbcProperties = new HashMap<String, JdbcDriverProperties>();

	static {
		try{
			m_log.info("initializing configuration for sql monitoring...");
			
			// Propertiesファイル読み込み
			Properties props = new Properties();
			props.load(new FileInputStream(System.getProperty(HINEMOS_ETC_DIR)+ File.separator + PROPERTIES_FILE));
			
			//JDBCドライバ数取得
			Integer count = Integer.valueOf(props.getProperty(JDBC_DRIVER, "0"));
			m_log.info("use " + count + " jdbc drivers for sql monitoring.");
			
			for(int i = 1; i <= count.intValue(); i++){
				String name = props.getProperty(JDBC_DRIVER_NAME + i, "");
				String classname = props.getProperty(JDBC_DRIVER_CLASSNAME + i, "");
				
				Integer loginTimeout = null;
				if (props.getProperty(JDBC_DRIVER_LOGINTIMEOUT + i) != null) {
					try {
						loginTimeout = Integer.parseInt(props.getProperty(JDBC_DRIVER_LOGINTIMEOUT + i));
					} catch (NumberFormatException e) {
						m_log.warn("jdbc driver definition is invalid. (" + (JDBC_DRIVER_LOGINTIMEOUT + i) + ")" );
					}
				}
				String properties = props.getProperty(JDBC_DRIVER_PROPERTIES + i, "");
				
				m_log.info("setting jdbc driver " + i + " : " + name + "(classname = " + classname + ", login_timeout = " + loginTimeout + ")");
				jdbcProperties.put(classname, new JdbcDriverProperties(classname, name, loginTimeout, properties, i));
			}
		} catch (FileNotFoundException e) {
			m_log.warn("file not found. (" + PROPERTIES_FILE + ")", e);
		} catch (IOException e){
			m_log.warn("file read failure. (" + PROPERTIES_FILE + ")", e);
		}
	}
	
	/**
	 * JDBCドライバ名、クラス名を取得
	 * 
	 * @return
	 */
	public ArrayList<JdbcDriverInfo> getJdbcDriver() {
		
		ArrayList<Map.Entry<String, JdbcDriverProperties>> entities = new ArrayList<Map.Entry<String, JdbcDriverProperties>>(jdbcProperties.entrySet());
		ArrayList<JdbcDriverInfo> list = new ArrayList<JdbcDriverInfo>();
		
		Collections.sort(entities, new Comparator<Map.Entry<String, JdbcDriverProperties>>() {
			public int compare(Map.Entry<String, JdbcDriverProperties> o1, Map.Entry<String, JdbcDriverProperties> o2) {
				return o1.getValue().getPriority() - o2.getValue().getPriority();
			}
		});
		
		for (Map.Entry<String, JdbcDriverProperties> entry : entities) {
			JdbcDriverInfo driver = new JdbcDriverInfo();
			driver.setJdbcDriverName(entry.getValue().getName());
			driver.setJdbcDriverClass(entry.getValue().getClassname());
			list.add(driver);
		}
		
		return list;
	}
	
	public static JdbcDriverProperties getJdbcDriverProperty(String classname) {
		return jdbcProperties.get(classname);
	}
	
}

