/*
 
Copyright (C) since 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.repository.factory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Locale;

import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.FacilityNotFoundException;
import com.clustercontrol.accesscontrol.factory.AccessLock;
import com.clustercontrol.bean.DeviceTypeConstant;
import com.clustercontrol.bean.FacilityConstant;
import com.clustercontrol.bean.FacilityTreeItem;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.commons.util.ObjectValidator;
import com.clustercontrol.repository.bean.FacilityAttributeConstant;
import com.clustercontrol.repository.bean.FacilityInfo;
import com.clustercontrol.repository.bean.FacilityTreeAttributeConstant;
import com.clustercontrol.repository.bean.NodeInfo;
import com.clustercontrol.repository.bean.ScopeInfo;
import com.clustercontrol.repository.ejb.entity.DeviceInfoData;
import com.clustercontrol.repository.ejb.entity.FacilityInfoData;
import com.clustercontrol.repository.ejb.entity.FacilityLocal;
import com.clustercontrol.repository.ejb.entity.FacilityRelationLocal;
import com.clustercontrol.repository.ejb.entity.FacilityUtil;
import com.clustercontrol.repository.ejb.entity.FileSystemInfoData;
import com.clustercontrol.repository.ejb.entity.NodeDeviceLocal;
import com.clustercontrol.repository.ejb.entity.NodeFilesystemLocal;
import com.clustercontrol.repository.ejb.entity.NodeHostnameLocal;
import com.clustercontrol.repository.ejb.entity.NodeLocal;
import com.clustercontrol.repository.ejb.entity.NodeNoteLocal;
import com.clustercontrol.repository.ejb.entity.NodeUtil;
import com.clustercontrol.repository.ejb.session.RepositoryControllerBean;
import com.clustercontrol.repository.util.FacilityValidator;
import com.clustercontrol.repository.util.ListSorter;
import com.clustercontrol.util.Messages;

public class FacilitySelector {

	protected static Log m_log = LogFactory.getLog(FacilitySelector.class);
	
	public static final String SEPARATOR = ">";
	
	/** ----- ファシリティの木情報のキャッシュ ----- */
	private static volatile HashMap<Locale, HashMap<Boolean, FacilityTreeItem>> cachedFacilityTree = new HashMap<Locale, HashMap<Boolean, FacilityTreeItem>>();
	private static volatile HashMap<Locale, HashMap<Boolean, FacilityTreeItem>> cachedFacilityTreeScope = new HashMap<Locale, HashMap<Boolean, FacilityTreeItem>>();
	
	static {
		initCacheFacilityTree();
	}
	
	/**
	 * ファシリティの木情報のキャッシュを初期化する
	 */
	public static void initCacheFacilityTree() {
		m_log.info("initializing a facility's tree to cache.");
		synchronized (cachedFacilityTree) {
			cachedFacilityTree = new HashMap<Locale, HashMap<Boolean, FacilityTreeItem>>();
		}
		synchronized (cachedFacilityTreeScope) {
			cachedFacilityTreeScope = new HashMap<Locale, HashMap<Boolean, FacilityTreeItem>>();
		}
	}
	
	/**
	 * ノードのファシリティ情報を取得する。<BR>
	 * 
	 * @param facilityId ノードのファシリティID
	 * @return ファシリティ情報
	 */
	public static FacilityInfoData getNodeFacilityData(String facilityId) {
		/** ローカル変数 */
		FacilityInfoData data = null;
		FacilityLocal facility = null;
		NodeLocal node = null;
		
		/** メイン処理 */
		m_log.debug("getting a node's data...");
		
		try {
			// 該当するファシリティインスタンスを取得する
			facility = FacilityUtil.getLocalHome().findByPrimaryKey(facilityId);
			
			// ファシリティがノードでない場合はエラーとする
			if (facility.getFacilityType() != FacilityConstant.TYPE_NODE) {
				throw new EJBException("this facility is not Node. (facilityId = " + facilityId + ")");
			}
			
			// 該当するノードインスタンスを取得する
			node = facility.getNode();
			
			// ファシリティデータを生成
			data = new FacilityInfoData();
			
			// ----- ファシリティ関連 -----
			data.setFacilityId(facility.getFacilityId());
			data.setCn(facility.getFacilityName());
			data.setType(facility.getFacilityType());
			data.setSortValue(facility.getDisplaySortOrder());
			data.setDescription(facility.getDescription());
			data.setValid(facility.isValid());
			data.setCreatorsName(facility.getCreateUserId());
			data.setCreateTimestamp(facility.getCreateDatetime().getTime() != 0 ? new Date(facility.getCreateDatetime().getTime()) : null);
			data.setModifiersName(facility.getModifyUserId());
			data.setModifyTimestamp(facility.getModifyDatetime().getTime() != 0 ? new Date(facility.getModifyDatetime().getTime()) : null);
			
			// ----- ノード関連 -----
			data.setPlatform(node.getPlatformFamily());
			data.setMachine(node.getHardwareType());
			data.setIconImage(node.getIconImage());
			
			// ----- SNMP関連 -----
			data.setSnmpPort(node.getSnmpPort() != -1 ? node.getSnmpPort() : null);
			data.setSnmpCommunity(node.getSnmpCommunity());
			data.setSnmpVersion(node.getSnmpVersion());
			data.setSnmpTimeout(node.getSnmpTimeout() != -1 ? node.getSnmpTimeout() : null);
			data.setSnmpRetries(node.getSnmpRetryCount() != -1 ? node.getSnmpRetryCount() : null);
			data.setSnmpProxy(node.getSnmpProxy());
			
			// ----- WBEM関連 -----
			data.setWbemPort(node.getWbemPort() != -1 ? node.getWbemPort() : null);
			data.setWbemUser(node.getWbemUser());
			data.setWbemUserPassword(node.getWbemUserPassword());
			data.setWbemProtocol(node.getWbemProtocol());
			data.setWbemTimeout(node.getWbemTimeout() != -1 ? node.getWbemTimeout() : null);
			data.setWbemRetries(node.getWbemRetryCount() != -1 ? node.getWbemRetryCount() : null);
			
			// ----- IPアドレス関連 -----
			data.setDhcpClient(node.isDhcpClient());
			data.setIpType(node.getIpAddressType() != -1 ? node.getIpAddressType() : null);
			data.setIpProtocolNumber(node.getIpAddressVersion() != -1 ? node.getIpAddressVersion() : null);
			data.setIpNetworkNumber(node.getIpAddressV4());
			data.setIpNetworkNumberV6(node.getIpAddressV6());
			
			// ----- OS関連 -----
			data.setNodeName(node.getNodeName());
			data.setOsName(node.getOsName());
			data.setOsRelease(node.getOsRelease());
			data.setOsVersion(node.getOsVersion());
			data.setCharSet(node.getCharacterSet());
			
			// ----- 仮想化関連 -----
			data.setVirtNodeType(node.getVirtualizationNodeType());
			data.setVMManagementNode(node.getVmManagementNode());
			data.setVmIndex(node.getVmIndex() != -1 ? node.getVmIndex() : null);
			data.setVmName(node.getVmName());
			data.setVirtSolution(node.getVirtualizationSolution());
			data.setVmId(node.getVmId());
			data.setVmUser(node.getVmUser());
			data.setVmUserpassword(node.getVmUserPassword());
			data.setVmProtocol(node.getVmProtocol());
			
			// ----- 管理関連 -----
			data.setManagerContact(node.getContact());
			data.setManagerName(node.getAdministrator());
			
			// ----- ホスト名 -----
			data.setHost(getHostnameList(node));
			
			// ----- 備考 -----
			data.setNote(getNoteList(node));
			
			// ----- デバイス情報 -----
			data.setDeviceInfo(getDeviceList(node));
			
			// ---- ファイルシステム情報 -----
			data.setFileSystemInfo(getFileSystemList(node));
		} catch (Exception e) {
			m_log.info("failure to get a node's data. (facilityId = " + facilityId + ")", e);
			throw new EJBException("failure to get a node's data. (facilityId = " + facilityId + ")", e);
		}
		
		m_log.debug("successful in getting a node's data.");
		return data;
	}
	
	/**
	 * ノードに属するホスト名の一覧を取得する。<BR>
	 * 
	 * @param node ノードインスタンス
	 * @return ホスト名の配列
	 */
	private static ArrayList<String> getHostnameList(NodeLocal node) {
		/** ローカル変数 */
		ArrayList<String> hostnameList = null;
		
		/** メイン処理 */
		hostnameList = new ArrayList<String>();
		if (node.getNodeHostname() != null) {
			for (NodeHostnameLocal nodeHostname : (Collection<NodeHostnameLocal>)node.getNodeHostname()) {
				hostnameList.add(nodeHostname.getHostname());
			}
		}
		
		return hostnameList;
	}
	
	/**
	 * ノードに属する備考の一覧を取得する。<BR>
	 * 
	 * @param node ノードインスタンス
	 * @return 備考の配列
	 */
	private static ArrayList<String> getNoteList(NodeLocal node) {
		/** ローカル変数 */
		ArrayList<String> noteList = null;
		
		/** メイン処理 */
		noteList = new ArrayList<String>();
		if (node.getNodeNote() != null) {
			for (NodeNoteLocal nodeNote : (Collection<NodeNoteLocal>)node.getNodeNote()) {
				noteList.add(nodeNote.getNote());
			}
		}
		
		return noteList;
	}
	
	/**
	 * ノードに属するデバイスの一覧を取得する。<BR>
	 * 
	 * @param facilityId ノードのファシリティID
	 * @return デバイス情報の配列
	 */
	public static ArrayList<DeviceInfoData> getDeviceList(String facilityId) {
		/** ローカル変数 */
		ArrayList<DeviceInfoData> devices = null;
		NodeLocal node = null;
		
		/** メイン処理 */
		m_log.debug("getting node's devices...");
		try {
			// 該当するノードインスタンスを取得
			node = NodeUtil.getLocalHome().findByPrimaryKey(facilityId);
			
			// デバイス一覧を取得
			devices = getDeviceList(node);
		} catch (Exception e) {
			m_log.info("failure to get node's devices. (facilityId = " + facilityId + ")", e);
			throw new EJBException("failure to get node's devices. (facilityId = " + facilityId + ")", e);
		}
		
		m_log.debug("successful in getting node's devices. (facilityId = " + facilityId + ")");
		return devices;
	}
	
	/**
	 * ノードに属するデバイスの一覧を取得する。<BR>
	 * 
	 * @param node ノードインスタンス
	 * @return デバイス情報の配列
	 */
	private static ArrayList<DeviceInfoData> getDeviceList(NodeLocal node) {
		/** ローカル変数 */
		ArrayList<DeviceInfoData> deviceInfoDataList = null;
		DeviceInfoData deviceInfoData = null;
		
		/** メイン処理 */
		deviceInfoDataList = new ArrayList<DeviceInfoData>();
		
		// デバイス情報の一覧を取得
		if (node.getNodeDevice() != null) {
			for (NodeDeviceLocal nodeDevice : (Collection<NodeDeviceLocal>)node.getNodeDevice()) {
				deviceInfoData = new DeviceInfoData();
				deviceInfoData.setCn(nodeDevice.getDeviceIndex().toString());
				deviceInfoData.setFacilityId(nodeDevice.getFacilityId());
				deviceInfoData.setDeviceType(nodeDevice.getDeviceType());
				deviceInfoData.setDeviceIndex(nodeDevice.getDeviceIndex());
				deviceInfoData.setDeviceName(nodeDevice.getDeviceName());
				deviceInfoData.setDisplayName(nodeDevice.getDeviceDisplayName());
				deviceInfoData.setSnmpOID(nodeDevice.getDeviceSnmpOid());
				deviceInfoData.setDescription(nodeDevice.getDeviceDescription());

				deviceInfoDataList.add(deviceInfoData);
			}
		}
		
		// ファイルシステム情報の一覧を取得
		if (node.getNodeFilesystem() != null) {
			for (NodeFilesystemLocal nodeFilesystem : (Collection<NodeFilesystemLocal>)node.getNodeFilesystem()) {
				deviceInfoData = new DeviceInfoData();
				
				deviceInfoData.setCn(nodeFilesystem.getFsIndex().toString());
				deviceInfoData.setFacilityId(nodeFilesystem.getFacilityId());
				deviceInfoData.setDeviceType(DeviceTypeConstant.DEVICE_FILESYSTEM);
				deviceInfoData.setDeviceIndex(nodeFilesystem.getFsIndex());
				deviceInfoData.setDeviceName(nodeFilesystem.getFsMountPoint());
				deviceInfoData.setDisplayName(nodeFilesystem.getFsDisplayName());
				deviceInfoData.setSnmpOID(nodeFilesystem.getFsSnmpOid());
				deviceInfoData.setDescription(nodeFilesystem.getFsDescription());
				
				deviceInfoDataList.add(deviceInfoData);
			}
		}
		
		return deviceInfoDataList;
	}
	
	/**
	 * ファイルシステム情報の一覧を取得する。<BR>
	 * 
	 * @param facilityId ノードのファシリティID
	 * @return ファイルシステム情報の一覧
	 */
	public static ArrayList<FileSystemInfoData> getFileSystemList(String facilityId) {
		/** ローカル変数 */
		ArrayList<FileSystemInfoData> filesystems = null;
		NodeLocal node = null;
		
		/** メイン処理 */
		m_log.debug("getting node's filesystems...");
		
		try {
			// 該当するノードインスタンスを取得
			node = NodeUtil.getLocalHome().findByPrimaryKey(facilityId);
			
			// ファイルシステム一覧を取得
			filesystems = getFileSystemList(node);
		} catch (Exception e) {
			m_log.info("failure to get node's filesystems. (facilityId = " + facilityId + ")", e);
			throw new EJBException("failure to get node's filesystems. (facilityId = " + facilityId + ")", e);
		}
		
		m_log.debug("successful in getting node's filesystems.");
		return filesystems;
	}
	
	/**
	 * ファイルシステム情報の一覧を取得する。<BR>
	 * 
	 * @param node ノードインスタンス
	 * @return ファイルシステム情報の配列
	 */
	private static ArrayList<FileSystemInfoData> getFileSystemList(NodeLocal node) {
		/** ローカル変数 */
		ArrayList<FileSystemInfoData> fsInfoDataList = null;
		FileSystemInfoData fsInfoData = null;
		
		/** メイン処理 */
		fsInfoDataList = new ArrayList<FileSystemInfoData>();
		
		if (node.getNodeFilesystem() != null) {
			for (NodeFilesystemLocal nodeFilesystem : (Collection<NodeFilesystemLocal>)node.getNodeFilesystem()) {
				fsInfoData = new FileSystemInfoData();
				
				fsInfoData.setFacilityId(nodeFilesystem.getFacilityId());
				fsInfoData.setCn(nodeFilesystem.getFsIndex());
				fsInfoData.setFileSystemMountPoint(nodeFilesystem.getFsMountPoint());
				fsInfoData.setDisplayName(nodeFilesystem.getFsDisplayName());
				fsInfoData.setFileSystemType(nodeFilesystem.getFsType());
				fsInfoData.setSnmpOID(nodeFilesystem.getFsSnmpOid());
				fsInfoData.setDescription(nodeFilesystem.getFsDescription());
				
				fsInfoDataList.add(fsInfoData);
			}
		}
		
		return fsInfoDataList;
	}
	
	/**
	 * ノード情報を取得する。<BR>
	 * 
	 * @param facilityId ノードのファシリティID
	 * @param attributes 取得する項目一覧
	 * @return ノード情報
	 * @throws FacilityNotFoundException 該当するファシリティが存在しなかった場合
	 */
	public static HashMap<String, Object> getNodeDetail(String facilityId, ArrayList<String> attributes) throws FacilityNotFoundException {
		/** ローカル変数 */
		HashMap<String, Object> map = null;
		FacilityLocal facility = null;
		NodeLocal node = null;
		
		/** メイン処理 */
		m_log.debug("getting a node's data...");
		
		try {
			// 該当するファシリティインスタンスを取得する
			facility = FacilityUtil.getLocalHome().findByPrimaryKey(facilityId);
			
			// ファシリティがノードかどうかを確認する
			if (facility.getFacilityType() != FacilityConstant.TYPE_NODE) {
				return null;
			}
		} catch (Exception e) {
			m_log.warn("failure to a get node's data. (facilityId = " + facilityId + ")");
			throw new FacilityNotFoundException("failure to a get node's data. (facilityId = " + facilityId + ")");
		}
		
		// ノードインスタンスを取得する
		node = facility.getNode();
		
		map = new HashMap();
		if (attributes != null) {
			// ----- ファシリティ関連 -----
			if (attributes.contains(FacilityAttributeConstant.FACILITYID)) {
				map.put(FacilityAttributeConstant.FACILITYID, facility.getFacilityId());
			}
			if (attributes.contains(FacilityAttributeConstant.CN)) {
				map.put(FacilityAttributeConstant.CN, facility.getFacilityName());
			}
			if (attributes.contains(FacilityAttributeConstant.DESCRIPTION)) {
				map.put(FacilityAttributeConstant.DESCRIPTION, facility.getDescription());
			}
			if (attributes.contains(FacilityAttributeConstant.SORT_VALUE)) {
				map.put(FacilityAttributeConstant.SORT_VALUE, facility.getDisplaySortOrder());
			}
			if (attributes.contains(FacilityAttributeConstant.VALID)) {
				map.put(FacilityAttributeConstant.VALID, facility.isValid());
			}
			if (attributes.contains(FacilityAttributeConstant.CREATORSNAME)) {
				map.put(FacilityAttributeConstant.CREATORSNAME, facility.getCreateUserId());
			}
			if (attributes.contains(FacilityAttributeConstant.CREATETIMESTAMP)) {
				map.put(FacilityAttributeConstant.CREATETIMESTAMP, facility.getCreateDatetime().getTime() == 0 ? null : new Date(facility.getCreateDatetime().getTime()));
			}
			if (attributes.contains(FacilityAttributeConstant.MODIFIERSNAME)) {
				map.put(FacilityAttributeConstant.MODIFIERSNAME, facility.getModifyUserId());
			}
			if (attributes.contains(FacilityAttributeConstant.MODIFYTIMESTAMP)) {
				map.put(FacilityAttributeConstant.MODIFYTIMESTAMP, facility.getModifyDatetime().getTime() == 0 ? null : new Date(facility.getModifyDatetime().getTime()));
			}
			
			// ----- ノード関連 -----
			if (attributes.contains(FacilityAttributeConstant.PLATFORM)) {
				map.put(FacilityAttributeConstant.PLATFORM, node.getPlatformFamily());
			}
			if (attributes.contains(FacilityAttributeConstant.MACHINE)) {
				map.put(FacilityAttributeConstant.MACHINE, node.getHardwareType());
			}
			if (attributes.contains(FacilityAttributeConstant.ICONIMAGE)) {
				map.put(FacilityAttributeConstant.ICONIMAGE, node.getIconImage());
			}
			
			// ----- SNMP関連 -----
			if (attributes.contains(FacilityAttributeConstant.SNMPPORT)) {
				map.put(FacilityAttributeConstant.SNMPPORT, node.getSnmpPort() == -1 ? null : node.getSnmpPort());
			}
			if (attributes.contains(FacilityAttributeConstant.SNMPCOMMUNITY)) {
				map.put(FacilityAttributeConstant.SNMPCOMMUNITY, node.getSnmpCommunity());
			}
			if (attributes.contains(FacilityAttributeConstant.SNMPVERSION)) {
				map.put(FacilityAttributeConstant.SNMPVERSION, node.getSnmpVersion());
			}
			if (attributes.contains(FacilityAttributeConstant.SNMPTIMEOUT)) {
				map.put(FacilityAttributeConstant.SNMPTIMEOUT, node.getSnmpTimeout() == -1 ? null : node.getSnmpTimeout());
			}
			if (attributes.contains(FacilityAttributeConstant.SNMPRETRIES)) {
				map.put(FacilityAttributeConstant.SNMPRETRIES, node.getSnmpRetryCount() == -1 ? null : node.getSnmpRetryCount());
			}
			if (attributes.contains(FacilityAttributeConstant.SNMPPROXY)) {
				map.put(FacilityAttributeConstant.SNMPPROXY, node.getSnmpProxy());
			}
			
			// ----- WBEM関連 -----
			if (attributes.contains(FacilityAttributeConstant.WBEMPORT)) {
				map.put(FacilityAttributeConstant.WBEMPORT, node.getWbemPort() == -1 ? null : node.getWbemPort());
			}
			if (attributes.contains(FacilityAttributeConstant.WBEMUSER)) {
				map.put(FacilityAttributeConstant.WBEMUSER, node.getWbemUser());
			}
			if (attributes.contains(FacilityAttributeConstant.WBEMUSERPASSWORD)) {
				map.put(FacilityAttributeConstant.WBEMUSERPASSWORD, node.getWbemUserPassword());
			}
			if (attributes.contains(FacilityAttributeConstant.WBEMPROTOCOL)) {
				map.put(FacilityAttributeConstant.WBEMPROTOCOL, node.getWbemProtocol());
			}
			if (attributes.contains(FacilityAttributeConstant.WBEMTIMEOUT)) {
				map.put(FacilityAttributeConstant.WBEMTIMEOUT, node.getWbemTimeout() == -1 ? null : node.getWbemTimeout());
			}
			if (attributes.contains(FacilityAttributeConstant.WBEMRETRIES)) {
				map.put(FacilityAttributeConstant.WBEMRETRIES, node.getWbemRetryCount() == -1 ? null : node.getWbemRetryCount());
			}
			
			// ----- IPアドレス関連 -----
			if (attributes.contains(FacilityAttributeConstant.DHCPCLIENT)) {
				map.put(FacilityAttributeConstant.DHCPCLIENT, node.isDhcpClient());
			}
			if (attributes.contains(FacilityAttributeConstant.IPTYPE)) {
				map.put(FacilityAttributeConstant.IPTYPE, node.getIpAddressType() == -1 ? null : node.getIpAddressType());
			}
			if (attributes.contains(FacilityAttributeConstant.IPPROTOCOLNUMBER)) {
				map.put(FacilityAttributeConstant.IPPROTOCOLNUMBER, node.getIpAddressVersion() == -1 ? null : node.getIpAddressVersion());
			}
			if (attributes.contains(FacilityAttributeConstant.IPNETWORKNUMBER)) {
				map.put(FacilityAttributeConstant.IPNETWORKNUMBER, node.getIpAddressV4());
			}
			if (attributes.contains(FacilityAttributeConstant.IPNETWORKNUMBERV6)) {
				map.put(FacilityAttributeConstant.IPNETWORKNUMBERV6, node.getIpAddressV6());
			}
			
			// ----- OS関連 -----
			if (attributes.contains(FacilityAttributeConstant.NODENAME)) {
				map.put(FacilityAttributeConstant.NODENAME, node.getNodeName());
			}
			if (attributes.contains(FacilityAttributeConstant.OSNAME)) {
				map.put(FacilityAttributeConstant.OSNAME, node.getOsName());
			}
			if (attributes.contains(FacilityAttributeConstant.OSRELEASE)) {
				map.put(FacilityAttributeConstant.OSRELEASE, node.getOsRelease());
			}
			if (attributes.contains(FacilityAttributeConstant.OSVERSION)) {
				map.put(FacilityAttributeConstant.OSVERSION, node.getOsVersion());
			}
			if (attributes.contains(FacilityAttributeConstant.CHARSET)) {
				map.put(FacilityAttributeConstant.CHARSET, node.getCharacterSet());
			}
			
			// ----- 仮想化関連 -----
			if (attributes.contains(FacilityAttributeConstant.VIRTNODETYPE)) {
				map.put(FacilityAttributeConstant.VIRTNODETYPE, node.getVirtualizationNodeType());
			}
			if (attributes.contains(FacilityAttributeConstant.VMMANAGEMENTNODE)) {
				map.put(FacilityAttributeConstant.VMMANAGEMENTNODE, node.getVmManagementNode());
			}
			if (attributes.contains(FacilityAttributeConstant.VMINDEX)) {
				map.put(FacilityAttributeConstant.VMINDEX, node.getVmIndex() == -1 ? null : node.getVmIndex());
			}
			if (attributes.contains(FacilityAttributeConstant.VMNAME)) {
				map.put(FacilityAttributeConstant.VMNAME, node.getVmName());
			}
			if (attributes.contains(FacilityAttributeConstant.VIRTSOLUTION)) {
				map.put(FacilityAttributeConstant.VIRTSOLUTION, node.getVirtualizationSolution());
			}
			if (attributes.contains(FacilityAttributeConstant.VMID)) {
				map.put(FacilityAttributeConstant.VMID, node.getVmId());
			}
			if (attributes.contains(FacilityAttributeConstant.VMUSER)) {
				map.put(FacilityAttributeConstant.VMUSER, node.getVmUser());
			}
			if (attributes.contains(FacilityAttributeConstant.VMUSERPASSWORD)) {
				map.put(FacilityAttributeConstant.VMUSERPASSWORD, node.getVmUserPassword());
			}
			if (attributes.contains(FacilityAttributeConstant.VMPROTOCOL)) {
				map.put(FacilityAttributeConstant.VMPROTOCOL, node.getVmProtocol());
			}
			
			// ----- 管理関連 -----
			if (attributes.contains(FacilityAttributeConstant.MANAGERNAME)) {
				map.put(FacilityAttributeConstant.MANAGERNAME, node.getAdministrator());
			}
			if (attributes.contains(FacilityAttributeConstant.MANAGERCONTACT)) {
				map.put(FacilityAttributeConstant.MANAGERCONTACT, node.getContact());
			}
			
			// ----- ホスト名 -----
			if (attributes.contains(FacilityAttributeConstant.HOST)) {
				map.put(FacilityAttributeConstant.HOST, getHostnameList(node));
			}
			
			// ----- 備考 -----
			if (attributes.contains(FacilityAttributeConstant.NOTE)) {
				map.put(FacilityAttributeConstant.NOTE, getNoteList(node));
			}
		}
		
		m_log.debug("successful in getting a node's data. (facilityId = " + facilityId + ")");
		return map;
	}
	
	/**
	 * 複数のノードのノード情報を取得する。<BR>
	 * 
	 * @param facilityIdList ノードのファシリティIDの配列
	 * @param attributes 取得する項目一覧
	 * @return ノード情報
	 * @throws FacilityNotFoundException 該当するファシリティが存在しなかった場合
	 */
	public static HashMap<String, HashMap<String, Object>>
		getNodeDetail(ArrayList<String> facilityIdList, ArrayList attributes) throws FacilityNotFoundException {
		/** ローカル変数 */
		HashMap<String, HashMap<String, Object>> map = null;
		
		/** メイン処理 */
		m_log.debug("getting node's details...");
		map = new HashMap<String, HashMap<String, Object>>();
		if (facilityIdList != null) {
			for (String facilityId : facilityIdList) {
				map.put(facilityId, getNodeDetail(facilityId, attributes));
			}
		}
		
		m_log.debug("successful in getting node's details.");
		return map;
	}
	
	/**
	 * ノード情報の配列を取得する。<BR>
	 * <PRE>
	 * {
	 *    {facilityId1, facilityName1, description1}, 
	 *    {facilityId2, facilityName2, description2}, 
	 *    ...
	 * }
	 * </PRE>
	 * 
	 * @return ノード情報の配列
	 */
	public static ArrayList<NodeInfo> getNodeList() {
		/** ローカル変数 */
		ArrayList<NodeInfo> nodeList = null;
		Collection nodes = null;
		
		/** メイン処理 */
		m_log.debug("getting a list of all nodes...");
		
		try {
			nodeList = new ArrayList<NodeInfo>();
			nodes = NodeUtil.getLocalHome().findAll();
			if (nodes != null) {
				for (NodeLocal node : (Collection<NodeLocal>)nodes) {
					nodeList.add(FacilityValidator.nodeToArrayList(node));
				}
			}
		} catch (FinderException e) {
			m_log.debug("no node is found.");
		} catch (Exception e) {
			m_log.info("failure to get a list of all nodes...");
			throw new EJBException("failure to get a list of all nodes...");
		}
		
		m_log.debug("successful in getting a list of all nodes.");
		return nodeList;
	}
	
	/**
	 * 検索条件にマッチしたノード情報の配列を取得する。<BR>
	 * <PRE>
	 * {
	 *    {facilityId1, facilityName1, description1}, 
	 *    {facilityId2, facilityName2, description2}, 
	 *    ...
	 * }
	 * </PRE>
	 * 
	 * @param property 検索条件
	 * @return ノード情報配列
	 */
	public static ArrayList<NodeInfo> getFilterNodeList(NodeInfo property) {
		/** ローカル変数 */
		ArrayList<NodeInfo> nodeList = null;
		String facilityId = null;
		String facilityName = null;
		String description = null;
		String ipAddressV4 = null;
		String ipAddressV6 = null;
		String osName = null;
		String osRelease = null;
		String administrator = null;
		String contact = null;
		
		boolean facilityIdFlg = false;
		boolean facilityNameFlg = false;
		boolean descriptionFlg = false;
		boolean ipAddressV4Flg = false;
		boolean ipAddressV6Flg = false;
		boolean osNameFlg = false;
		boolean osReleaseFlg = false;
		boolean administratorFlg = false;
		boolean contactFlg = false;
		
		Collection nodeAll = null;
		FacilityLocal facility = null;
		
		/** メイン処理 */
		m_log.debug("getting a list of nodes by using filter...");
		
		// 検索条件の入力値の取得
		facilityId = property.getFacilityId();
		
		facilityName = property.getFacilityName();
		
		description = property.getDescription();
		
		ipAddressV4 = property.getIpAddressV4();
		
		ipAddressV6 = property.getIpAddressV6();
		
		osName = property.getOsName();
		
		osRelease = property.getOsRelease();
		
		administrator = property.getAdministrator();
		
		contact = property.getContact();
		
		// 検索条件の有効確認
		if (! ObjectValidator.isEmptyString(facilityId)) {
			facilityIdFlg = true;
		}
		if (! ObjectValidator.isEmptyString(facilityName)) {
			facilityNameFlg = true;
		}
		if (! ObjectValidator.isEmptyString(description)) {
			descriptionFlg = true;
		}
		if (! ObjectValidator.isEmptyString(ipAddressV4)) {
			ipAddressV4Flg = true;
		}
		if (! ObjectValidator.isEmptyString(ipAddressV6)) {
			ipAddressV6Flg = true;
		}
		if (! ObjectValidator.isEmptyString(osName)) {
			osNameFlg = true;
		}
		if (! ObjectValidator.isEmptyString(osRelease)) {
			osReleaseFlg = true;
		}
		if (! ObjectValidator.isEmptyString(administrator)) {
			administratorFlg = true;
		}
		if (! ObjectValidator.isEmptyString(contact)) {
			contactFlg = true;
		}
		
		try {
			nodeAll = NodeUtil.getLocalHome().findAll();
		} catch (FinderException e) {
			m_log.debug("no node is found.");
		} catch (Exception e) {
			m_log.info("failure to get a list of nodes by using filter.", e);
			throw new EJBException("failure to get a list of nodes by using filter.", e);
		}
		
		nodeList = new ArrayList<NodeInfo>();
		if (nodeAll != null) {
			// 文字列が部分一致した場合、マッチしたノードとする
			for (NodeLocal node : (Collection<NodeLocal>)nodeAll) {
				facility = node.getFacility();
				
				if (facilityIdFlg && facility.getFacilityId().indexOf(facilityId) == -1) {
					continue;
				}
				if (facilityNameFlg && facility.getFacilityName().indexOf(facilityName) == -1) {
					continue;
				}
				if (descriptionFlg && facility.getDescription().indexOf(description) == -1) {
					continue;
				}
				if (ipAddressV4Flg && node.getIpAddressV4().indexOf(ipAddressV4) == -1) {
					continue;
				}
				if (ipAddressV6Flg && node.getIpAddressV6().indexOf(ipAddressV6) == -1) {
					continue;
				}
				if (osNameFlg && node.getOsName().indexOf(osName) == -1) {
					continue;
				}
				if (osReleaseFlg && node.getOsRelease().indexOf(osRelease) == -1) {
					continue;
				}
				if (administratorFlg && node.getAdministrator().indexOf(administrator) == -1) {
					continue;
				}
				if (contactFlg && node.getContact().indexOf(contact) == -1) {
					continue;
				}
				
				nodeList.add(FacilityValidator.nodeToArrayList(node));
			}
		}
		
		m_log.debug("successful in getting a list of nodes by using filter.");
		return nodeList;
	}
	
	/**
	 * スコープ配下にあるノード情報の一覧を取得する。<BR>
	 * 
	 * @param parentFacilityId スコープのファシリティID
	 * @param level 取得する階層数
	 * @return ノード情報の一覧
	 */
	public static ArrayList<NodeInfo> getNodeList(String parentFacilityId, int level) {
		/** ローカル変数 */
		ArrayList<NodeInfo> nodes = null;
		ArrayList<FacilityLocal> facilities = null;
		NodeLocal node = null;
		
		/** メイン処理 */
		facilities = getFacilityList(parentFacilityId, level);
		
		nodes = new ArrayList<NodeInfo>();
		if (facilities != null) {
			for (FacilityLocal facility : facilities) {
				// ノードの場合、配列に追加
				if (facility.isNode()) {
					node = facility.getNode();
					nodes.add(FacilityValidator.nodeToArrayList(node));
				}
			}
		}
		
		return nodes;
	}
	
	/**
	 * スコープ配下にあるファシリティIDの一覧を取得してソートする。<BR>
	 * parentFacilityIdがノードの場合は、そのノードのファシリティIDを返す。
	 * 
	 * @param parentFacilityId スコープのファシリティID
	 * @param level 取得する階層数
	 * @param sort ファシリティIDをソートする場合はtrue, ソートしない場合はfalse
	 * @return ファシリティIDの配列
	 */
	public static ArrayList<String> getFacilityIdList(String parentFacilityId, int level, boolean sort) {
		/** ローカル変数 */
		ArrayList<String> facilityIds = null;
		ArrayList<FacilityLocal> facilities = null;
		
		/** メイン処理 */
		facilities = getFacilityList(parentFacilityId, level);
		
		facilityIds = new ArrayList<String>();
		if (facilities != null) {
			for (FacilityLocal facility : facilities) {
				facilityIds.add(facility.getFacilityId());
			}
		}
		
		if (sort) {
			ListSorter sorter = new ListSorter();
			sorter.sort(facilityIds);
		}
		
		return facilityIds;
	}
	
	/**
	 * スコープ配下にあるファシリティIDの一覧を有効/無効を指定して取得してソートする。<BR>
	 * parentFacilityIdが有効ノードの場合は、ノードのファシリティIDを返す。
	 * parentFacilityIdが無効ノードの場合は、空のArrayListを返す。
	 * 
	 * @param parentFacilityId スコープのファシリティID
	 * @param level 取得する階層数
	 * @param sort ファシリティIDをソートする場合はtrue, ソートしない場合はfalse
	 * @param validFlg 有効/無効の指定
	 * @return ファシリティIDの配列
	 */
	public static ArrayList<String> getNodeFacilityIdList(String parentFacilityId, int level, boolean sort, Boolean validFlg) {
		/** ローカル変数 */
		ArrayList<String> facilityIdList = null;
		ArrayList<FacilityLocal> facilityList = null;
		
		/** メイン処理 */
		facilityList = getFacilityList(parentFacilityId, level);
		
		facilityIdList = new ArrayList<String>();
		if (facilityList != null) {
			for (FacilityLocal facility : facilityList) {
				if (facility.isNode()) {
					if (validFlg == null) {
						facilityIdList.add(facility.getFacilityId());
					} else if (validFlg == facility.isValid()) {
						facilityIdList.add(facility.getFacilityId());
					}
				}
			}
		}
		
		if (sort) {
			ListSorter sorter = new ListSorter();
			sorter.sort(facilityIdList);
		}
		
		return facilityIdList;
	}
	
	/**
	 * 全てのノードのファシリティIDの一覧を取得する。<BR>
	 * 
	 * @param sort ファシリティIDをソートする場合はtrue, ソートしない場合はfalse
	 * @return ファシリティIDの配列
	 */
	public static ArrayList<String> getNodeFacilityIdList(boolean sort) {
		/** ローカル変数 */
		ArrayList<String> facilityIdList = null;
		Collection nodes = null;
		
		/** メイン処理 */
		try {
			facilityIdList = new ArrayList<String>();
			
			nodes = NodeUtil.getLocalHome().findAll();
			
			if (nodes != null) {
				for (NodeLocal node : (Collection<NodeLocal>)nodes) {
					facilityIdList.add(node.getFacilityId());
				}
			}
			
			if (sort) {
				ListSorter sorter = new ListSorter();
				sorter.sort(facilityIdList);
			}
		} catch (FinderException e) {
			m_log.debug("no node is found.");
		} catch (Exception e) {
			m_log.info("failure to get a list of all node.", e);
			throw new EJBException("failure to get a list of all node.", e);
		}
		
		return facilityIdList;
	}
	
	/**
	 * ホスト名とIPv4アドレスを指定して、該当するノードのファシリティIDの一覧を取得する。<BR>
	 * 
	 * @param hostname ホスト名
	 * @param ipAddressV4 IPv4アドレス
	 * @return ファシリティIDの配列
	 */
	public static ArrayList<String> getFacilityIdList(String hostname, String ipAddressV4) {

		m_log.debug("getFacilityIdList() start : hostname = " + hostname + ", ipAddressV4 = " + ipAddressV4);
		
		/** ローカル変数 */
		ArrayList<String> facilityIds = null;
		Collection<NodeLocal> nodes = null;
		
		/** メイン処理 */
		try {
			facilityIds = new ArrayList<String>();
			
			// hostname変数のNodeプロパティのnodename(必須項目)をLowerCaseで検索
			if (hostname != null && ipAddressV4 != null) {
				nodes = (Collection<NodeLocal>)NodeUtil.getLocalHome().findByNodename(hostname.toLowerCase());
				if (nodes != null){
					for (NodeLocal node : (Collection<NodeLocal>)nodes){
						
						m_log.debug("getFacilityIdList() List " +
								" FacilityId = " + node.getFacilityId() + 
								" NodeName = " + node.getNodeName() + 
								" IpAddressV4 = " + node.getIpAddressV4() + 
								" IpAddressV6 = " + node.getIpAddressV6());

						// IPv4とマッチ
						if(node.getIpAddressV4() != null && node.getIpAddressV4().equals(ipAddressV4)){
							m_log.debug("getFacilityIdList() hit facilityId = " + node.getFacilityId());
							facilityIds.add(node.getFacilityId());
						}
					}
				}
			}

		} catch (FinderException e) {
			m_log.debug("no node is found.");
		} catch (Exception e) {
			m_log.info("failure to get a list of nodes by using hostname and IPv4 address filter.", e);
			throw new EJBException("failure to get a list of nodes by using hostname and IPv4 address filter.", e);
		}
		
		// Debugが有効の場合のみ取得したIPアドレスを表示させる
		if(m_log.isDebugEnabled()){
			for (Iterator iter = facilityIds.iterator(); iter.hasNext();) {
				String facilityId = (String) iter.next();
				m_log.debug("getFacilityIdList() hostname = " + hostname 
						+ ", ipAddressV4 = " + ipAddressV4 + " has " + facilityId);
			}
		}
		
		return facilityIds;
	}
	
	/**
	 * 検索条件にマッチするノードのファシリティIDの一覧を取得する。<BR>
	 * 
	 * @param condition 検索条件
	 * @return ファシリティIDの配列
	 */
	public static ArrayList<String> getNodeFacilityIdListByCondition(HashMap condition) {
		/** ローカル変数 */
		ArrayList<NodeLocal> nodes = null;
		ArrayList<String> facilityIds = null;
		
		/** メイン処理 */
		nodes = getNodeByCondition(condition);
		facilityIds = new ArrayList<String>();
		if (nodes != null) {
			for (NodeLocal node : nodes) {
				facilityIds.add(node.getFacilityId());
			}
		}
		
		return facilityIds;
	}
	
	/**
	 * 検索条件にマッチするノードのファシリティインスタンスの一覧を取得する。<BR>
	 * 
	 * @param condition 検索条件
	 * @return ファシリティインスタンスの配列
	 */
	public static ArrayList<NodeLocal> getNodeByCondition(HashMap condition) {
		/** ローカル変数 */
		ArrayList<NodeLocal> nodes = null;
		String value = null;
		String valueTarget = null;
		Collection nodeAll = null;
		
		/** メイン処理 */
		m_log.debug("getting nodes by using filter...");
		
		// ノードの一覧を取得する
		try {
			nodeAll = NodeUtil.getLocalHome().findAll();
		} catch (FinderException e) {
			m_log.debug("no node is found.");
		} catch (Exception e) {
			m_log.info("failure to get nodes by using filter.", e);
			throw new EJBException("failure to get nodes by using filter.", e);
		}
		
		// 該当するノードの一覧を生成する
		nodes = new ArrayList<NodeLocal>();
		if (nodeAll != null) {
			for (NodeLocal node : (Collection<NodeLocal>)nodeAll) {
				if (condition != null) {
					boolean matchFlg = true;
					
					for (String attribute : (Collection<String>)condition.keySet()) {
						if (ObjectValidator.isEmptyString(attribute)) {
							continue;
						}
						value = (String)condition.get(attribute);
						if (value == null) {
							continue;
						}
						
						valueTarget = null;
						
						// ファシリティ関連
						if (attribute.compareTo(FacilityAttributeConstant.FACILITYID) == 0) {
							valueTarget = node.getFacilityId();
						} else if(attribute.compareTo(FacilityAttributeConstant.CN) == 0) {
							valueTarget = node.getFacility().getFacilityName();
						} else if(attribute.compareTo(FacilityAttributeConstant.DESCRIPTION) == 0) {
							valueTarget = node.getFacility().getDescription();
						} else if(attribute.compareTo(FacilityAttributeConstant.SORT_VALUE) == 0){
							valueTarget = node.getFacility().getDisplaySortOrder().toString();
						} else if(attribute.compareTo(FacilityAttributeConstant.VALID) == 0){
							// 旧格納データベースであるLDAPの物理格納形式に変換（VMオプション対応）
							valueTarget = node.getFacility().getValid() == ValidConstant.TYPE_VALID ? new Boolean(true).toString().toUpperCase() : new Boolean(false).toString().toUpperCase();
						} else if(attribute.compareTo(FacilityAttributeConstant.CREATORSNAME) == 0) {
							valueTarget = node.getFacility().getCreateUserId();
						} else if(attribute.compareTo(FacilityAttributeConstant.CREATETIMESTAMP) == 0) {
							valueTarget = node.getFacility().getCreateDatetime().toString();
						} else if(attribute.compareTo(FacilityAttributeConstant.MODIFIERSNAME) == 0) {
							valueTarget = node.getFacility().getModifyUserId();
						} else if(attribute.compareTo(FacilityAttributeConstant.MODIFYTIMESTAMP) == 0) {
							valueTarget = node.getFacility().getModifyDatetime().toString();
						}
						
						// ノード関連
						else if(attribute.compareTo(FacilityAttributeConstant.PLATFORM) == 0){
							valueTarget = node.getPlatformFamily();
						} else if(attribute.compareTo(FacilityAttributeConstant.MACHINE) == 0){
							valueTarget = node.getHardwareType();
						} else if(attribute.compareTo(FacilityAttributeConstant.ICONIMAGE) == 0){
							valueTarget = node.getIconImage();
						}
						
						// SNMP関連
						else if(attribute.compareTo(FacilityAttributeConstant.SNMPPORT) == 0){
							valueTarget = node.getSnmpPort().toString();
						} else if(attribute.compareTo(FacilityAttributeConstant.SNMPCOMMUNITY) == 0){
							valueTarget = node.getSnmpCommunity();
						} else if(attribute.compareTo(FacilityAttributeConstant.SNMPVERSION) == 0){
							valueTarget = node.getSnmpVersion();
						} else if(attribute.compareTo(FacilityAttributeConstant.SNMPTIMEOUT) == 0){
							valueTarget = node.getSnmpTimeout().toString();
						} else if(attribute.compareTo(FacilityAttributeConstant.SNMPRETRIES) == 0){
							valueTarget = node.getSnmpRetryCount().toString();
						} else if(attribute.compareTo(FacilityAttributeConstant.SNMPPROXY) == 0){
							valueTarget = node.getSnmpProxy();
						}
						
						// WBEM関連
						else if(attribute.compareTo(FacilityAttributeConstant.WBEMUSER) == 0){
							valueTarget = node.getWbemUser();
						} else if(attribute.compareTo(FacilityAttributeConstant.WBEMUSERPASSWORD) == 0){
							valueTarget = node.getWbemUserPassword();
						} else if(attribute.compareTo(FacilityAttributeConstant.WBEMPORT) == 0){
							valueTarget = node.getWbemPort().toString();
						} else if(attribute.compareTo(FacilityAttributeConstant.WBEMPROTOCOL) == 0){
							valueTarget = node.getWbemProtocol();
						} else if(attribute.compareTo(FacilityAttributeConstant.WBEMTIMEOUT) == 0){
							valueTarget = node.getWbemTimeout().toString();
						} else if(attribute.compareTo(FacilityAttributeConstant.WBEMRETRIES) == 0){
							valueTarget = node.getWbemRetryCount().toString();
						}
						
						// IPアドレス関連
						else if(attribute.compareTo(FacilityAttributeConstant.DHCPCLIENT) == 0){
							// 旧格納データベースであるLDAPの物理格納形式に変換（VMオプション対応）
							valueTarget = node.getDhcpClient() == ValidConstant.TYPE_VALID ? new Boolean(true).toString().toUpperCase() : new Boolean(false).toString().toUpperCase();
						} else if(attribute.compareTo(FacilityAttributeConstant.IPTYPE) == 0){
							valueTarget = node.getIpAddressType().toString();
						} else if(attribute.compareTo(FacilityAttributeConstant.IPPROTOCOLNUMBER) == 0){
							valueTarget = node.getIpAddressVersion().toString();
						} else if(attribute.compareTo(FacilityAttributeConstant.IPNETWORKNUMBER) == 0){
							valueTarget = node.getIpAddressV4();
						} else if(attribute.compareTo(FacilityAttributeConstant.IPNETWORKNUMBERV6) == 0){
							valueTarget = node.getIpAddressV6();
						}
						
						// OS関連
						else if(attribute.compareTo(FacilityAttributeConstant.NODENAME) == 0){
							valueTarget = node.getNodeName();
						} else if(attribute.compareTo(FacilityAttributeConstant.OSNAME) == 0){
							valueTarget = node.getOsName();
						} else if(attribute.compareTo(FacilityAttributeConstant.OSRELEASE) == 0){
							valueTarget = node.getOsRelease();
						} else if(attribute.compareTo(FacilityAttributeConstant.OSVERSION) == 0){
							valueTarget = node.getOsVersion();
						} else if(attribute.compareTo(FacilityAttributeConstant.CHARSET) == 0){
							valueTarget = node.getCharacterSet();
						}
						
						// 仮想化関連
						else if(attribute.compareTo(FacilityAttributeConstant.VIRTNODETYPE) == 0){
							valueTarget = node.getVirtualizationNodeType();
						} else if(attribute.compareTo(FacilityAttributeConstant.VMMANAGEMENTNODE) == 0){
							valueTarget = node.getVmManagementNode();
						} else if(attribute.compareTo(FacilityAttributeConstant.VMINDEX) == 0){
							valueTarget = node.getVmIndex().toString();
						} else if(attribute.compareTo(FacilityAttributeConstant.VMNAME) == 0){
							valueTarget = node.getVmName();
						} else if(attribute.compareTo(FacilityAttributeConstant.VIRTSOLUTION) == 0){
							valueTarget = node.getVirtualizationSolution();
						} else if(attribute.compareTo(FacilityAttributeConstant.VMID) == 0){
							valueTarget = node.getVmId();
						} else if(attribute.compareTo(FacilityAttributeConstant.VMUSER) == 0){
							valueTarget = node.getVmUser();
						} else if(attribute.compareTo(FacilityAttributeConstant.VMUSERPASSWORD) == 0){
							valueTarget = node.getVmUserPassword();
						} else if(attribute.compareTo(FacilityAttributeConstant.VMPROTOCOL) == 0){
							valueTarget = node.getVmProtocol();
						}
						
						// 管理関連
						else if(attribute.compareTo(FacilityAttributeConstant.MANAGERNAME) == 0){
							valueTarget = node.getAdministrator();
						} else if(attribute.compareTo(FacilityAttributeConstant.MANAGERCONTACT) == 0){
							valueTarget = node.getContact();
						}
						
						else {
							m_log.info("a filter's attribute is invalid. (attribute = " + attribute + ")");
							throw new EJBException("a filter's attribute is invalid. (attribute = " + attribute + ")");
						}
						
						// 文字列としての値の比較
						if (valueTarget == null || value.compareTo(valueTarget) != 0) {
							m_log.debug("a node's attribute is not equal. (attribute = " + attribute + ", value = " + valueTarget + ", compareValue = " + value + ")");
							matchFlg = false;
							break;
						}
					}
					
					if (matchFlg) {
						m_log.debug("a node is matched. (facilityId = " + node.getFacilityId() + ")");
						nodes.add(node);
					}
				}
			}
		}
		
		m_log.debug("successful in getting nodes by using filter.");
		return nodes;
	}
	
	/**
	 * 特定のスコープ配下の直下に属するファシリティの一覧を取得する。<BR>
	 * 
	 * @param parentFacilityId スコープのファシリティID
	 * @return ファシリティインスタンスの配列
	 */
	public static ArrayList<ScopeInfo> getFacilityListAssignedScope(String parentFacilityId) {
		/** ローカル変数 */
		ArrayList<ScopeInfo> scopeList = null;
		FacilityLocal scope = null;
		
		/** メイン処理 */
		m_log.debug("getting a list of facilities under a scope...");
		
		try {
			scopeList = new ArrayList<ScopeInfo>();
			
			if (ObjectValidator.isEmptyString(parentFacilityId)) {
				// コンポジットアイテムが選択された場合
				for (FacilityLocal rootScope : getRootScopeList()) {
					scopeList.add(FacilityValidator.scopeToArrayList(rootScope));
				}
			} else {
				// スコープが選択された場合
				scope = FacilityUtil.getLocalHome().findByPrimaryKey(parentFacilityId);
				
				if (scope.getFacilityRelationAsParent() != null) {
					for (FacilityRelationLocal relation : (Collection<FacilityRelationLocal>)scope.getFacilityRelationAsParent()) {
						// スコープ配下に属する全てのファシリティを追加
						scopeList.add(FacilityValidator.scopeToArrayList(relation.getChildFacility()));
					}
				}
			}
		} catch (FinderException e) {
			m_log.debug("no facility is found.");
		} catch (Exception e) {
			m_log.info("failure to get a list of facilities under a scope. (parentFacilityId = " + parentFacilityId + ")");
			throw new EJBException("failure to get a list of facilities under a scope. (parentFacilityId = " + parentFacilityId + ")");
		}
		
		m_log.debug("successful in getting a list of facilities under a scope...");
		return scopeList;
	}
	
	/**
	 * ノードが属するスコープのパス名の一覧を取得する。<BR>
	 * 
	 * @param facilityId ノードのファシリティID
	 * @return スコープのパス名の配列
	 */
	public static ArrayList<String> getNodeScopeList(String facilityId	) {
		/** ローカル変数 */
		ArrayList<String> scopePathList = null;
		NodeLocal node = null;
		
		/** メイン処理 */
		m_log.debug("getting scope paths of a node...");
		
		try {
			node = NodeUtil.getLocalHome().findByPrimaryKey(facilityId);
		} catch (Exception e) {
			m_log.info("failure to get scope paths of a node. (facilityId = " + facilityId + ")", e);
			throw new EJBException("failure to get scope paths of a node. (facilityId = " + facilityId + ")", e);
		}
		
		scopePathList = new ArrayList<String>();
		if (node.getFacility().getFacilityRelationAsChild() != null) {
			for (FacilityRelationLocal relation : (Collection<FacilityRelationLocal>)node.getFacility().getFacilityRelationAsChild()) {
				scopePathList.add(getNodeScopePathRecursive(relation.getParentFacility(), null));
			}
		}
		
		m_log.debug("successful in getting scope paths of a node.");
		return scopePathList;
	}
	
	/**
	 * ファシリティのパス名を取得する。<BR>
	 * 
	 * @param parentFacilityId 起点となるファシリティのファシリティID
	 * @param facilityId パスを取得するファシリティID
	 * @return ファシリティのパス名
	 */
	public static String getNodeScopePath(String parentFacilityId, String facilityId) {
		/** ローカル変数 */
		FacilityLocal facility = null;
		FacilityLocal parentFacility = null;
		String path = null;
		
		/** メイン処理 */
		m_log.debug("getting a scope path of a facility's relation...");
		
		try {
			path = "";
			
			if (! ObjectValidator.isEmptyString(facilityId)) {
				try {
					facility = FacilityUtil.getLocalHome().findByPrimaryKey(facilityId);
				} catch (FinderException e) {
					// すでにファシリティが削除されていた場合はから文字を返す
					m_log.debug("this facility is already deleted. (facilityId = " + facilityId + ")");
				}
				
				if (facility != null) {
					if (! ObjectValidator.isEmptyString(parentFacilityId)) {
						parentFacility = FacilityUtil.getLocalHome().findByPrimaryKey(parentFacilityId);
					}
					path = getNodeScopePathRecursive(facility, parentFacility);
					if (ObjectValidator.isEmptyString(path)) {
						path = SEPARATOR;
					}
				}
			}
		} catch (Exception e) {
			m_log.info("failure to get a scope path of a facility's relation. (parentFacilityId = " + parentFacilityId + ", facilityId = " + facilityId + ")", e);
			throw new EJBException("failure to get a scope path of a facility's relation. (parentFacilityId = " + parentFacilityId + ", facilityId = " + facilityId + ")", e);
		}
		
		m_log.debug("successful in getting a scope path of a facility's relation.");
		return path;
	}
	
	/**
	 * ファシリティ関連インスタンスに属するファシリティの相対パス名を取得する。<BR>
	 * スコープの場合は「facilityName1>facilityName2>facilityName3」、ノードの場合は「facilityName」となる。<BR>
	 *  
	 * @param facility パスを取得するファシリティインスタンス
	 * @param parentFacility 相対パスを取得する場合は起点となるファシリティインスタンス, 絶対パスを取得する場合はnull
	 * @return ファシリティのパス名
	 */
	private static String getNodeScopePathRecursive(FacilityLocal facility, FacilityLocal parentFacility) {
		/** ローカル変数 */
		String path = null;
		
		/** メイン処理 */
		if (facility.isNode()) {
			// ノードの場合、ファシリティ名を返す
			return facility.getFacilityName();
		}
		
		if (parentFacility != null && facility.getFacilityId().compareTo(parentFacility.getFacilityId()) == 0) {
			// 起点となるファシリティと一致していた場合、区切り文字だけ返す
			return "";
		}
		
		if (facility.getFacilityRelationAsChild() == null || facility.getFacilityRelationAsChild().size() == 0) {
			// ルートスコープの場合
			if (parentFacility == null) {
				path = facility.getFacilityName() + SEPARATOR;
			}
		} else {
			// 再帰的にスコープのパス名を生成する
			for (FacilityRelationLocal relation : (Collection<FacilityRelationLocal>)facility.getFacilityRelationAsChild()) {
				if (parentFacility == null) {
					// 絶対パス名を取得する場合
					path = getNodeScopePathRecursive(relation.getParentFacility(), parentFacility) + facility.getFacilityName() + SEPARATOR;
				} else if (facility.getFacilityId().compareTo(parentFacility.getFacilityId()) != 0) {
					// 相対パス名を取得する場合
					path = getNodeScopePathRecursive(relation.getParentFacility(), parentFacility) + facility.getFacilityName() + SEPARATOR;
				}
				
				// スコープ-スコープ間の関連は単一であるため、ループを抜ける
				break;
			}
		}
		
		return path;
	}
	
	/**
	 * ファシリティの木構造を取得する。<BR>
	 * 
	 * @param locale ロケール情報
	 * @param scopeOnly スコープのみの場合はtrue, 全てのファシリティの場合はfalse
	 * @param validFlg 有効/無効（nullの場合は全てのノード）
	 * @return ファシリティの木構造
	 */
	public static FacilityTreeItem getFacilityTree(Locale locale, boolean scopeOnly, Boolean validFlg) {
		/** ローカル変数 */
		FacilityTreeItem rootTree = null;
		FacilityInfo rootInfo = null;
		FacilityInfo scopeInfo = null;
		FacilityTreeItem scopeTree = null;
		
		/** メイン処理 */
		m_log.debug("getting tree data of facilities...");
		
		// 更新排他制御（更新がCOMMITされる前にTOPICからのアクセスが発生する可能性があるため）
		AccessLock.lock(AccessLock.REPOSITORY);
		
		// キャッシュに情報が存在すれば、それを返す
		rootTree = getFacilityTreeOnCache(locale, scopeOnly, validFlg);
		if (rootTree != null) {
			return rootTree;
		}
		
		// 木構造最上位インスタンスの生成
		rootInfo = new FacilityInfo();
		rootInfo.setFacilityName(FacilityConstant.STRING_COMPOSITE);
		rootInfo.setFacilityType(FacilityConstant.TYPE_COMPOSITE);
		rootTree = new FacilityTreeItem(null, rootInfo);
		
		// コンポジットアイテムの生成
		scopeInfo = new FacilityInfo();
		scopeInfo.setFacilityName(Messages.getString("scope", locale));
		scopeInfo.setFacilityType(FacilityConstant.TYPE_COMPOSITE);
		scopeTree = new FacilityTreeItem(rootTree, scopeInfo);
		
		try {
			for (FacilityLocal facility : getRootScopeList()) {
				getFacilityTreeRecursive(scopeTree, facility, scopeOnly, validFlg);
			}
		} catch (Exception e) {
			m_log.info("failure to get a tree data of facilities.", e);
		}
		
		// 初期値をキャッシュに登録（次回から高速化に初期値を取得）
		m_log.info("adding a facility's tree to cache. (locale = " + locale + ", scopeOnly = " + scopeOnly + ", validFlg = " + validFlg + ")");
		if (scopeOnly) {
			synchronized (cachedFacilityTreeScope) {
				cachedFacilityTreeScope.get(locale).put(validFlg, rootTree);
			}
		} else {
			synchronized (cachedFacilityTree) {
				cachedFacilityTree.get(locale).put(validFlg, rootTree);
			}
		}
		
		m_log.debug("successful in getting tree data of facilities.");
		return rootTree;
	}
	
	/**
	 * キャッシュ上に存在するファシリティの木構造を取得する。<BR>
	 * キャッシュ上に存在しなければ、nullを返す。
	 * 
	 * @param locale ロケール情報
	 * @param scopeOnly スコープのみの場合はtrue, 全てのファシリティの場合はfalse
	 * @param validFlg 有効/無効（nullの場合は全てのノード）
	 * @return ファシリティの木構造
	 */
	private static FacilityTreeItem getFacilityTreeOnCache(Locale locale, boolean scopeOnly, Boolean validFlg) {
		// キャッシュに情報が存在すれば、それを返す
		if (scopeOnly) {
			if (cachedFacilityTreeScope.containsKey(locale)) {
				if (cachedFacilityTreeScope.get(locale).containsKey(validFlg)) {
					m_log.debug("using a facility's tree in cache. (locale = " + locale + ", scopeOnly = " + scopeOnly + ", validFlg = " + validFlg + ")");
					return cachedFacilityTreeScope.get(locale).get(validFlg);
				}
			} else {
				m_log.info("adding a facility's tree to cache. (locale = " + locale + ")");
				cachedFacilityTreeScope.put(locale, new HashMap<Boolean, FacilityTreeItem>());
			}
		} else {
			if (cachedFacilityTree.containsKey(locale)) {
				if (cachedFacilityTree.get(locale).containsKey(validFlg)) {
					m_log.debug("using a facility's tree in cache. (locale = " + locale + ", scopeOnly = " + scopeOnly + ", validFlg = " + validFlg + ")");
					return cachedFacilityTree.get(locale).get(validFlg);
				}
			} else {
				m_log.info("adding a facility's tree to cache. (locale = " + locale + ")");
				cachedFacilityTree.put(locale, new HashMap<Boolean, FacilityTreeItem>());
			}
		}
		
		return null;
	}
	
	/**
	 * ファシリティの木構造を再帰的に取得する。<BR>
	 * 
	 * @param parentTree 親となるファシリティの木構造
	 * @param facility 格納するファシリティインスタンス
	 * @param scopeOnly スコープのみの場合はtrue, 全てのファシリティの場合はfalse
	 * @param validFlg 有効/無効（nullの場合は全てのノード）
	 */
	private static void getFacilityTreeRecursive(FacilityTreeItem parentTree, FacilityLocal facility, boolean scopeOnly, Boolean validFlg) {
		/** ローカル変数 */
		FacilityInfo facilityInfo = null;
		FacilityTreeItem childTree = null;
		
		/** メイン処理 */
		// 木構造への格納必要性の確認
		if (facility.isNode()) {
			// スコープのみの場合、ノードは格納しない
			if (scopeOnly) {
				return;
			}
			// 有効/無効の確認（nullならば必ず格納）
			if (validFlg != null && facility.isValid() != validFlg) {
				return;
			}
		}
		
		// ファシリティの格納
		facilityInfo = new FacilityInfo();
		facilityInfo.setFacilityId(facility.getFacilityId());
		facilityInfo.setFacilityName(facility.getFacilityName());
		facilityInfo.setFacilityType(facility.getFacilityType());
		facilityInfo.setDisplaySortOrder(facility.getDisplaySortOrder());
		facilityInfo.setBuiltInFlg(isBuildinScope(facility));
		facilityInfo.setValid(facility.isValid());
		
		childTree = new FacilityTreeItem(parentTree, facilityInfo);
		
		// 再帰的にファシリティを格納する
		if (facility.isScope() && facility.getFacilityRelationAsParent() != null) {
			for (FacilityRelationLocal facilityRelation : (Collection<FacilityRelationLocal>)facility.getFacilityRelationAsParent()) {
				getFacilityTreeRecursive(childTree, facilityRelation.getChildFacility(), scopeOnly, validFlg);
			}
		}
	}
	
	/**
	 * ルートとなるスコープの一覧を取得する。<BR>
	 * 
	 * @return ファシリティインスタンスの配列
	 * @throws FinderException
	 * @throws NamingException
	 */
	private static ArrayList<FacilityLocal> getRootScopeList() throws FinderException, NamingException {
		/** ローカル変数 */
		ArrayList<FacilityLocal> scopeList = null;
		Collection facilityAll = null;
		
		/** メイン処理 */
		facilityAll = FacilityUtil.getLocalHome().findAll();
		
		scopeList = new ArrayList<FacilityLocal>();
		if (facilityAll != null) {
			for (FacilityLocal facility : (Collection<FacilityLocal>)facilityAll) {
				if (facility.isScope()) {
					if (facility.getFacilityRelationAsChild() == null || facility.getFacilityRelationAsChild().size() == 0) {
						scopeList.add(facility);
					}
				}
			}
		}
		
		return scopeList;
	}
	
	/**
	 * スコープ配下にあるファシリティの一覧を取得する。<BR>
	 * 引数がノードの場合は、そのノードのファシリティIDを返す。
	 * 
	 * @param parentFacilityId スコープのファシリティID
	 * @param level 取得する階層数
	 * @return ファシリティの配列
	 */
	private static ArrayList<FacilityLocal> getFacilityList(String parentFacilityId, int level) {
		/** ローカル変数 */
		ArrayList<FacilityLocal> facilityList = null;
		FacilityLocal parentFacility = null;
		
		/** メイン処理 */
		m_log.debug("getting facilities under a scope. (scopeFacilityId = " + parentFacilityId + ")");
		
		try {
			if (ObjectValidator.isEmptyString(parentFacilityId)) {
				if (level == RepositoryControllerBean.ONE_LEVEL) { 
					facilityList = getRootScopeList();
				} else {
					facilityList = new ArrayList<FacilityLocal>();
					for (FacilityLocal rootScope : getRootScopeList()) {
						getFacilityListRecursive(rootScope, level > 1 ? level - 1 : level, facilityList);
					}
				}
			} else {
				facilityList = new ArrayList<FacilityLocal>();
				parentFacility = FacilityUtil.getLocalHome().findByPrimaryKey(parentFacilityId);
				if (parentFacility.isNode()) {
					facilityList = new ArrayList<FacilityLocal> ();
					facilityList.add(parentFacility);
					return facilityList;
				}
				getFacilityListRecursive(parentFacility, level, facilityList);
			}
		} catch (Exception e) {
			m_log.info("failure to get facilities under a scope (scopeFacilityId = " + parentFacilityId + ")", e);
			throw new EJBException("failure to get facilities under a scope (scopeFacilityId = " + parentFacilityId + ")", e);
		}
		
		m_log.debug("successful in getting facilities under a scope. (scopeFacilityId = " + parentFacilityId + ")");
		return facilityList;
	}
	
	/**
	 * スコープ配下にあるファシリティの一覧を取得する。<BR>
	 * 
	 * @param parentFacility スコープのファシリティインスタンス
	 * @param level 取得する階層数
	 * @param facilityList 格納先となるファシリティの配列
	 * @throws FinderException
	 * @throws NamingException
	 */
	private static void getFacilityListRecursive(FacilityLocal parentFacility, int level, ArrayList<FacilityLocal> facilityList) throws FinderException, NamingException {
		/** ローカル変数 */
		FacilityLocal childFacility = null;
		boolean recursive = false;
		int nextLevel = 0;
		
		/** メイン処理 */
		// 階層数による再帰的処理の必要性の確認
		if (level == RepositoryControllerBean.ALL) {
			recursive = true;
			nextLevel = RepositoryControllerBean.ALL;
		} else if (level > 1) {
			recursive = true;
			nextLevel = level - 1;
		}
		
		// 再帰的にファシリティを配列に追加する
		if (parentFacility.getFacilityRelationAsParent() != null) {
			for (FacilityRelationLocal relation : (Collection<FacilityRelationLocal>)parentFacility.getFacilityRelationAsParent()) {
				childFacility = relation.getChildFacility();
				
				if (recursive && childFacility.isScope()) {
					getFacilityListRecursive(childFacility, nextLevel, facilityList);
				}
				if (! facilityList.contains(childFacility)) {
					facilityList.add(childFacility);
				}
			}
		}
	}
	
	/**
	 * ファシリティインスタンスが組み込みスコープであるかどうかを確認する。<BR>
	 * 
	 * @param facility ファシリティインスタンス
	 * @return 組み込みスコープの場合はtrue, それ以外はfalse
	 */
	private static boolean isBuildinScope(FacilityLocal facility) {
		/** メイン処理 */
		if (facility == null || ! facility.isScope()) {
			return false;
		}
		
		if (facility.getFacilityId().compareTo(FacilityTreeAttributeConstant.INTERNAL_SCOPE) == 0) {
			return true;
		} else if (facility.getFacilityId().compareTo(FacilityTreeAttributeConstant.REGISTEREFD_SCOPE) == 0) {
			return true;
		} else if (facility.getFacilityId().compareTo(FacilityTreeAttributeConstant.UNREGISTEREFD_SCOPE) == 0) { 
			return true;
		} else {
			return false;
		}
	}
	
	/**
	 * ファシリティの表示ソート順位を取得する。<BR>
	 * 
	 * @param facilityId ファシリティID
	 * @return 表示ソート順位
	 * @throws NamingException
	 */
	public static int getDisplaySortOrder(String facilityId) throws NamingException {
		/** ローカル変数 */
		FacilityLocal facility = null;
		
		/** メイン処理 */
		m_log.debug("getting display sort order of a facility...");
		
		try {
			facility = FacilityUtil.getLocalHome().findByPrimaryKey(facilityId);
		} catch (Exception e) {
			m_log.info("failure to get display sort order of a facility. (facilityId = " + facilityId + ")", e);
			throw new EJBException("failure to get display sort order of a facility. (facilityId = " + facilityId + ")", e);
		}
		
		m_log.debug("successful in getting display's sort order of a facility. (facilityId = " + facilityId + ")");
		return facility.getDisplaySortOrder();
	}
	
	/**
	 * ファシリティがノードかどうかを確認する。<BR>
	 * 
	 * @param facilityId ファシリティID
	 * @return ノードの場合はtrue, それ以外の場合はfalse
	 * @throws FacilityNotFoundException 
	 */
	public static boolean isNode(String facilityId) throws FacilityNotFoundException {
		/** ローカル変数 */
		FacilityLocal facility = null;
		
		/** メイン処理 */
		m_log.debug("checking whether a facility is node...");
		try {
			facility = FacilityUtil.getLocalHome().findByPrimaryKey(facilityId);
		} catch (FinderException e) {
			m_log.debug("facility not found. (facilityId = " + facilityId + ")");
			throw new FacilityNotFoundException("facility not found. (facilityId = " + facilityId + ")", e);
		} catch (Exception e) {
			m_log.debug("failure to check whether a facility is node. (facilityId = " + facilityId + ")", e);
			throw new EJBException("failure to check whether a facility is node. (facilityId = " + facilityId + ")", e);
		}
		
		m_log.debug("successful in checking whether a facility is node or not.");
		return facility.isNode();
	}
	
}
