/*
 
Copyright (C) since 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.repository.factory;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.UsedFacilityException;
import com.clustercontrol.accesscontrol.factory.AccessLock;
import com.clustercontrol.bean.FacilityConstant;
import com.clustercontrol.bean.PluginConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.commons.util.ObjectValidator;
import com.clustercontrol.repository.bean.FacilityAttributeConstant;
import com.clustercontrol.repository.bean.FacilityInfo;
import com.clustercontrol.repository.bean.FacilityTreeAttributeConstant;
import com.clustercontrol.repository.bean.NodeDeviceInfo;
import com.clustercontrol.repository.bean.NodeFilesystemInfo;
import com.clustercontrol.repository.bean.NodeHostnameInfo;
import com.clustercontrol.repository.bean.NodeInfo;
import com.clustercontrol.repository.bean.NodeNoteInfo;
import com.clustercontrol.repository.bean.ScopeInfo;
import com.clustercontrol.repository.bean.TopicConstant;
import com.clustercontrol.repository.ejb.entity.FacilityLocal;
import com.clustercontrol.repository.ejb.entity.FacilityRelationLocal;
import com.clustercontrol.repository.ejb.entity.FacilityRelationUtil;
import com.clustercontrol.repository.ejb.entity.FacilityUtil;
import com.clustercontrol.repository.ejb.entity.NodeDeviceLocal;
import com.clustercontrol.repository.ejb.entity.NodeDeviceUtil;
import com.clustercontrol.repository.ejb.entity.NodeFilesystemLocal;
import com.clustercontrol.repository.ejb.entity.NodeFilesystemUtil;
import com.clustercontrol.repository.ejb.entity.NodeHostnameLocal;
import com.clustercontrol.repository.ejb.entity.NodeHostnameUtil;
import com.clustercontrol.repository.ejb.entity.NodeLocal;
import com.clustercontrol.repository.ejb.entity.NodeNoteLocal;
import com.clustercontrol.repository.ejb.entity.NodeNoteUtil;
import com.clustercontrol.repository.ejb.entity.NodeUtil;
import com.clustercontrol.repository.util.SendTopic;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * ファシリティの更新処理を実装したクラス<BR>
 */
public class FacilityModifier {

	protected static Log m_log = LogFactory.getLog(FacilityModifier.class);
	
	/** スコープを追加する。<BR>
	 * 
	 * @param parentFacilityId スコープを配置する親スコープのファシリティID（空文字の場合はルートスコープとなる）
	 * @param property 追加するスコープ情報
	 * @param modifyUserId 作業ユーザID
	 * @param displaySortOrder 表示ソート順位
	 * @param topicSendFlg 更新を周知する場合はtrue, 周知しない場合はfalse
	 * @throws DuplicateKeyException
	 */
	public static void addScope(String parentFacilityId, ScopeInfo property, String modifyUserId, int displaySortOrder, boolean topicSendFlg) throws DuplicateKeyException {
		/** ローカル変数 */
		FacilityLocal parentFacility = null;
		FacilityLocal facility = null;
		String facilityId = null;
		
		/** メイン処理 */
		m_log.debug("adding a scope...");
		
		try {
			// 更新排他制御
			AccessLock.lock(AccessLock.REPOSITORY);
			
			// 入力値（ファシリティID）の格納
			facilityId = property.getFacilityId();
			
			// 親ファシリティがスコープかどうかを確認する
			if (! ObjectValidator.isEmptyString(parentFacilityId)) {
				parentFacility = FacilityUtil.getLocalHome().findByPrimaryKey(parentFacilityId);
				if (! parentFacility.isScope()) {
					throw new EJBException("a parent's facility is not a scope. (parentFacilityId = " + parentFacilityId + ", facilityId = " + facilityId + ")");
				}
			}
			
			// ファシリティインスタンスの生成
			facility = FacilityUtil.getLocalHome().create(facilityId);
			facility.setFacilityType(FacilityConstant.TYPE_SCOPE);
			facility.setDisplaySortOrder(displaySortOrder);
			facility.setValid(ValidConstant.TYPE_VALID);
			setFacility(facility, property, modifyUserId, false);
			
			// ファシリティ関連インスタンスの生成
			if (! ObjectValidator.isEmptyString(parentFacilityId)) {
				assignFacilityToScope(parentFacilityId, facilityId, false);
			}
		} catch (DuplicateKeyException e) {
			AplLogger apllog = new AplLogger("REP", "rep");
			String[] args = {facilityId};
			apllog.put("SYS", "012", args);
			
			m_log.info("failure to add a scope. this scope's id is duplicated. (parentFacilityId = " + parentFacilityId + ", facilityId = " + facilityId + ")", e);
			throw e;
		} catch (Exception e) {
			AplLogger apllog = new AplLogger("REP", "rep");
			String[] args = {facilityId};
			apllog.put("SYS", "012", args);
			
			m_log.warn("failure to add a scope. (facilityId = " + facilityId + ")", e);
			throw new EJBException("failure to add a scope. (parentFacilityId = " + parentFacilityId + ", facilityId = " + facilityId + ")", e);
		}
		
		// ファシリティの木情報のキャッシュを初期化する
		FacilitySelector.initCacheFacilityTree();
		
		// ファシリティの変更を通知する
		if (topicSendFlg) {
			SendTopic.sendTopic(TopicConstant.TOPIC_NAME_EXECUTE);
		}
		
		m_log.info("successful in adding a scope . (parentFacilityId = " + parentFacilityId + ", facilityId = " + facilityId + ")");
	}
	
	/**
	 * スコープを変更する。<BR>
	 * 
	 * @param property 変更後のスコープ情報
	 * @param modifyUserId 作業ユーザID
	 * @param topicSendFlg 更新を周知する場合はtrue, 周知しない場合はfalse
	 */
	public static void modifyScope(ScopeInfo property, String modifyUserId, boolean topicSendFlg) {
		/** ローカル変数 */
		FacilityLocal facility = null;
		String facilityId = null;
		
		/** メイン処理 */
		m_log.debug("modifing a scope...");
		
		try {
			// 更新排他制御
			AccessLock.lock(AccessLock.REPOSITORY);
			
			// 入力値（ファシリティID）の格納
			facilityId = property.getFacilityId();
			
			// ファシリティインスタンスを取得
			facility = FacilityUtil.getLocalHome().findByPrimaryKey(facilityId);
			
			// ファシリティがスコープかどうかを確認する
			if (! facility.isScope()) {
				throw new FinderException("this facility is not a scope. (facilityId = " + facilityId + ")");
			}
			
			// 変更後の値を格納する
			setFacility(facility, property, modifyUserId, false);
		} catch (Exception e) {
			AplLogger apllog = new AplLogger("REP", "rep");
			String[] args = {facilityId};
			apllog.put("SYS", "014", args);
			
			m_log.warn("failure to modify a scope. (facilityId = " + facilityId + ")", e);
			throw new EJBException("failure to modify a scope. (facilityId = " + facilityId + ")", e);
		}
		
		// ファシリティの木情報のキャッシュを初期化する
		FacilitySelector.initCacheFacilityTree();
		
		// ファシリティの変更を通知する
		if (topicSendFlg) {
			SendTopic.sendTopic(TopicConstant.TOPIC_NAME_EXECUTE);
		}
		
		m_log.info("successful in modifing a scope. (facilityId = " + facilityId + ")");
	}
	
	/**
	 * スコープを削除する。<BR>
	 * 
	 * @param facilityId 削除するスコープのファシリティID
	 * @param modifyUserId 作業ユーザID
	 * @param topicSendFlg 更新を周知する場合はtrue, 周知しない場合はfalse
	 */
	public static void deleteScope(String facilityId, String modifyUserId, boolean topicSendFlg) throws UsedFacilityException {
		/** ローカル変数 */
		FacilityLocal facility = null;
		
		/** メイン処理 */
		m_log.debug("deleting a scope with sub scopes...");
		
		try {
			// 更新排他制御
			AccessLock.lock(AccessLock.REPOSITORY);
			
			// 該当するファシリティインスタンスを取得
			facility = FacilityUtil.getLocalHome().findByPrimaryKey(facilityId);
			
			// 関連インスタンス、ファシリティインスタンスを削除する
			deleteScopeRecursive(facility);
		} catch (UsedFacilityException e) {
			AplLogger apllog = new AplLogger("REP", "rep");
			String[] args = {facilityId};
			apllog.put("SYS", "013", args);
			
			throw e;
		} catch (Exception e) {
			AplLogger apllog = new AplLogger("REP", "rep");
			String[] args = {facilityId};
			apllog.put("SYS", "013", args);
			
			throw new EJBException("failure to delete a scope with sub scopes. (facilityId = " + facilityId + ")", e);
		}
		
		// ファシリティの木情報のキャッシュを初期化する
		FacilitySelector.initCacheFacilityTree();
		
		// ファシリティの変更を通知する
		if (topicSendFlg) {
			SendTopic.sendTopic(TopicConstant.TOPIC_NAME_EXECUTE);
		}
		
		m_log.info("successful in deleting a scope with sub scopes. (facilityId = " + facilityId + ")");
	}
	
	/**
	 * サブスコープを含めて、スコープを削除する。<BR>
	 * 
	 * @param scope 削除するスコープインスタンス
	 * @throws UsedFacilityException
	 */
	private static void deleteScopeRecursive(FacilityLocal scope) throws UsedFacilityException {
		/** ローカル変数 */
		String facilityId = null;
		
		/** メイン処理 */
		facilityId = scope.getFacilityId();
		
		// スコープでない場合はエラーとする
		if (! scope.isScope()) {
			throw new EJBException("this facility is not a scope. (facilityId = " + facilityId + ")");
		}
		
		// 直下にスコープを存在する場合、そのスコープおよびサブスコープを削除する
		if (scope.getFacilityRelationAsParent() != null) {
			for (FacilityRelationLocal relation : (Collection<FacilityRelationLocal>)scope.getFacilityRelationAsParent()) {
				if (relation.getChildFacility().isScope()) {
					deleteScopeRecursive(relation.getChildFacility());
				}
			}
		}
		try {
			// ファシリティインスタンスの削除（DELETE CASCADEにて関連するインスタンスも削除）
			scope.remove();
		} catch (RemoveException e) {
			m_log.warn("failure to delete a scope. this scope is used for other settings. (facilityId = " + facilityId + ")", e);
			throw new UsedFacilityException(PluginConstant.TYPE_REPOSITORY, e);
		}
		
		m_log.info("successful in deleting a scope. (facilityId = " + facilityId + ")");
	}
	
	/**
	 * ノードを追加する。<BR>
	 * 
	 * @param property 追加するノード情報
	 * @param modifyUserId 作業ユーザID
	 * @param displaySortOrder 表示ソート順位
	 * @param topicSendFlg 更新を周知する場合はtrue, 周知しない場合はfalse
	 * @throws DuplicateKeyException 
	 */
	public static void addNode(NodeInfo property, String modifyUserId, int displaySortOrder, boolean topicSendFlg) throws DuplicateKeyException {
		/** ローカル変数 */
		FacilityLocal facility = null;
		NodeLocal node = null;
		ArrayList inputValue = null;
		String facilityId = null;
		Boolean valid = true;
		
		/** メイン処理 */
		m_log.debug("adding a node...");
		
		try {
			// 更新排他制御
			AccessLock.lock(AccessLock.REPOSITORY);
			
			facilityId = property.getFacilityId();
			
			valid = property.isValid();
			
			// ファシリティインスタンスの生成
			facility = FacilityUtil.getLocalHome().create(facilityId);
			facility.setFacilityType(FacilityConstant.TYPE_NODE);
			facility.setDisplaySortOrder(displaySortOrder);
			facility.setValid(valid);
			setFacility(facility, property, modifyUserId, false);
			
			// ノードインスタンスの生成
			node = NodeUtil.getLocalHome().create(facilityId);
			setNode(node, property, false);
			
			// ファシリティ関連インスタンスの生成
			assignFacilityToScope(FacilityTreeAttributeConstant.REGISTEREFD_SCOPE, facilityId, false);
		} catch (DuplicateKeyException e) {
			AplLogger apllog = new AplLogger("REP", "rep");
			String[] args = {facilityId};
			apllog.put("SYS", "009", args);
			
			m_log.info("failure to add a node. a node's id is depulicated. (facilityId = " + facilityId + ")", e);
			throw e;
		} catch (Exception e) {
			AplLogger apllog = new AplLogger("REP", "rep");
			String[] args = {facilityId};
			apllog.put("SYS", "009", args);
			
			m_log.warn("failure to add a node. (facilityId = " + facilityId + ")", e);
			throw new EJBException("failure to add a node. (facilityId = " + facilityId + ")", e);
		}
		
		// ファシリティの木情報のキャッシュを初期化する
		FacilitySelector.initCacheFacilityTree();
		
		// ファシリティの変更を通知する
		if (topicSendFlg) {
			SendTopic.sendTopic(TopicConstant.TOPIC_NAME_EXECUTE);
		}
		
		m_log.info("successful in adding a node. (facilityId = " + facilityId + ")");
	}
	
	/**
	 * ノードを変更する。<BR>
	 * 
	 * @param property 変更後のノード情報
	 * @param modifyUserId 作業ユーザID
	 * @param topicSendFlg 更新を周知する場合はtrue, 周知しない場合はfalse
	 */
	public static void modifyNode(NodeInfo property, String modifyUserId, boolean topicSendFlg) {
		/** ローカル変数 */
		FacilityLocal facility = null;
		NodeLocal node = null;
		String facilityId = null;
		Boolean valid = true;
		
		/** メイン処理 */
		m_log.debug("modifing a node...");
		
		try {
			// 更新排他制御
			AccessLock.lock(AccessLock.REPOSITORY);
			
			facilityId = property.getFacilityId();
			
			valid = property.isValid();
			
			// ファシリティインスタンスの生成
			facility = FacilityUtil.getLocalHome().findByPrimaryKey(facilityId);
			if (valid == null) {
				throw new CreateException("node's valid is invalid . (valid = " + valid + ")");
			} else {
				facility.setValid(valid);
			}
			
			// ファシリティがノードかどうかを確認する
			if (! facility.isNode()) {
				throw new FinderException("this facility is not a node. (facilityId = " + facilityId + ")");
			}
			
			setFacility(facility, property, modifyUserId, false);
			
			// ノードインスタンスの取得
			node = NodeUtil.getLocalHome().findByPrimaryKey(facilityId);
			
			// 変更情報の反映
			setNode(node, property, false);
		} catch (Exception e) {
			AplLogger apllog = new AplLogger("REP", "rep");
			String[] args = {facilityId};
			apllog.put("SYS", "011", args);
			
			m_log.warn("failure to modify a node. (facilityId = " + facilityId + ")", e);
			throw new EJBException("failure to modify a node. (facilityId = " + facilityId + ")", e);
		}
		
		// ファシリティの木情報のキャッシュを初期化する
		FacilitySelector.initCacheFacilityTree();
		
		// ファシリティの変更を通知する
		if (topicSendFlg) {
			SendTopic.sendTopic(TopicConstant.TOPIC_NAME_EXECUTE);
		}
		
		m_log.info("successful in modifing a node. (facilityId = " + facilityId + ")");
	}
	
	/**
	 * ノードを削除する。<BR>
	 * 
	 * @param facilityId 削除するノードのファシリティID
	 * @param modifyUserId 作業ユーザID
	 * @param topicSendFlg 更新を周知する場合はtrue, 周知しない場合はfalse
	 * @throws FacilityUsedException 
	 */
	public static void deleteNode(String facilityId, String modifyUserId, boolean topicSendFlg) throws UsedFacilityException {
		/** ローカル変数 */
		FacilityLocal facility = null;
		
		/** メイン処理 */
		m_log.debug("deleting a node...");
		
		// 更新排他制御
		AccessLock.lock(AccessLock.REPOSITORY);
		
		try {
			// 該当するファシリティインスタンスを取得
			facility = FacilityUtil.getLocalHome().findByPrimaryKey(facilityId);
			
			// ノードでない場合はエラーとする
			if (! facility.isNode()) {
				throw new EJBException("this facility is not a node. (facilityId = " + facilityId + ")");
			}
			
			// ファシリティインスタンスを削除する（DELETE CASCADEで関連するインスタンスやも削除される）
			facility.remove();
		} catch (RemoveException e) {
			AplLogger apllog = new AplLogger("REP", "rep");
			String[] args = {facilityId};
			apllog.put("SYS", "010", args);
			
			m_log.info("failure to delete a node. this node is used by other settings. (facilityId = " + facilityId + ")");
			throw new UsedFacilityException(PluginConstant.TYPE_REPOSITORY, e);
		} catch (Exception e) {
			AplLogger apllog = new AplLogger("REP", "rep");
			String[] args = {facilityId};
			apllog.put("SYS", "010", args);
			
			m_log.warn("failure to delete a node. (facilityId = " + facilityId + ")");
			throw new EJBException("failure to delete a node. (facilityId = " + facilityId + ")");
		}
		
		// ファシリティの木情報のキャッシュを初期化する
		FacilitySelector.initCacheFacilityTree();
		
		// ファシリティの変更を通知する
		if (topicSendFlg) {
			SendTopic.sendTopic(TopicConstant.TOPIC_NAME_EXECUTE);
		}
		
		m_log.info("successful in deleting a node. (facilityId = " + facilityId + ")");
	}
	
	/**
	 * ノードインスタンスに属するホスト名をすべて削除する。<BR>
	 * 
	 * @param node ホスト名を削除するノードインスタンス
	 * @throws RemoveException
	 */
	private static void deleteNodeHostname(NodeLocal node) throws RemoveException {
		/** ローカル変数 */
		String facilityId = null;
		String hostname = null;
		
		/** メイン処理 */
		if (node.getNodeHostname() != null) {
			for (NodeHostnameLocal nodeHostname : (Collection<NodeHostnameLocal>)node.getNodeHostname()) {
				try {
					facilityId = nodeHostname.getFacilityId();
					hostname = nodeHostname.getHostname();
					nodeHostname.remove();
					m_log.debug("successful in deleting a node's hostname. (facilityId = " + facilityId + ", hostname = " + hostname + ")");
				} catch (RemoveException e) {
					m_log.warn("failure to delete a node's hostname. (facilityId = " + facilityId + ", hostname = " + hostname + ")");
					throw e;
				}
			}
		}
	}
	
	/**
	 * ノードインスタンスに属する備考をすべて削除する。<BR>
	 * 
	 * @param node 備考を削除するノードインスタンス
	 * @throws RemoveException
	 */
	private static void deleteNodeNote(NodeLocal node) throws RemoveException {
		/** ローカル変数 */
		String facilityId = null;
		Integer noteId = null;
		
		/** メイン処理 */
		if (node.getNodeNote() != null) {
			for (NodeNoteLocal nodeNote : (Collection<NodeNoteLocal>)node.getNodeNote()) {
				try {
					facilityId = node.getFacilityId();
					noteId = nodeNote.getNoteId();
					nodeNote.remove();
					m_log.debug("successful in deleting a node's note. (facilityId = " + facilityId + ", noteId = " + noteId + ")");
				} catch (RemoveException e) {
					m_log.warn("failure to delete a node's note. (facilityId = " + facilityId + ", noteId = " + noteId + ")");
					throw e;
				}
			}
		}
	}
	
	/**
	 * ノードインスタンスに属するデバイスをすべて削除する。<BR>
	 * 
	 * @param node デバイスを削除するノードインスタンス
	 * @throws RemoveException
	 */
	private static void deleteNodeDevice(NodeLocal node) throws RemoveException {
		/** ローカル変数 */
		String facilityId = null;
		String deviceType = null;
		Integer deviceIndex = null;
		
		/** メイン処理 */
		if (node.getNodeDevice() != null) {
			for (NodeDeviceLocal nodeDevice : (Collection<NodeDeviceLocal>)node.getNodeDevice()) {
				try {
					facilityId = nodeDevice.getFacilityId();
					deviceType = nodeDevice.getDeviceType();
					deviceIndex = nodeDevice.getDeviceIndex();
					nodeDevice.remove();
					m_log.debug("successful in deleting a node's device. (facilityId = " + facilityId + ", deviceType = " + deviceType + ", deviceIndex = " + deviceIndex + ")");
				} catch (RemoveException e) {
					m_log.warn("failure to delete a node's device. (facilityId = " + facilityId + ", deviceType = " + deviceType + ", deviceIndex = " + deviceIndex + ")");
					throw e;
				}
			}
		}
	}
	
	/**
	 * ノードインスタンスに属するファイルシステムをすべて削除する。<BR>
	 * 
	 * @param node ファイルシステムを削除するノードインスタンス
	 * @throws RemoveException
	 */
	private static void deleteNodeFilesystem(NodeLocal node) throws RemoveException {
		/** ローカル変数 */
		String facilityId = null;
		Integer fsIndex = null;
		
		/** メイン処理 */
		if (node.getNodeFilesystem() != null) {
			for (NodeFilesystemLocal nodeFilesystem : (Collection<NodeFilesystemLocal>)node.getNodeFilesystem()) {
				try {
					facilityId = nodeFilesystem.getFacilityId();
					fsIndex = nodeFilesystem.getFsIndex();
					nodeFilesystem.remove();
					m_log.debug("successful in deleting a node's filesystem. (facilityId = " + facilityId + ", fsIndex = " + fsIndex + ")");
				} catch (RemoveException e) {
					m_log.warn("failure to delete a node's filesystem. (facilityId = " + facilityId + ", fsIndex = " + fsIndex + ")");
					throw e;
				}
			}
		}
	}
	
	/**
	 * スコープに属するノードを一括変更する。<BR>
	 * 
	 * @param property 変更後のノード情報
	 * @param parentFacilityId スコープのファシリティID
	 * @param modifyUserId 作業ユーザID
	 * @param topicSendFlg 更新を周知する場合はtrue, 周知しない場合はfalse
	 * @throws NamingException
	 * @throws CreateException
	 * @throws FinderException
	 */
	public static void modifyAllNode(NodeInfo property, String parentFacilityId, String modifyUserId, boolean topicSendFlg) {
		/** ローカル変数 */
		FacilityLocal parentFacility = null;
		FacilityLocal childFacility = null;
		String facilityId = null;
		
		/** メイン処理 */
		m_log.debug("modifing nodes under a scope...");
		
		try {
			// 更新排他制御
			AccessLock.lock(AccessLock.REPOSITORY);
			
			// 親ファシリティを取得
			parentFacility = FacilityUtil.getLocalHome().findByPrimaryKey(parentFacilityId);
			
			// 親ファシリティがスコープかどうかを確認する
			if (! parentFacility.isScope()) {
				throw new EJBException("this parent's facility is not a scope. (facilityId = " + facilityId + ")");
			}
			
			// 親ファシリティに属するノードをすべて変更する
			if (parentFacility.getFacilityRelationAsParent() != null) {
				for (FacilityRelationLocal facilityRelation : (Collection<FacilityRelationLocal>)parentFacility.getFacilityRelationAsParent()) {
					childFacility = facilityRelation.getChildFacility();
					facilityId = childFacility.getFacilityId();
					
					// 子ファシリティがノードでない場合は変更しない
					if (! childFacility.isNode()) {
						continue;
					}
					
					// ファシリティ情報を更新する
					setFacility(childFacility, property, modifyUserId, true);
					
					// ノード情報を更新する（ホスト名、備考、デバイス情報、ファイルシステム情報を除く）
					setNode(childFacility.getNode(), property, true);
				}
			}
		} catch (Exception e) {
			m_log.warn("failure to modify a node. (facilityId = " + facilityId + ")", e);
			throw new EJBException("failure to modify a node. (facilityId = " + facilityId + ")", e);
		}
		
		// ファシリティの木情報のキャッシュを初期化する
		FacilitySelector.initCacheFacilityTree();
		
		// ファシリティの変更を通知する
		if (topicSendFlg) {
			SendTopic.sendTopic(TopicConstant.TOPIC_NAME_EXECUTE);
		}
		
		m_log.info("successful in modifing nodes under a scope. (parentFacilityId = " + parentFacilityId + ")");
	}
	
	/**
	 * スコープにノードを割り当てる。<BR>
	 * 
	 * @param parentFacilityId スコープのファシリティID
	 * @param property ノード情報
	 * @param topicSendFlg 更新を周知する場合はtrue, 周知しない場合はfalse
	 */
	public static void assignFacilityToScope(String parentFacilityId, NodeInfo property, boolean topicSendFlg) {
		/** ローカル変数 */
		String facilityId = null;
		
		/** メイン処理 */
		facilityId = property.getFacilityId();
		assignFacilityToScope(parentFacilityId, facilityId, topicSendFlg);
	}
	
	/**
	 * スコープにノードを割り当てる。<BR>
	 * 
	 * @param parentFacilityId スコープのファシリティID
	 * @param facilityId ノードのファシリティID
	 * @param topicSendFlg 更新を周知する場合はtrue, 周知しない場合はfalse
	 */
	public static void assignFacilityToScope(String parentFacilityId, String facilityId, boolean topicSendFlg) {
		/** ローカル変数 */
		String[] facilityIds = { facilityId };
		
		/** メイン処理 */
		assignFacilityToScope(parentFacilityId, facilityIds, topicSendFlg);
	}
	
	/**
	 * スコープにノードを割り当てる。<BR>
	 * 
	 * @param parentFacilityId スコープのファシリティID
	 * @param facilityIds ノードのファシリティID配列
	 * @param topicSendFlg 更新を周知する場合はtrue, 周知しない場合はfalse
	 */
	public static void assignFacilityToScope(String parentFacilityId, String[] facilityIds, boolean topicSendFlg) {
		/** ローカル変数 */
		FacilityLocal scope = null;
		boolean relationExist = false;
		String facilityId = null;
		
		/** メイン処理 */
		m_log.debug("assigning facilities to a scope...");
		
		try {
			// 更新排他制御
			AccessLock.lock(AccessLock.REPOSITORY);
			
			// スコープのファシリティインスタンスの取得
			scope = FacilityUtil.getLocalHome().findByPrimaryKey(parentFacilityId);
			
			if (! scope.isScope()) {
				m_log.warn("parent's facility is not a scope. (parentfacilityId = " + parentFacilityId + ")");
				throw new EJBException("parent's facility is not a scope. (facilityId = " + parentFacilityId + ")");
			}
			
			for (int i = 0; i < facilityIds.length; i++) {
				facilityId = facilityIds[i];
				
				// 割り当て済みのファシリティかどうかを確認する
				relationExist = false;
				if (scope.getFacilityRelationAsParent() != null) {
					for (FacilityRelationLocal relation : (Collection<FacilityRelationLocal>)scope.getFacilityRelationAsParent()) {
						if (relation.getChildFacilityId().compareTo(facilityId) == 0) {
							relationExist = true;
							break;
						}
					}
				}
				if (relationExist) {
					m_log.info("skipped assinging a facility to a scope. (parentFacilityId = " + parentFacilityId + ", facilityId = " + facilityId + ")");
				} else {
					FacilityRelationUtil.getLocalHome().create(parentFacilityId, facilityId);
					m_log.info("successful in assinging a facility to a scope. (parentFacilityId = " + parentFacilityId + ", facilityId = " + facilityId + ")");
				}
			}
		} catch (Exception e) {
			AplLogger apllog = new AplLogger("REP", "rep");
			String[] args = {facilityId, parentFacilityId};
			apllog.put("SYS", "015", args);
			
			m_log.warn("failure to assign a node to a scope. (parentFacilityId = " + parentFacilityId + ", facilityId = " + facilityId + ")", e);
			throw new EJBException("failure to assign a node to a scope. (parentFacilityId = " + parentFacilityId + ", facilityId = " + facilityId + ")", e);
		}
		
		// ファシリティの木情報のキャッシュを初期化する
		FacilitySelector.initCacheFacilityTree();
		
		// ファシリティの変更を通知する
		if (topicSendFlg) {
			SendTopic.sendTopic(TopicConstant.TOPIC_NAME_EXECUTE);
		}
		
		m_log.info("successful in assigning facilities to a scope.");
	}
	
	/**
	 * スコープからノードの割り当てを解除する。<BR>
	 * 
	 * @param parentFacilityId スコープのファシリティID
	 * @param facilityIds ノードのファシリティID
	 * @param modifyUserId 作業ユーザID
	 * @param topicSendFlg 更新を周知する場合はtrue, 周知しない場合はfalse
	 */
	public static void releaseNodeFromScope(String parentFacilityId, String[] facilityIds, String modifyUserId, boolean topicSendFlg) {
		/** ローカル変数 */
		FacilityLocal facility = null;
		String facilityId = null;
		
		/** メイン処理 */
		m_log.debug("releasing nodes from a scope...");
		
		try {
			// 更新排他制御
			AccessLock.lock(AccessLock.REPOSITORY);
			
			// 該当するファシリティインスタンスを取得
			facility = FacilityUtil.getLocalHome().findByPrimaryKey(parentFacilityId);
			
			if (! facility.isScope()) {
				throw new EJBException("parent's facility is not a scope. (parentFacilityId = " + parentFacilityId + ")");
			}
			
			if (facility.getFacilityRelationAsParent() != null) {
				for (int i = 0; i < facilityIds.length; i++) {
					facilityId = facilityIds[i];
					for (FacilityRelationLocal relation : (Collection<FacilityRelationLocal>)facility.getFacilityRelationAsParent()) {
						if (relation.getChildFacility().getFacilityId().compareTo(facilityId) == 0) {
							relation.remove();
							m_log.info("successful in releaseing a node. (parentFacilityId = " + parentFacilityId + ", facilityId = " + facilityId + ")");
							break;
						}
					}
				}
			}
		} catch (Exception e) {
			AplLogger apllog = new AplLogger("REP", "rep");
			String[] args = {facilityId, parentFacilityId};
			apllog.put("SYS", "016", args);
			
			m_log.warn("failure to release a node. (parentFacilityId = " + parentFacilityId + ", facilityId = " + facilityId + ")", e);
			throw new EJBException("failure to release a node. (parentFacilityId = " + parentFacilityId + ", facilityId = " + facilityId + ")", e);
		}
		
		// ファシリティの木情報のキャッシュを初期化する
		FacilitySelector.initCacheFacilityTree();
		
		// ファシリティの変更を通知する
		if (topicSendFlg) {
			SendTopic.sendTopic(TopicConstant.TOPIC_NAME_EXECUTE);
		}
		
		m_log.info("successful in releasing nodes from a scope.");
	}
	
	/**
	 * Propertyインスタンスに格納された値をファシリティインスタンスに格納する。<BR>
	 * 
	 * @param facility 格納先となるファシリティインスタンス
	 * @param property 格納する情報
	 * @param modifyUserId 作業ユーザID
	 * @param skipIfEmptyFlg trueにすると、Propertyインスタンスの各格納値がnullあるいは空文字の場合に格納しない
	 */
	private static void setFacility(FacilityLocal facility, FacilityInfo property, String modifyUserId, boolean skipIfEmptyFlg) {
		/** ローカル変数 */
		ArrayList inputValue = null;
		String facilityName = null;
		String description = null;
		Timestamp now = null;
		
		/** メイン処理 */
		// 現在日時を取得
		now = new Timestamp(new Date().getTime());
		
		if (facility.isScope()) {
			// 入力値（ファシリティ名）の格納
			facilityName = property.getFacilityName();
			
			// 入力値（説明）の格納
			description = property.getDescription();
		} else if (facility.isNode()) {
			// 入力値（ファシリティ名）の格納
			facilityName = property.getFacilityName();
			
			// 入力値（説明）の格納
			description = property.getDescription();
		} else {
			throw new EJBException("this facility's type is invalid. (facilityType = " + facility.getFacilityType() + ")");
		}
		
		// ファシリティインスタンスへの入力値の格納
		if ( ! (skipIfEmptyFlg && ObjectValidator.isEmptyString(facilityName)) ) {
			facility.setFacilityName(facilityName);
		}
		if ( ! (skipIfEmptyFlg && ObjectValidator.isEmptyString(description)) ) {
			facility.setDescription(description);
		}
		if (ObjectValidator.isEmptyString(facility.getCreateUserId()) && facility.getCreateDatetime().getTime() == 0) {
			facility.setCreateUserId(modifyUserId);
			facility.setCreateDatetime(now);
		}
		facility.setModifyUserId(modifyUserId);
		facility.setModifyDatetime(now);
	}
	
	/**
	 * Propertyインスタンス上のノード情報をノードインスタンスに格納する。<BR>
	 * 
	 * @param node 格納先となるノードインスタンス
	 * @param property ノード情報
	 * @param skipIfEmptyFlg trueにすると、Propertyインスタンスの各格納値がnullあるいは空文字の場合に格納しない
	 * @throws NamingException
	 * @throws CreateException
	 * @throws RemoveException 
	 */
	private static void setNode(NodeLocal node, NodeInfo property, boolean skipIfEmptyFlg) throws NamingException, CreateException, RemoveException {
		/** ローカル変数 */
		String platformFamily = null; 
		String hardwareType = null; 
		String iconImage = null; 
		
		Integer snmpPort = null; 
		String snmpCommunity = null; 
		String snmpVersion = null; 
		Integer snmpTimeout = null; 
		Integer snmpRetryCount = null; 
		String snmpProxy = null;
		String wbemUser = null; 
		String wbemUserPassword = null; 
		Integer wbemPort = null; 
		String wbemProtocol = null; 
		Integer wbemTimeout = null; 
		Integer wbemRetryCount = null; 
		Boolean dhcpClient = null; 
		Integer ipAddressType = null; 
		Integer ipAddressVersion = null; 
		String ipAddressV4 = null; 
		String ipAddressV6 = null; 
		
		String nodeName = null; 
		String osName = null; 
		String osRelease = null; 
		String osVersion = null; 
		String characterSet = null; 
		
		String virtualizationNodeType = null; 
		String vmManagementNode = null; 
		Integer vmIndex = null; 
		String vmName = null; 
		String virtualizationSolution = null; 
		String vmId = null; 
		String vmUser = null; 
		String vmUserPassword = null; 
		String vmProtocol = null; 
		
		String administrator = null; 
		String contact = null; 
		
		NodeNoteLocal nodeNote = null;
		int noteId = 0;
		
		NodeDeviceLocal nodeDevice = null;
		Integer deviceIndex = null;
		String deviceType = null;
		
		NodeFilesystemLocal nodeFilesystem = null;
		Integer fsIndex = null;
		
		/** メイン処理 */
		
		// ノード関連
		platformFamily = property.getPlatformFamily();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(platformFamily))) {
			node.setPlatformFamily(platformFamily);
		}
		
		hardwareType = property.getHardwareType();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(hardwareType))) {
			node.setHardwareType(hardwareType);
		}
		
		iconImage = property.getIconImage();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(iconImage))) {
			node.setIconImage(iconImage);
		}
		
		// SNMP関連
		snmpPort = property.getSnmpPort();
		if (! (skipIfEmptyFlg && snmpPort == -1)) {
			node.setSnmpPort(snmpPort);
		}
		
		snmpCommunity = property.getSnmpCommunity();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(snmpCommunity))) {
			node.setSnmpCommunity(snmpCommunity);
		}
		
		snmpVersion = property.getSnmpVersion();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(snmpVersion))) {
			node.setSnmpVersion(snmpVersion);
		}
		
		snmpTimeout = property.getSnmpTimeout();
		if (! (skipIfEmptyFlg && snmpTimeout == -1)) {
			node.setSnmpTimeout(snmpTimeout);
		}
		
		snmpRetryCount = property.getSnmpRetryCount();
		if (! (skipIfEmptyFlg && snmpRetryCount == -1)) {
			node.setSnmpRetryCount(snmpRetryCount);
		}
		
		snmpProxy = property.getSnmpProxy();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(snmpProxy))) {
			node.setSnmpProxy(snmpProxy);
		}
		
		// WBEM関連
		wbemUser = property.getWbemUser();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(wbemUser))) {
			node.setWbemUser(wbemUser);
		}
		
		wbemUserPassword = property.getWbemUserPassword();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(wbemUserPassword))) {
			node.setWbemUserPassword(wbemUserPassword);
		}
		
		wbemPort = property.getWbemPort();
		if (! (skipIfEmptyFlg && wbemPort == -1)) {
			node.setWbemPort(wbemPort);
		}
		
		wbemProtocol = property.getWbemProtocol();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(wbemProtocol))) {
			node.setWbemProtocol(wbemProtocol);
		}
		
		wbemTimeout = property.getWbemTimeout();
		if (! (skipIfEmptyFlg && wbemTimeout == -1)) {
			node.setWbemTimeout(wbemTimeout);
		}
		
		wbemRetryCount = property.getWbemRetryCount();
		if (! (skipIfEmptyFlg && wbemRetryCount == -1)) {
			node.setWbemRetryCount(wbemRetryCount);
		}
		
		// IPアドレス関連
		dhcpClient = property.getDhcpClient();
		if (! (skipIfEmptyFlg && dhcpClient == null)) {
			node.setDhcpClient(dhcpClient);
		}
		
		ipAddressType = property.getIpAddressType();
		if (! (skipIfEmptyFlg && ipAddressType == -1)) {
			node.setIpAddressType(ipAddressType);
		}
		
		ipAddressVersion = property.getIpAddressVersion();
		if (! (skipIfEmptyFlg && ipAddressVersion == -1)) {
			node.setIpAddressVersion(ipAddressVersion);
		}
		
		ipAddressV4 = property.getIpAddressV4();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(ipAddressV4))) {
			node.setIpAddressV4(ipAddressV4);
		}
		
		ipAddressV6 = property.getIpAddressV6();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(ipAddressV6))) {
			node.setIpAddressV6(ipAddressV6);
		}
		
		// OS関連
		nodeName = property.getNodeName();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(nodeName))) {
			node.setNodeName(nodeName);
		}
		
		osName = property.getOsName();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(osName))) {
			node.setOsName(osName);
		}
		
		osRelease = property.getOsRelease();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(osRelease))) {
			node.setOsRelease(osRelease);
		}
		
		osVersion = property.getOsVersion();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(osVersion))) {
			node.setOsVersion(osVersion);
		}
		
		characterSet = property.getCharacterSet();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(characterSet))) {
			node.setCharacterSet(characterSet);
		}
		
		// VM関連
		virtualizationNodeType = property.getVirtualizationNodeType();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(virtualizationNodeType))) {
			node.setVirtualizationNodeType(virtualizationNodeType);
		}
		
		vmManagementNode = property.getVmManagementNode();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(vmManagementNode))) {
			node.setVmManagementNode(vmManagementNode);
		}
		
		vmIndex = property.getVmIndex();
		if (! (skipIfEmptyFlg && vmIndex == -1)) {
			node.setVmIndex(vmIndex);
		}
		
		vmName = property.getVmName();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(vmName))) {
			node.setVmName(vmName);
		}
		
		virtualizationSolution = property.getVirtualizationSolution();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(virtualizationSolution))) {
			node.setVirtualizationSolution(virtualizationSolution);
		}
		
		vmId = property.getVmId();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(vmId))) {
			node.setVmId(vmId);
		}
		
		vmUser = property.getVmUser();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(vmUser))) {
			node.setVmUser(vmUser);
		}
		
		vmUserPassword = property.getVmUserPassword();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(vmUserPassword))) {
			node.setVmUserPassword(vmUserPassword);
		}
		
		vmProtocol = property.getVmProtocol();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(vmProtocol))) {
			node.setVmProtocol(vmProtocol);
		}
		
		// 管理情報関連
		administrator = property.getAdministrator();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(administrator))) {
			node.setAdministrator(administrator);
		}
		
		contact = property.getContact();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(contact))) {
			node.setContact(contact);
		}
		
		// ホスト名
		if (! skipIfEmptyFlg) {
			deleteNodeHostname(node);
			if (property.getNodeHostnameInfo() != null) {
				for (NodeHostnameInfo hostname : property.getNodeHostnameInfo()) {
					try {
						NodeHostnameUtil.getLocalHome().create(node.getFacilityId(), hostname.getHostname());
					} catch (CreateException e) {
						m_log.warn("failure to add a node's hostname. (facilityId = " + node.getFacilityId() + ", hostname = " + hostname + ")", e);
						throw e;
					}
				}
			}
		}
		
		// 備考
		if (! skipIfEmptyFlg) {
			deleteNodeNote(node);
			if (property.getNodeNoteInfo() != null) {
				for (NodeNoteInfo note : property.getNodeNoteInfo()) {
					try {
						nodeNote = NodeNoteUtil.getLocalHome().create(node.getFacilityId(), note.getNoteId());
						nodeNote.setNote(note.getNote());
					} catch (CreateException e) {
						m_log.warn("failure to add a node's note. (facilityId = " + node.getFacilityId() + ", noteId = " + noteId + ")", e);
						throw e;
					}
				}
			}
		}
		
		// デバイス関連
		if (! skipIfEmptyFlg) {
			deleteNodeDevice(node);
			if (property.getNodeDeviceInfo() != null) {
				for (NodeDeviceInfo deviceProperty : property.getNodeDeviceInfo()) {
					deviceIndex = deviceProperty.getDeviceIndex();
					
					deviceType = deviceProperty.getDeviceType();
					
					if (deviceIndex != -1 && ! ObjectValidator.isEmptyString(deviceType)) {
						try {
							nodeDevice = NodeDeviceUtil.getLocalHome().create(node.getFacilityId(), deviceType, deviceIndex);
							
							nodeDevice.setDeviceName(deviceProperty.getDeviceName());
							
							nodeDevice.setDeviceDisplayName(deviceProperty.getDeviceDisplayName());
							
							nodeDevice.setDeviceSnmpOid(deviceProperty.getDeviceSnmpOid());
							
							nodeDevice.setDeviceDescription(deviceProperty.getDeviceDescription());
						} catch (CreateException e) {
							m_log.warn("failure to add a node's device. (facilityId = " + node.getFacilityId() + ", deviceType = " + deviceType + ", deviceIndex = " + deviceIndex + ")", e);
							throw e;
						}
					} else {
						throw new CreateException("both type and index of device are required. (deviceType = " + deviceType + ", deviceIndex = " + deviceIndex + ")");
					}
				}
			}
		}
		
		// ファイルシステム関連
		if (! skipIfEmptyFlg) {
			deleteNodeFilesystem(node);
			if (property.getNodeFilesystemInfo() != null) {
				for (NodeFilesystemInfo fsProperty : property.getNodeFilesystemInfo()) {
					fsIndex = fsProperty.getFsIndex();
					
					if (fsIndex != -1) {
						try { 
							nodeFilesystem = NodeFilesystemUtil.getLocalHome().create(node.getFacilityId(), fsIndex);
							
							nodeFilesystem.setFsMountPoint(fsProperty.getFsMountPoint());
							
							nodeFilesystem.setFsDisplayName(fsProperty.getFsDisplayName());
							
							nodeFilesystem.setFsType(fsProperty.getFsType());
							
							nodeFilesystem.setFsSnmpOid(fsProperty.getFsSnmpOid());
							
							nodeFilesystem.setFsDescription(fsProperty.getFsDescription());
						} catch (CreateException e) {
							m_log.warn("failure to add a node's filesystem. (facilityId = " + node.getFacilityId() + ", fsIndex = " + fsIndex + ")", e);
							throw e;
						}
					} else {
						throw new CreateException("both index of filesystem are required. (fsIndex = " + fsIndex + ")");
					}
				}
			}
		}
	}
	
	/**
	 * 指定された項目のノード情報を変更する。<BR>
	 * 
	 * @param facilityId ノードのファシリティID
	 * @param attributes 変更項目一覧
	 */
	public static void setNodeDetail(String facilityId, HashMap<String, Object> attributes) {
		/** ローカル変数 */
		FacilityLocal facility = null;
		NodeLocal node = null;
		NodeNoteLocal nodeNote = null;
		int noteId = 0;
		
		/** メイン処理 */
		m_log.debug("setting node's detail...");
		
		try {
			// 更新排他制御
			AccessLock.lock(AccessLock.REPOSITORY);
			
			node = NodeUtil.getLocalHome().findByPrimaryKey(facilityId);
			facility = node.getFacility();
			
			// ファシリティ関連
			if (attributes.containsKey(FacilityAttributeConstant.CN)) {
				facility.setFacilityName(ObjectValidator.objectToString(attributes.get(FacilityAttributeConstant.CN)));
			}
			
			if (attributes.containsKey(FacilityAttributeConstant.DESCRIPTION)) {
				facility.setDescription(ObjectValidator.objectToString(attributes.get(FacilityAttributeConstant.DESCRIPTION)));
			}
			
			if (attributes.containsKey(FacilityAttributeConstant.VALID)) {
				facility.setValid(ObjectValidator.objectToBoolean(attributes.get(FacilityAttributeConstant.VALID)));
			}
			
			if (attributes.containsKey(FacilityAttributeConstant.CREATORSNAME)) {
				facility.setCreateUserId(ObjectValidator.objectToString(attributes.get(FacilityAttributeConstant.CREATORSNAME)));
			}
			
			if (attributes.containsKey(FacilityAttributeConstant.CREATETIMESTAMP)) {
				facility.setCreateDatetime(new Timestamp(ObjectValidator.objectToDate(attributes.get(FacilityAttributeConstant.CREATETIMESTAMP)).getTime()));
			}
			
			if (attributes.containsKey(FacilityAttributeConstant.MODIFIERSNAME)) {
				facility.setModifyUserId(ObjectValidator.objectToString(attributes.get(FacilityAttributeConstant.MODIFIERSNAME)));
			}
			
			if (attributes.containsKey(FacilityAttributeConstant.MODIFYTIMESTAMP)) {
				facility.setModifyDatetime(new Timestamp(ObjectValidator.objectToDate(attributes.get(FacilityAttributeConstant.MODIFYTIMESTAMP)).getTime()));
			}
			
			// ノード関連
			if (attributes.containsKey(FacilityAttributeConstant.PLATFORM)) {
				node.setPlatformFamily(ObjectValidator.objectToString(attributes.get(FacilityAttributeConstant.PLATFORM)));
			}
			
			if (attributes.containsKey(FacilityAttributeConstant.MACHINE)) {
				node.setHardwareType(ObjectValidator.objectToString(attributes.get(FacilityAttributeConstant.MACHINE)));
			}
			
			if (attributes.containsKey(FacilityAttributeConstant.ICONIMAGE)) {
				node.setIconImage(ObjectValidator.objectToString(attributes.get(FacilityAttributeConstant.ICONIMAGE)));
			}
			
			// SNMP関連
			if (attributes.containsKey(FacilityAttributeConstant.SNMPPORT)) {
				node.setSnmpPort(ObjectValidator.objectToInteger(attributes.get(FacilityAttributeConstant.SNMPPORT)));
			}
			
			if (attributes.containsKey(FacilityAttributeConstant.SNMPCOMMUNITY)) {
				node.setSnmpCommunity(ObjectValidator.objectToString(attributes.get(FacilityAttributeConstant.SNMPCOMMUNITY)));
			}
			
			if (attributes.containsKey(FacilityAttributeConstant.SNMPVERSION)) {
				node.setSnmpVersion(ObjectValidator.objectToString(attributes.get(FacilityAttributeConstant.SNMPVERSION)));
			}
			
			if (attributes.containsKey(FacilityAttributeConstant.SNMPTIMEOUT)) {
				node.setSnmpTimeout(ObjectValidator.objectToInteger(attributes.get(FacilityAttributeConstant.SNMPTIMEOUT)));
			}
			
			if (attributes.containsKey(FacilityAttributeConstant.SNMPRETRIES)) {
				node.setSnmpRetryCount(ObjectValidator.objectToInteger(attributes.get(FacilityAttributeConstant.SNMPRETRIES)));
			}
			
			if (attributes.containsKey(FacilityAttributeConstant.SNMPPROXY)) {
				node.setSnmpProxy(ObjectValidator.objectToString(attributes.get(FacilityAttributeConstant.SNMPPROXY)));
			}
			
			// WBEM関連
			if (attributes.containsKey(FacilityAttributeConstant.WBEMUSER)) {
				node.setWbemUser(ObjectValidator.objectToString(attributes.get(FacilityAttributeConstant.WBEMUSER)));
			}
			
			if (attributes.containsKey(FacilityAttributeConstant.WBEMUSERPASSWORD)) {
				node.setWbemUserPassword(ObjectValidator.objectToString(attributes.get(FacilityAttributeConstant.WBEMUSERPASSWORD)));
			}
			
			if (attributes.containsKey(FacilityAttributeConstant.WBEMPORT)) {
				node.setWbemPort(ObjectValidator.objectToInteger(attributes.get(FacilityAttributeConstant.WBEMPORT)));
			}
			
			if (attributes.containsKey(FacilityAttributeConstant.WBEMPROTOCOL)) {
				node.setWbemProtocol(ObjectValidator.objectToString(attributes.get(FacilityAttributeConstant.WBEMPROTOCOL)));
			}
			
			if (attributes.containsKey(FacilityAttributeConstant.WBEMTIMEOUT)) {
				node.setWbemTimeout(ObjectValidator.objectToInteger(attributes.get(FacilityAttributeConstant.WBEMTIMEOUT)));
			}
			
			if (attributes.containsKey(FacilityAttributeConstant.WBEMRETRIES)) {
				node.setWbemRetryCount(ObjectValidator.objectToInteger(attributes.get(FacilityAttributeConstant.WBEMRETRIES)));
			}
			
			// IPアドレス関連
			if (attributes.containsKey(FacilityAttributeConstant.DHCPCLIENT)) {
				node.setDhcpClient(ObjectValidator.objectToBoolean(attributes.get(FacilityAttributeConstant.DHCPCLIENT)));
			}
			
			if (attributes.containsKey(FacilityAttributeConstant.IPTYPE)) {
				node.setIpAddressType(ObjectValidator.objectToInteger(attributes.get(FacilityAttributeConstant.IPTYPE)));
			}
			
			if (attributes.containsKey(FacilityAttributeConstant.IPPROTOCOLNUMBER)) {
				node.setIpAddressVersion(ObjectValidator.objectToInteger(attributes.get(FacilityAttributeConstant.IPPROTOCOLNUMBER)));
			}
			
			if (attributes.containsKey(FacilityAttributeConstant.IPNETWORKNUMBER)) {
				node.setIpAddressV4(ObjectValidator.objectToString(attributes.get(FacilityAttributeConstant.IPNETWORKNUMBER)));
			}
			
			if (attributes.containsKey(FacilityAttributeConstant.IPNETWORKNUMBERV6)) {
				node.setIpAddressV6(ObjectValidator.objectToString(attributes.get(FacilityAttributeConstant.IPNETWORKNUMBERV6)));
			}
			
			// OS関連
			if (attributes.containsKey(FacilityAttributeConstant.NODENAME)) {
				node.setNodeName(ObjectValidator.objectToString(attributes.get(FacilityAttributeConstant.NODENAME)));
			}
			
			if (attributes.containsKey(FacilityAttributeConstant.OSNAME)) {
				node.setOsName(ObjectValidator.objectToString(attributes.get(FacilityAttributeConstant.OSNAME)));
			}
			
			if (attributes.containsKey(FacilityAttributeConstant.OSRELEASE)) {
				node.setOsRelease(ObjectValidator.objectToString(attributes.get(FacilityAttributeConstant.OSRELEASE)));
			}
			
			if (attributes.containsKey(FacilityAttributeConstant.OSVERSION)) {
				node.setOsVersion(ObjectValidator.objectToString(attributes.get(FacilityAttributeConstant.OSVERSION)));
			}
			
			if (attributes.containsKey(FacilityAttributeConstant.CHARSET)) {
				node.setCharacterSet(ObjectValidator.objectToString(attributes.get(FacilityAttributeConstant.CHARSET)));
			}
			
			// 仮想化関連
			if (attributes.containsKey(FacilityAttributeConstant.VIRTNODETYPE)) {
				node.setVirtualizationNodeType(ObjectValidator.objectToString(attributes.get(FacilityAttributeConstant.VIRTNODETYPE)));
			}
			
			if (attributes.containsKey(FacilityAttributeConstant.VMMANAGEMENTNODE)) {
				node.setVmManagementNode(ObjectValidator.objectToString(attributes.get(FacilityAttributeConstant.VMMANAGEMENTNODE)));
			}
			
			if (attributes.containsKey(FacilityAttributeConstant.VMINDEX)) {
				node.setVmIndex(ObjectValidator.objectToInteger(attributes.get(FacilityAttributeConstant.VMINDEX)));
			}
			
			if (attributes.containsKey(FacilityAttributeConstant.VMNAME)) {
				node.setVmName(ObjectValidator.objectToString(attributes.get(FacilityAttributeConstant.VMNAME)));
			}
			
			if (attributes.containsKey(FacilityAttributeConstant.VIRTSOLUTION)) {
				node.setVirtualizationSolution(ObjectValidator.objectToString(attributes.get(FacilityAttributeConstant.VIRTSOLUTION)));
			}
			
			if (attributes.containsKey(FacilityAttributeConstant.VMID)) {
				node.setVmId(ObjectValidator.objectToString(attributes.get(FacilityAttributeConstant.VMID)));
			}
			
			if (attributes.containsKey(FacilityAttributeConstant.VMUSER)) {
				node.setVmUser(ObjectValidator.objectToString(attributes.get(FacilityAttributeConstant.VMUSER)));
			}
			
			if (attributes.containsKey(FacilityAttributeConstant.VMUSERPASSWORD)) {
				node.setVmUserPassword(ObjectValidator.objectToString(attributes.get(FacilityAttributeConstant.VMUSERPASSWORD)));
			}
			
			if (attributes.containsKey(FacilityAttributeConstant.VMPROTOCOL)) {
				node.setVmProtocol(ObjectValidator.objectToString(attributes.get(FacilityAttributeConstant.VMPROTOCOL)));
			}
			
			// 管理情報関連
			if (attributes.containsKey(FacilityAttributeConstant.MANAGERNAME)) {
				node.setAdministrator(ObjectValidator.objectToString(attributes.get(FacilityAttributeConstant.MANAGERNAME)));
			}
			
			if (attributes.containsKey(FacilityAttributeConstant.MANAGERCONTACT)) {
				node.setContact(ObjectValidator.objectToString(attributes.get(FacilityAttributeConstant.MANAGERCONTACT)));
			}
			
			// ホスト名
			if (attributes.containsKey(FacilityAttributeConstant.HOST)) {
				deleteNodeHostname(node);
				for (String hostname : ObjectValidator.objectToArrayListString(attributes.get(FacilityAttributeConstant.HOST))) {
					NodeHostnameUtil.getLocalHome().create(facilityId, hostname);
				}
			}
			
			// 備考
			if (attributes.containsKey(FacilityAttributeConstant.NOTE)) {
				deleteNodeNote(node);
				noteId = 0;
				for (String note : ObjectValidator.objectToArrayListString(attributes.get(FacilityAttributeConstant.NOTE))) {
					nodeNote = NodeNoteUtil.getLocalHome().create(facilityId, noteId);
					nodeNote.setNote(note);
					noteId++;
				}
			}
		} catch (Exception e) {
			m_log.warn("failure to set node's detail. (facilityId = " + facilityId + ")", e);
			throw new EJBException("failure to set node's detail. (facilityId = " + facilityId + ")", e);
		}
		
		// ファシリティの木情報のキャッシュを初期化する
		FacilitySelector.initCacheFacilityTree();
		
		m_log.info("successful in setting node's detail. (facilityId = " + facilityId + ")");
	}
	
	
}
