/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.quartzmanager.ejb.session;

import java.rmi.RemoteException;

import javax.ejb.EJBException;
import javax.ejb.SessionContext;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.quartz.JobDetail;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.Trigger;

import com.clustercontrol.quartzmanager.bean.QuartzConstant;

/**
 *
 * <!-- begin-user-doc --> Quartzの制御を行う session bean <BR>
 * 
 * 各種監視などから呼ばれ定周期の実行をquartzに設定・解除します。
 * <!-- end-user-doc --> *
 *
 * <!-- begin-xdoclet-definition --> 
 * @ejb.bean name="QuartzManager"	
 *           jndi-name="QuartzManager"
 *           type="Stateless" 
 *           transaction-type="Container"
 *                    
 * @ejb.permission
 *     unchecked="true"
 *           
 * <!-- end-xdoclet-definition --> 
 * @generated
 */
public abstract class QuartzManagerBean implements javax.ejb.SessionBean {
	protected static Log m_log = LogFactory.getLog( QuartzManagerBean.class ); 
	
	@SuppressWarnings("unused")
	private SessionContext m_context;
	
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}
	
	/**
	 * スケジュール登録を行います。<BR>
	 * 
	 * 各種監視などが１分間隔などで実行される場合
	 * このメソッドを利用してQuartzに監視を登録します。
	 * 
	 * @ejb.interface-method
     *     
	 * @version 2.0.0
	 * @since 2.0.0
	 * 
	 * @param job Quartzで実行される定義情報（Hinemosのジョブではない）
	 * @param trigger トリガー（実行制御情報）
	 * @throws NamingException
	 * @throws SchedulerException
	 * @see com.clustercontrol.monitor.run.factory.ModifySchedule
	 */
	public void addSchedule(JobDetail job, Trigger trigger) throws NamingException, SchedulerException{
		m_log.debug("addSchedule() : start");
		
		// QuartzのSchedulerを取得する
		Scheduler scheduler = null;
		try {
			scheduler = getScheduler();
		} catch (NamingException e) {
			m_log.warn("addSchedule() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}

		try{
			// Schedulerにジョブを登録する
			scheduler.scheduleJob(job, trigger);
		}catch(SchedulerException e){
			m_log.warn("addSchedule() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}
	}
	
    /**
     * スケジュールをQuartzから削除します。<BR>
     * 
     * 
     * @ejb.interface-method
     *     
	 * @version 2.0.0
	 * @since 2.0.0
     * 
     * @param jobName ジョブ名 (Hinemosの監視項目など)
     * @param jobGroup ジョブグループ ()
     * @throws NamingException
     * @throws SchedulerException
     * @see com.clustercontrol.monitor.run.factory.ModifySchedule
     */
    public void deleteSchedule(String jobName, String jobGroup) throws NamingException, SchedulerException {
        m_log.debug("deleteSchedule() : Job Group =" + jobGroup + "Job Name =" + jobName);
        
        // QuartzのSchedulerを取得する
        Scheduler scheduler = null;
		try {
			scheduler = getScheduler();
		} catch (NamingException e) {
			m_log.warn("deleteSchedule() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}

        try{
        	// Schedulerからジョブを削除する
        	scheduler.deleteJob(jobName, jobGroup);
        }catch(SchedulerException e){
			m_log.warn("deleteSchedule() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
        }
    }
    
    /**
	 * スケジュールを一時停止します。<BR>
	 * 
	 * @ejb.interface-method
     *     
	 * @version 2.0.0
	 * @since 2.0.0
	 * 
	 * @param jobName ジョブ名
     * @param jobGroup ジョブグループ
	 * @throws NamingException
	 * @throws SchedulerException 
	 */
	public void pauseSchedule(String jobName, String jobGroup) throws NamingException, SchedulerException {
		m_log.debug("deleteSchedule() : Job Group =" + jobGroup + "Job Name =" + jobName);
		
		// QuartzのSchedulerを取得する
		Scheduler scheduler = null;
		try {
			scheduler = getScheduler();
		} catch (NamingException e) {
			m_log.warn("pauseSchedule() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}

		try{
			// ジョブをポーズする
			scheduler.pauseJob(jobName, jobGroup);
		}catch(SchedulerException e){
			m_log.warn("pauseSchedule() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}
	}
    
    /**
	 * ジョブ（周期実行情報）詳細を取得します。<BR>
	 * 
     * @ejb.interface-method
     *     
	 * @version 2.0.0
	 * @since 2.0.0
     * 
	 * @param jobName ジョブ名
     * @param jobGroup ジョブグループ
     * @return ジョブ詳細
     * @throws NamingException
     * @throws SchedulerException
     */
	public JobDetail getJobDetail(String jobName, String jobGroup) throws NamingException, SchedulerException {
		m_log.debug("getJobDetail() : Job Group =" + jobGroup + "Job Name =" + jobName);
		
		// QuartzのSchedulerを取得する
		Scheduler scheduler = getScheduler();
	        
		// JobDetail取得
		JobDetail jobDetail = scheduler.getJobDetail(jobName, jobGroup);
	        
	    return jobDetail;
    }
    
    /**
	 * ジョブ(周期実行)名リストを取得します。<BR>
	 * 
     * @ejb.interface-method
     *     
	 * @version 2.0.0
	 * @since 2.0.0
     * 
	 * @param jobGroup ジョブグループ
     * @return ジョブ名リスト
     * @throws NamingException
     * @throws SchedulerException
     */
	public String[] getJobs(String jobGroup) throws NamingException, SchedulerException {
		m_log.debug("getJobs() : Job Group =" + jobGroup);
		
		// QuartzのSchedulerを取得する
		Scheduler scheduler = getScheduler();
	        
        //ジョブ名リスト取得
		String[] jobs = scheduler.getJobNames(jobGroup);
	        
	    return jobs;
    }
	
	/**
	 * QuartzSchedulerを取得します。<BR>
	 * 
	 * @ejb.interface-method
     *     
	 * @version 2.0.0
	 * @since 2.0.0
	 * 
	 * @return Scheduler
	 * @throws NamingException
	 */
	public Scheduler getScheduler() throws NamingException {

        //QuartzのSchedulerをルックアップ
		InitialContext context = new InitialContext();
		Scheduler scheduler = (Scheduler)context.lookup(QuartzConstant.QUARTZ_NAME);
		
		return scheduler;
	}
}
