/*
                                                                                                
Copyright (C) 2008 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
 
*/

package com.clustercontrol.poller;

import java.lang.reflect.Constructor;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import javax.ejb.FinderException;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.rmi.PortableRemoteObject;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.quartz.CronTrigger;
import org.quartz.JobDetail;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.impl.StdScheduler;

import com.clustercontrol.FacilityNotFoundException;
import com.clustercontrol.repository.bean.NodeInfo;
import com.clustercontrol.repository.factory.NodeProperty;
import com.clustercontrol.sharedtable.DataTable;
import com.clustercontrol.sharedtable.DataTableNotFoundException;
import com.clustercontrol.sharedtable.SharedTable;
import com.clustercontrol.vm.bean.VmMethodTypeConstant;
import com.clustercontrol.vm.ejb.entity.VmMethodMstLocal;
import com.clustercontrol.vm.ejb.entity.VmMethodMstPK;
import com.clustercontrol.vm.ejb.entity.VmMethodMstUtil;
import com.clustercontrol.vm.VmPollerImplInterface;
import com.clustercontrol.bean.PollerProtocolConstant;
import com.clustercontrol.bean.SnmpVersionConstant;
import com.clustercontrol.poller.impl.SnmpPollerImpl;
import com.clustercontrol.poller.impl.WbemPollerImpl;
import com.clustercontrol.poller.job.PollingJob;

public class PollingController {
	private static Log m_log = LogFactory.getLog( PollingController.class );
	
	private Object modifyLock = new Object();

	private final static String SHARED_TABLE_JNDI_NAME = "SharedTable";
	private final static String POLLER_MANAGER_JNDI_NAME = "PollerManager";
	
	private final static int RUNNING = 1;
	private final static int STOPPED = 2;
	
	private final String m_quartzJndiName = "QuartzRAM";
	private final String m_quartzJobName;  // ポーラ名とする
	private final String m_quartzGroupName;  // ポーラグループとする
	
	private final String m_pollerGroup;
	private final String m_pollerName;
	
	private final String m_targetFacilityId;
	
	private final PollingControllerConfig m_pollingConfig;
	
	private final String m_tableGroup;
	private final String m_tableName;
	
	// ポーラの状態を示す
	volatile private int m_status;
	
	public PollingController(
			String pollerGroup, 
			String pollerName, 
			String targetFacilityId,
			boolean indexCheckFlg,
			String tableGroup, 
			String tableName)
	throws NotInitializedException, DataTableNotFoundException{
		m_pollerGroup = pollerGroup;
		m_pollerName = pollerName;
		m_targetFacilityId = targetFacilityId;
		m_pollingConfig = new PollingControllerConfig(indexCheckFlg);
		m_tableGroup = tableGroup;
		m_tableName = tableName;

		m_quartzJobName = m_pollerName;
		m_quartzGroupName = m_pollerGroup;
		
		m_status = PollingController.STOPPED;
		
		try {
			// 書き込み対象テーブルが存在するか否かを確認する
			SharedTable sharedTable = lookupSharedTable();
			
			if(!sharedTable.containsDataTable(tableGroup, tableName)){
				// テーブルが存在しないため例外を投げる
				throw new DataTableNotFoundException(tableGroup, tableName);
			}
		} catch (NamingException e) {
			m_log.error("get SharedTable:" + e.getMessage());
			throw new NotInitializedException(
					"Table is not initialized. (" + tableName + " in " + tableGroup + " )");
		}
	}
	
	protected void scheduleJob(){
		// 収集間隔を取得する
		int interval = m_pollingConfig.getMinPollingInterval();

		// デバッグ出力
		m_log.debug("scheduleJob : " + interval);
		
		//QuartzのSchedulerをルックアップ
		Scheduler scheduler = null;
		try {
			InitialContext ctx = new InitialContext();
			Object obj = ctx.lookup(m_quartzJndiName);
			scheduler = (Scheduler)PortableRemoteObject.narrow(obj, StdScheduler.class);

		} catch (NamingException e) {
			m_log.error(e.getMessage(), e);
		}
		
		// スケジューラの取得に失敗した場合は終了
		if(scheduler == null){
			return;
		}
		
		// 収集間隔が-1の場合は新規に登録しない
		if(interval == -1){
			try {
				// 既に登録されているジョブを削除する(登録されていない場合は何もおこらない)
				scheduler.deleteJob(m_quartzJobName, m_quartzGroupName);

				// ステータス変更
				m_status = PollingController.STOPPED;
			} catch (SchedulerException e) {
				m_log.error(e.getMessage(), e);
			}
			return;
		}
		
		// 新規にジョブを生成
		JobDetail job = new JobDetail(m_quartzJobName, m_quartzGroupName, PollingJob.class);
		//ジョブ完了時に削除されないようにする
		job.setDurability(true);
		
		//JobDetailに変数を設定
		job.getJobDataMap().put("jndiName", POLLER_MANAGER_JNDI_NAME);
		job.getJobDataMap().put("pollerGroup", m_pollerGroup);
		job.getJobDataMap().put("pollerName", m_pollerName);
		
		
		//CronTriggerを作成
		CronTrigger cronTrigger = new CronTrigger(m_quartzJobName, m_quartzGroupName);

		//起動失敗した場合は、次の起動予定時刻をセットするように設定
		cronTrigger.setMisfireInstruction(CronTrigger.MISFIRE_INSTRUCTION_DO_NOTHING);

		try {
			String cronString = PollingInterval.parseCronExpression(interval);
			cronTrigger.setCronExpression(cronString);
		} catch (ParseException e) {
			// エラー処理
			m_log.error(e.getMessage(), e);
		}
		

		try {
			// 既に登録されているジョブを削除する(登録されていない場合は何もおこらない)
			scheduler.deleteJob(m_quartzJobName, m_quartzGroupName);

			// ジョブを登録する
			scheduler.scheduleJob(job, cronTrigger);

			// ジョブを実行する
			scheduler.triggerJob(m_quartzJobName, m_quartzGroupName);
			
			// ステータス変更
			m_status = PollingController.RUNNING;
		} catch (SchedulerException e) {
			// エラー処理
			m_log.error(e.getMessage(), e);
		}
	}
	
	protected void polling(){
		try {
			//QuartzのSchedulerをルックアップ
			InitialContext ctx = new InitialContext();
			Object obj = ctx.lookup(m_quartzJndiName);
			Scheduler scheduler = (Scheduler)PortableRemoteObject.narrow(obj, StdScheduler.class);

			// ジョブを実行する
			scheduler.triggerJob(m_quartzJobName, m_quartzGroupName);
			
			// ステータス変更
			m_status = PollingController.RUNNING;
		} catch (SchedulerException e) {
			// エラー処理
			m_log.error(e.getMessage(), e);
		} catch (NamingException e) {
			// エラー処理
			m_log.error(e.getMessage(), e);
		}
	}
	
	/**
	 * ポーリングを開始する
	 * 既に指定の収集名が登録されている場合は設定を上書きする（ver3.2以降）
	 * 
	 * @param collectorName 収集名
	 * @param interval 収集間隔
	 * @param maps 収集プロトコルとポーリング対象のマップ
	 * @throws CollectorAlreadyExistException 既に該当の収集名で収集が開始されている
	 */
	public void startPolling(
			String collectorName,
			int interval, 
			HashMap<String, List<String>> maps)
	throws CollectorAlreadyExistException{
		synchronized (modifyLock) {
			// 指定の収集名が既に登録されている場合も、上書きする（ver3.2より）
			if(m_pollingConfig.containsCollectorName(collectorName)){
				m_log.debug(collectorName + "is already exist.");
//				// 既に収集名が設定されているため例外を投げる
//				throw new CollectorAlreadyExistException(collectorName);
			}

			try {
				// 書き込み対象テーブルが存在するか否かを確認する
				SharedTable stable = lookupSharedTable();

				if(!stable.containsCollectorName(m_tableGroup, m_tableName, collectorName)){
					// ない場合は新規に作成する
					stable.registerCollector(m_tableGroup, m_tableName, collectorName, interval);
				}
			} catch (DataTableNotFoundException e) {
				m_log.error(e.getMessage(), e);
				return;
			} catch (NamingException e) {
				m_log.error(e.getMessage(), e);
				return;
			}
			
			// 指定の収集名で、収集間隔、収集プロトコルとポーリング対象のマップを登録する
			m_pollingConfig.putPollingTargets(collectorName, interval, maps);

			// ポーリングのスケジュールを設定
			scheduleJob();
		}
	}

	/**
	 * ポーリングを終了する
	 * @param collectorName 収集名
	 * @throws FacilityNotFoundExceptionBK
	 */
	public void stopPolling(String collectorName){
		synchronized (modifyLock) {
			m_log.debug("stop polling : " + m_pollerGroup + ", " + m_pollerName + ", " + collectorName);

			// 指定の収集名で、収集間隔、ポーリング対象のリストを削除する
			boolean modifyFlg = m_pollingConfig.removePollingTargets(collectorName);

			// 最小収集間隔が変更されている場合はスケジューリング
			if(modifyFlg){
				scheduleJob();
			}

			try {
				// 書き込み対象テーブルの登録情報を削除する
				SharedTable stable = lookupSharedTable();

				stable.unregisterCollector(m_tableGroup, m_tableName, collectorName);
			} catch (DataTableNotFoundException e) {
				m_log.error(e.getMessage(), e);
				return;
			} catch (NamingException e) {
				m_log.error(e.getMessage(), e);
				return;
			}
		}
	}
	
	/**
	 * 全ての収集名のポーリングを終了する
	 * @throws FacilityNotFoundExceptionBK
	 */
	public void stopPollingAll(){
		synchronized (modifyLock) {
			m_log.debug("stop polling : " + m_pollerGroup + ", " + m_pollerName);

			// 全ての収集間隔、ポーリング対象のリストを削除する
			boolean modifyFlg = m_pollingConfig.removePollingAllTargets();

			// 最小収集間隔が変更されている場合はスケジューリング
			if(modifyFlg){
				scheduleJob();
			}

			try {
				// 書き込み対象テーブルを削除する
				SharedTable sharedtable = lookupSharedTable();
				if(sharedtable != null){
					sharedtable.removeDataTable(m_tableGroup, m_tableName);
				}
			} catch (NamingException e) {
				m_log.error(e.getMessage(), e);
				return;
			}
		}
	}	
	
	/**
	 * ポーリングを実行する
	 */
	public void run() {
		// デバッグログ出力
		m_log.debug("run start : " + m_pollerGroup + "  " + m_pollerName);
	
		SharedTable sharedTable = null;
		// 収集値格納用テーブルをルックアップ
		try {
			// 共有テーブルをルックアップして取得する
			sharedTable = lookupSharedTable();
			if(sharedTable == null){
				// 書き込み対象テーブルを取得できないため終了する
				return;
			}
		} catch (NamingException e) {
			m_log.error(e.getMessage(), e);
		}
		
		try {
			/* ポーリングしても意味のないポーリング対象がないかチェックする */

			/* ポーラが管理している収集名のうち、書き込み対象である共有テーブルでは
			 * 既に参照対象でなくなっている収集名を特定する */
			
			// 共有テーブルで参照される収集名のセットを取得
			Set<String> retainCollectorNames = 
				new HashSet<String>(sharedTable.getCollectorNames(m_tableGroup, m_tableName));

			// ポーラで保持している収集名のセットを取得
			Set<String> pollerCollectorNames = m_pollingConfig.getCollectorNames();
			
			// 共通要素を抽出
			retainCollectorNames.retainAll(pollerCollectorNames);
			
			// 削除対象の収集名
			Set<String> removeCollectoName = 
				new HashSet<String>(m_pollingConfig.getCollectorNames());
			removeCollectoName.removeAll(retainCollectorNames);
			
			synchronized (modifyLock) {
				Iterator<String> itr = removeCollectoName.iterator();
				
				boolean modifyFlg = false;
				while(itr.hasNext()){
					String collectorName = itr.next();
					// 指定の収集名で、収集間隔、ポーリング対象のリストを削除する
					modifyFlg = m_pollingConfig.removePollingTargets(collectorName);
				}

				// 最小収集間隔が変更されている場合はスケジューリング
				if(modifyFlg){
					scheduleJob();
				}
			}
		} catch (DataTableNotFoundException e) {
			m_log.error(e.getMessage(), e);
			// 書き込み対象テーブルを取得できないため終了する
			return;
		}

		// 共有テーブルが保持しているデータホルダのうち有効である収集間隔のセットを取得
		Set<Integer> holderIntervals;
		try {
			holderIntervals = sharedTable.getIntervals(m_tableGroup, m_tableName);
		} catch (DataTableNotFoundException e) {
			m_log.error(e.getMessage(), e);
			// 書き込み対象テーブルを取得できないため終了する
			return;
		}
	
		// 更新すべきテーブルがない場合は処理終了
		if(holderIntervals.size() <= 0){
			return;
		}		
	
		// 現在時刻を取得
		long now = System.currentTimeMillis();

		// 今のタイミングで更新すべきテーブルを特定するため、
		// 更新の必要のある収集間隔のリストを取得する
		List<Integer> intervals = m_pollingConfig.getCurrentRefreshIntervals(now);
	
		ArrayList<Integer> tmpList = new ArrayList<Integer>(intervals);
	
		// 全エンティティ内を調べるループをまわすためテンポラリのリストを作成
		Iterator<Integer> itr = tmpList.iterator();
		while(itr.hasNext()){
			int interval = itr.next();
			// 存在しない場合は収集対象から削除する
			if(!holderIntervals.contains(interval)){
				m_log.debug("Remove : " + interval);
				intervals.remove(new Integer(interval));
			}
		}
	
		// 更新すべきテーブルがない場合は処理終了
		if(intervals.size() <= 0){
			return;
		}
	
		// ポーリング対象の中から今のタイミングで収集すべきものを抽出する
		HashMap<String, List<String>> pollingTargetMap = m_pollingConfig.getCurrentTargetMap(now);
	
		m_log.debug("run() pollingTargetMap : " + pollingTargetMap);
		// 収集すべきポーリング対象がない場合は処理終了
		if(pollingTargetMap.size() <= 0){
			m_log.debug("polling targets are nothing.");
			return;
		}
		
		// ポーリングにより取得した値を格納するテーブル
		DataTable dataTable = null;
		
		NodeInfo nodeInfo = null;
		try {
			nodeInfo = NodeProperty.getProperty(m_targetFacilityId);
		} catch (FacilityNotFoundException e1) {
			m_log.warn("facility not found 1. FacilityId = " + m_targetFacilityId);
			// ポーリングは実行しない
			// 取得値が空のテーブルを作成する
			dataTable = new DataTable();
		}
		if(nodeInfo == null) {
			m_log.warn("facility not found 2. FacilityId = " + m_targetFacilityId);
			// ポーリングは実行しない
			// 取得値が空のテーブルを作成する
			dataTable = new DataTable();
		} else if (nodeInfo.getFacilityId() == null || "".equals(nodeInfo.getFacilityId())){
			m_log.warn("invalid FacilityId. FacilityId = " + nodeInfo.getFacilityId());
			// ポーリングは実行しない
			// 取得値が空のテーブルを作成する
			dataTable = new DataTable();
		} else if (!nodeInfo.isValid()){
			m_log.debug("FacilityId = " + nodeInfo.getFacilityId() + " is disable.");
			// ポーリングは実行しない
			// 取得値が空のテーブルを作成する
			dataTable = new DataTable();
		}

		// ポーリングを行い値を収集
		if(dataTable == null){
			dataTable = polling(nodeInfo, pollingTargetMap, -1, -1);
		}
		m_log.debug("dataTable" + dataTable);
		
		// 収集値を共有テーブルに格納する
		for(int interval : intervals){
			// 新規収集した情報をテーブルの先頭に挿入する
			m_log.debug("insert " + m_tableGroup + ", " + m_tableName + ", " + interval);
			try {
				// テーブルホルダに収集値を挿入する
				// IPアドレスを識別キーとすることで、ファシリティで定義されているIPアドレスが変更になった場合に
				// 変更前と変更後の収集値が同じテーブルホルダに格納されることを防ぐ
				sharedTable.insertDataTable(
						m_tableGroup,
						m_tableName, 
						interval, 
						dataTable,
						nodeInfo.getAvailableIpAddress());
			} catch (DataTableNotFoundException e) {
				m_log.error(e.getMessage(), e);
			}
		}
	
		// デバッグログ出力
		m_log.debug("run end   : " + m_pollerGroup + "  " + m_pollerName);
	}

	private DataTable polling(
			NodeInfo nodeInfo, 
			HashMap<String, List<String>> pollingTargetMap, 
			int retries,
			int timeout){
		DataTable dataTable = new DataTable();
		DataTable tmpDataTable = null;
		
		// *****************************
		// SNMPのpollerの設定
		// *****************************
		if(pollingTargetMap.get(PollerProtocolConstant.PROTOCOL_SNMP) != null) {
			SnmpPollerImpl poller = new SnmpPollerImpl();
			
			// joeSNMPではint型でSNMPのバージョンを定義
			// 0:SNMP V1 protocol, 1:SNMP V2 protocol
			int snmpVersionForJoeSNMP = 1;
			String versionStr = nodeInfo.getSnmpVersion();
			m_log.debug("version   : " + versionStr);
			if(versionStr != null && versionStr != ""){
				Integer version = SnmpVersionConstant.stringToType(versionStr);
				if(versionStr != null){
					// 未定義(-1)の場合はv2cに設定
					if(version == -1){
						version = 1;
					}
					
					snmpVersionForJoeSNMP = version;
				}
			}
				
			tmpDataTable = poller.polling(
					nodeInfo.getAvailableIpAddress(), 
					nodeInfo.getSnmpPort(),
					snmpVersionForJoeSNMP,
					nodeInfo.getSnmpCommunity(),
					retries == -1 ? nodeInfo.getSnmpRetryCount() : retries,
					timeout == -1 ? nodeInfo.getSnmpTimeout() : timeout,
					pollingTargetMap.get(PollerProtocolConstant.PROTOCOL_SNMP), 
					m_pollingConfig.isIndexCheckFlg());
			dataTable.putAll(tmpDataTable);
		}
		
		// *****************************
		// WBEMのpollerの設定
		// *****************************
		if(pollingTargetMap.get(PollerProtocolConstant.PROTOCOL_WBEM) != null) {
			WbemPollerImpl poller = new WbemPollerImpl();
			tmpDataTable = poller.polling(
					nodeInfo.getAvailableIpAddress(), 
					nodeInfo.getWbemPort(),
					nodeInfo.getWbemProtocol(),
					nodeInfo.getWbemUser(),
					nodeInfo.getWbemUserPassword(),
					nodeInfo.getWbemNameSpace(),
					retries == -1 ? nodeInfo.getWbemRetryCount() : retries,
					timeout == -1 ? nodeInfo.getWbemTimeout() : timeout,
					pollingTargetMap.get(PollerProtocolConstant.PROTOCOL_WBEM));
			dataTable.putAll(tmpDataTable);
		}
		
		// *****************************
		// 仮想化のpollerの設定
		//   プロトコルがSNMP,WBEM以外の場合(VMware,XENなど)は下記が処理される。
		// *****************************
		Iterator<String> pollerProtocolItr = pollingTargetMap.keySet().iterator();
		while (pollerProtocolItr.hasNext()) {
			String pollerProtocol = pollerProtocolItr.next();
			m_log.debug("pollerProtocol : " + pollerProtocol);
			if (pollerProtocol.equals(PollerProtocolConstant.PROTOCOL_SNMP) ||
					pollerProtocol.equals(PollerProtocolConstant.PROTOCOL_WBEM)) {
				continue;
			}
			VmMethodMstLocal local = null;
			try {
				local = VmMethodMstUtil.getLocalHome().findByPrimaryKey(new VmMethodMstPK(
					pollerProtocol, VmMethodTypeConstant.POLLERIMPL));
			} catch (FinderException e) {
				e.printStackTrace();
			} catch (NamingException e) {
				e.printStackTrace();
			} 
			if(local == null){
				m_log.warn(pollerProtocol + " : Execution Class is not registered. (" +
						VmMethodTypeConstant.POLLERIMPL +
						")");
				continue;
			}
			
			// pollerProtocolの実行クラス名をcc_vm_solution_mstから取得。
			// 存在していない場合、中止。
			String className = local.getClassName();
			m_log.debug("action() : className = " + className);
			if(className == null) {
				m_log.warn(pollerProtocol + " : Execution Class is not registered.");
				continue;
			}
			
			// コンストラクタの生成
			// 失敗したら、中止。
			Constructor pollerConstructor = null;
			try {
				pollerConstructor = Class.forName(className).getConstructor();
			} catch (SecurityException e) {
				m_log.error(e.getMessage(), e);
			} catch (NoSuchMethodException e) {
				m_log.error(e.getMessage(), e);
			} catch (ClassNotFoundException e) {
				m_log.warn(className + " is not poller class");
				continue;
			}
			if (pollerConstructor == null) {
				m_log.warn("action() : pollerConstructor is null");
				continue;
			}
				
			// ポーラーの取得。
			// 取得できたら、ポーリング実行。
			VmPollerImplInterface poller = null;
			try {
				poller = (VmPollerImplInterface) pollerConstructor.newInstance();
			} catch (Exception e) {
				e.printStackTrace();
			}
			if (poller == null) {
				m_log.warn("action() : poller is null");
				continue;
			}
			
			String vmName = nodeInfo.getVmName();
			m_log.debug("vmname : " + vmName);
			
			String vmNodeType = nodeInfo.getVirtualizationNodeType();
			m_log.debug("vmNodeType : " + vmNodeType);

			// ホストノードから値を取得する
			NodeInfo targetNodeInfo = null;
			if(vmNodeType != null) {
				if(vmNodeType.equals("host")) {
					// ホストノードのパラメータを用いる
					targetNodeInfo = nodeInfo;
				} else if(vmNodeType.equals("guest")){
					// ゲストノード用ポーリングパラメータを用いる
					try {
						targetNodeInfo = NodeProperty.getProperty(nodeInfo.getVmManagementNode());
					} catch (FacilityNotFoundException e) {
						m_log.warn("facility not found 3. FacilityId = " + m_targetFacilityId);
						// ポーリングは実行しない
						continue;
					}
				}
			}

			poller.init(pollerProtocol);
			tmpDataTable = poller.polling(
					nodeInfo.getAvailableIpAddress(),
					nodeInfo.getVirtualizationNodeType(),
					targetNodeInfo.getAvailableIpAddress(),
					nodeInfo.getVmName(),
					targetNodeInfo.getVmUser(),
					targetNodeInfo.getVmUserPassword(),
					targetNodeInfo.getVmProtocol(),
					pollingTargetMap.get(pollerProtocol));
			dataTable.putAll(tmpDataTable);
		}
		return dataTable;
	}

	/**
	 * 指定のOIDに対してポーリングを実行します。
	 * 
	 * @param targetValues ポーリング対象
	 * @return 収集したMIB値が格納されたデータテーブル
	 */
	public DataTable polling(HashMap<String, List<String>> targetMaps, int retries, int timeout){
		// 各ポーラーで共通で使用するDataTableを定義する
		DataTable dataTable = new DataTable();
		
		NodeInfo nodeInfo;
		try {
			nodeInfo = NodeProperty.getProperty(m_targetFacilityId);
		} catch (FacilityNotFoundException e) {
			m_log.warn("facility not found 4. FacilityId = " + m_targetFacilityId);
			// ポーリングは実行しない
			return dataTable;
		}
		
		if(nodeInfo == null) {
			m_log.warn("facility not found 5. FacilityId = " + nodeInfo.getFacilityId());
			// ポーリングは実行しない
			return dataTable;
		} else if (nodeInfo.getFacilityId() == null || "".equals(nodeInfo.getFacilityId())){
			m_log.warn("invalid FacilityId. FacilityId = " + nodeInfo.getFacilityId());
			// ポーリングは実行しない
			return dataTable;
		}
		
		dataTable.putAll(polling(nodeInfo, targetMaps, retries, timeout));
		
		return dataTable;
	}

	protected SharedTable lookupSharedTable() throws NamingException{
		try {
			// 指定の値格納用テーブルが存在するか確認する
			InitialContext ctx = new InitialContext();
			
			Object obj = ctx.lookup(SHARED_TABLE_JNDI_NAME);

			SharedTable table = 
				(SharedTable)PortableRemoteObject.narrow(obj, SharedTable.class);
			
			return table;
		} catch (NamingException e) {
			m_log.error(e.getMessage(), e);
			throw e;
		}
	}
	
	/**
	 * ポーラグループ名を返します。
	 * @return ポーラグループ名
	 */
	public String getPollerGroup() {
		return m_pollerGroup;
	}

	/**
	 * ポーラ名を返します。
	 * @return ポーラ名
	 */
	public String getPollerName() {
		return m_pollerName;
	}
	
	/**
	 * ポーリング設定を返します。
	 * @return ポーリング設定
	 */
	public PollingControllerConfig getPollingConfig() {
		return m_pollingConfig;
	}

	public String getQuartzJndiName() {
		return m_quartzJndiName;
	}

	/**
	 * ポーラの実行状態を返します。
	 * @return 実行状態
	 */
	public int getStatus() {
		return m_status;
	}
}
