/*
                                                                                                                                                                 
Copyright (C) 2006 NTT DATA Corporation
                                                                                                                                                                 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
                                                                                                                                                                 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
                                                                                                                                                                 
*/

package com.clustercontrol.performance.job;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.quartz.Job;
import org.quartz.JobDataMap;
import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;

import com.clustercontrol.performance.bean.QuartzConstant;
import com.clustercontrol.performance.util.JNDIConnectionManager;
import com.clustercontrol.performance.bean.CollectorType;
import com.clustercontrol.performance.ejb.bmp.CollectorLocal;
import com.clustercontrol.performance.ejb.bmp.CollectorPK;
import com.clustercontrol.performance.ejb.bmp.CollectorUtil;
import com.clustercontrol.performance.ejb.session.CollectorControllerLocal;
import com.clustercontrol.performance.ejb.session.CollectorControllerUtil;

/**
 * Quartzから呼び出され収集を実行するクラス
 *
 */
public class CollectJob implements Job{
	private static Log m_log = LogFactory.getLog( CollectJob.class );

	// 収集から値計算までの間隔（収集間隔1分以下の設定）
	private final static int _delayTimeUnder1min = QuartzConstant.getDelayTimeUnder1min();
	
	// 収集から値計算までの間隔（収集間隔1分以上の設定）
	private final static int _delayTimeOver1min = QuartzConstant.getDelayTimeOver1min();
	
	/**
	 * Quartzから呼び出され、リアルタイム、実績、リソース監視の各収集方法に応じてEntityBeanのメソッドを呼び出す。
	 */
	public void execute(JobExecutionContext context) throws JobExecutionException {
		// デバッグログ出力
		m_log.debug("CollectJob execute : start");
		
		// 引数を取得します
		JobDataMap dataMap = context.getJobDetail().getJobDataMap();

		CollectorPK pk = (CollectorPK) dataMap.get("pk");
		int collectorType = dataMap.getInt("type");
		
		CollectorLocal bean = null;
		try {
			// デバッグログ出力
			m_log.debug("CollectJob execute : " + pk + "  type : " + collectorType);

			// 収集用EntityBeanをルックアップ
			try {
				// リポジトリ情報の変更をチェックし、変更があった場合はスコープツリー情報を更新する
				try {
					CollectorControllerLocal ctrl = CollectorControllerUtil
							.getLocalHome().create();
					ctrl.refreshScopeTree();
				} catch (CreateException e) {
					m_log.error(e.getMessage(), e);
				} catch (NamingException e) {
					m_log.error(e.getMessage(), e);
				}

				bean = CollectorUtil.getLocalHome().findByPrimaryKey(pk);

				if (bean.getCollectorID() == null
						|| collectorType == CollectorType.UNKNOWN) {
					// コレクタが正常に動作していない
					m_log.warn("CollectorBean is not collectly initialized ");
				} else if (collectorType == CollectorType.REALTIME) {
					// リアルタイム収集の場合

					// 設定回数まで参照されていない場合は収集を停止する。
					if (!bean.checkLifetime()) {
						m_log.debug("stop collect : " + pk.collectorID);
						bean.stopCollect();
						bean.remove();
					} else {
						// DBへの出力は行わずポーラーからの値取得のみ行う
						bean.fetchMibValue();
					}
				} else if (collectorType == CollectorType.RECORD) {
					// 実績収集の場合

					// 終了予定時刻を過ぎている場合は終了する
					long now = System.currentTimeMillis();
					long interval = bean.getIntervalSec() * 1000l;

					// もっとも最近SNMPポーリングを実行したであろう時刻
					// 実際にはJBossが停止していた等で実行されていない可能性がある
					long lastPollingTime = (now / interval) * interval;
					
					if (bean.getScheduledStopDate().getTime() < lastPollingTime) {
						m_log.debug("stop collect : " + pk.collectorID);
						bean.stopCollect();
					} else {
						// ポーラーからの値取得後にDBに出力する
						bean.storeCalcValue();
					}
				} else if (collectorType == CollectorType.MONITOR) {
					// 閾値監視の場合

					// ポーラーからの値取得後に閾値判定する
					bean.monitorCalcValue();
				} else {
					// 何もしない
				}
			} catch (NamingException e) {
				m_log.error(e);
				// リアルタイム収集の場合
				if (collectorType == CollectorType.REALTIME) {
					// Quartzの収集ジョブを停止
					deleteJob(pk.getCollectorID());
				}
			} catch (FinderException e) {
				m_log.error(pk.collectorID + " is already stopped.");
				m_log.error(e);
				// リアルタイム収集の場合
				if (collectorType == CollectorType.REALTIME) {
					// Quartzの収集ジョブを停止
					deleteJob(pk.getCollectorID());
				}
			} catch (EJBException e) {
				m_log.error(e);
			} catch (RemoveException e) {
				m_log.error(e);
				// リアルタイム収集の場合
				if (collectorType == CollectorType.REALTIME) {
					// Quartzの収集ジョブを停止
					deleteJob(pk.getCollectorID());
				}
			}
		} finally {
			if(bean != null){
				m_log.debug("LastCollectDate=" + bean.getLastCollectDate());
				m_log.debug("IntervalSec=" + bean.getIntervalSec());
				m_log.debug("ScheduledStopDate=" + bean.getScheduledStopDate());
				
				// 次回の収集予定時刻が、収集終了時刻を過ぎている場合は停止させる
				if((bean.getLastCollectDate().getTime() + bean.getIntervalSec() * 1000)
						 > bean.getScheduledStopDate().getTime()){
					m_log.debug("call stopCollect()");
					bean.stopCollect();
				}
			}
			
		}
		
		// デバッグログ出力
		m_log.debug("CollectJob execute : end " + pk);
	}
	
	/**
	 * 収集起動ジョブを削除します。
	 */
	private void deleteJob(String collectorId) {
		m_log.debug("deleteJob() start : " + collectorId);

		//QuartzのSchedulerをルックアップ
		try {
			InitialContext iniCtx = JNDIConnectionManager.getInitialContext();
			Scheduler scheduler = (Scheduler)iniCtx.lookup(QuartzConstant.getQuartzName());

			// 既に登録されているジョブを削除する(登録されていない場合は何もおこらない)
			scheduler.deleteJob(collectorId, QuartzConstant.GROUP_COLLECT);
		} catch (NamingException e) {
			throw new EJBException(e);
		} catch (SchedulerException e) {
			m_log.error(e);
		}

		m_log.debug("deleteJob() end   : " + collectorId);
	}
}