/*
 
 Copyright (C) 2009 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.notify.util;

import java.sql.Timestamp;
import java.util.Date;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.OutputBasicInfo;
import com.clustercontrol.commons.util.HinemosProperties;
import com.clustercontrol.notify.ejb.entity.MonitorStatusLocal;
import com.clustercontrol.notify.ejb.entity.MonitorStatusPK;
import com.clustercontrol.notify.ejb.entity.MonitorStatusUtil;
import com.clustercontrol.repository.ejb.entity.FacilityUtil;

public class MonitorResultStatusUpdater {
	/** ログ出力のインスタンス */
	private static Log m_log = LogFactory.getLog( MonitorResultStatusUpdater.class );
	
	/**
	 * 同一重要度カウンタの最大値（この値を超えた場合は、Longの最大値が設定され、以降は更新されない）
	 */
	private static long MAX_COUNT = 100;
	
	// プロパティファイルの読み込み
	static{
		try {
			MAX_COUNT = Integer.parseInt(HinemosProperties.getProperty("notify.initial.count.max", "100"));
		} catch (Exception e) {
			m_log.warn(e.getMessage(), e);
		} 
	}
	
	/**
	 * 直前の監視結果と現在の監視結果の重要度を比較し、変更がある場合は、DBで保持している同一重要度カウンタをリセット。
	 * 戻り値として、trueを返す。
	 * 重要度変化がない場合は、同一重要度カウンタをカウントアップし、DB情報を更新。
	 * 戻り値として、falseを返す。
	 * 
	 * @param facilityId ファシリティID
	 * @param pluginId プラグインID
	 * @param monitorId 監視項目ID
	 * @param generateDate 監視結果生成時刻
	 * @param currentPriority 現在の監視結果の重要度
	 * @return 重要度変化の有無（有:true / 無:false）
	 */
	private static boolean update(String facilityId, String pluginId, String monitorId, Date generateDate, int currentPriority){
		boolean prioityChangeFlag = true;
		MonitorStatusLocal monitorStatus = null;
		
		MonitorStatusPK pk = new MonitorStatusPK(facilityId, pluginId, monitorId);
		
		try {
			try {
				// 比較対象となる直前の重要度を保持する監視結果ステータス情報を検索
				monitorStatus = MonitorStatusUtil.getLocalHome().findByPrimaryKey(pk);

				// 検索の結果該当タプルが存在した場合

				// 重要が変化しているか確認
				if(currentPriority != monitorStatus.getPriority()){
					// 重要度が変化している場合
					prioityChangeFlag = true;
					if(m_log.isDebugEnabled()){
						m_log.debug("prioityChangeFlag = true. " + pk + " ," + monitorStatus.getPriority() + " to " + currentPriority);
					}

					// 重要度を更新
					monitorStatus.setPriority(currentPriority);

					// 同一重要度カウンタを0にリセット
					monitorStatus.setCounter(0l);
				} else {
					// 重要は変化していない
					prioityChangeFlag = false;
				}
			} catch (FinderException e) {
				// 検索の結果該当タプルが存在しない場合
				m_log.debug("create new entity. " + pk);
				
				// 今までの監視結果の重要度がないものについては
				// 重要度変化フラグはtrueに設定する
				prioityChangeFlag = true;

				try {
					FacilityUtil.getLocalHome().findByPrimaryKey(facilityId);
				} catch (FinderException e1) {
					// 出力先のファシリティIDが存在しない場合
					// 未登録のファシリティIDで通知される場合もある
					// その場合ここを通るため、エラーログは出さない。
					m_log.debug(e1.getMessage() + " FacilityId = " + facilityId);
					return prioityChangeFlag;
				}

				// 新規タプルを生成
				try {
					// 同一重要度カウンタは0で生成
					monitorStatus = MonitorStatusUtil.getLocalHome().create(
							facilityId, pluginId, monitorId, currentPriority, new Timestamp(generateDate.getTime()), 0l);
				} catch (CreateException e1) {
					m_log.error(e.getMessage(), e);
					return prioityChangeFlag;
				}
			}
		} catch (NamingException e) {
			m_log.error(e.getMessage(), e);
			return true;
		}

		// 同一重要度カウンタをカウントアップ
		long oldCount = monitorStatus.getCounter();

		// 最大カウント数を超えると、longの最大値を設定しそれ以上は増やさない（DBへのupdateを減らすための方策）
		if(oldCount <= MAX_COUNT){
			monitorStatus.setCounter(oldCount+1);
			monitorStatus.setLastUpdate(new Timestamp(generateDate.getTime()));
		}

		return prioityChangeFlag;
	}
	
	public static boolean update(OutputBasicInfo output){
		return update(output.getFacilityId(), output.getPluginId(), output.getMonitorId(), output.getGenerationDate(), output.getPriority());
	}
}
