/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.monitor.run.factory;

import java.util.Collection;
import java.util.Iterator;
import java.util.Set;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import javax.ejb.CreateException;
import javax.jms.JMSException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.ProcessConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.monitor.run.bean.MonitorJudgementInfo;
import com.clustercontrol.monitor.run.bean.MonitorStringValueInfo;
import com.clustercontrol.monitor.run.ejb.entity.MonitorStringValueInfoLocal;
import com.clustercontrol.notify.bean.NotifyRelationInfo;
import com.clustercontrol.notify.ejb.entity.NotifyRelationInfoUtil;

/**
 * 文字列監視を実行する抽象クラス<BR>
 * <p>
 * 文字列監視を行う各監視管理クラスで継承してください。
 *
 * @version 3.0.0
 * @since 2.1.0
 */
abstract public class RunMonitorStringValueType extends RunMonitor{
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog( RunMonitorStringValueType.class );
	
	/** 監視取得値 */
	protected String m_value;
	
	/** 通知グループID */
	protected String m_separateNotifyGroupId;
	
	
	/**
	 * コンストラクタ。
	 * 
	 * @throws NamingException
	 * @throws JMSException
	 * @throws CreateException
	 */
	public RunMonitorStringValueType() throws NamingException, JMSException, CreateException{
		super();
	}
	
	/* (非 Javadoc)
	 * @see com.clustercontrol.monitor.run.factory.RunMonitor#collect(java.lang.String)
	 */
	@Override
	public abstract boolean collect(String facilityId);
	
	/**
	 * 判定結果を返します。
	 * <p>
	 * 判定情報マップにセットしてある各順序のパターンマッチ表現から、
	 * 監視取得値がどのパターンマッチ表現にマッチするか判定し、マッチした順序を返します。
	 * 
	 * @see com.clustercontrol.monitor.run.ejb.entity.MonitorStringValueInfoBean#getOrderNo()
	 * @see com.clustercontrol.monitor.run.bean.MonitorStringValueInfo
	 */
	@Override
	public int getCheckResult(boolean ret) {
		
		int result = -1;
//		int result = m_failurePriority;
		boolean matchFlg = false;
		m_separateNotifyGroupId = m_notifyGroupId;
				
		// 値取得の成功時
		if(ret){
			MonitorStringValueInfo info = null;
			Pattern pattern = null;
			Matcher matcher = null;
			
			// 文字列監視判定情報で順番にフィルタリング
			Set set = m_judgementInfoList.keySet();
			for (Iterator iter = set.iterator(); iter.hasNext();) {
				Integer key = (Integer)iter.next();
				info = (MonitorStringValueInfo) m_judgementInfoList.get(key);
				
				// この設定が有効な場合
				if (info != null && info.isValidFlg()) {
					try {
						String patternText = info.getPattern();
						
						pattern = Pattern.compile(patternText, Pattern.DOTALL);
						matcher = pattern.matcher(m_value);
						
						// パターンマッチ表現でマッチング
						if (matcher.matches()) {
							
							result = info.getOrderNo();
							
							// 条件に一致したら処理する場合
							if (info.getProcessType() == ProcessConstant.TYPE_YES) {
								
								m_log.debug("getCheckResult() true : description=" + info.getDescription() + ", value=" + m_value);
								m_log.debug("getCheckResult() true : messageId=" + info.getMessageId() + ", message=" + info.getMessage());
								
								// 個別の通知ID指定
								if(info.getNotifyGroupId() != null && !"".equals(info.getNotifyGroupId())){
								
									// 通知グループIDから通知設定のコレクションを取得
									Collection<NotifyRelationInfo> notify = NotifyRelationInfoUtil.getLocalHome().findByNotifyGroupId(info.getNotifyGroupId());
									
									// 通知設定のコレクションが存在する場合は、通知グループIDを上書きする
									if(notify != null && notify.size() != 0){
										m_separateNotifyGroupId = info.getNotifyGroupId();
									}
									
								}
								matchFlg = true;
								break;
								
							// 条件に一致したら処理しないならば、処理終了
							} else {
								m_log.debug("getCheckResult():" + info.getDescription() + ":" + m_value);
								break;
							}
						}
					} catch(PatternSyntaxException e){
						m_log.error("getCheckResult(): PatternSyntax is not valid. description="+info.getDescription() + ", patternSyntax="+info.getPattern() + ", value=" + m_value, e);
						
					} catch (Exception  e) {
						m_log.error("getCheckResult(): PatternSyntax is not valid. description="+info.getDescription() + ", patternSyntax="+info.getPattern() + ", value=" + m_value, e);
					}
				}
			}
			if(!matchFlg){
				// 条件に一致したら処理しない または 条件に一致しない場合、通知しない
				m_separateNotifyGroupId = null; 
			}
		}
		return result;
	}

	/**
	 * 判定情報を設定します。
	 * <p>
	 * <ol>
	 * <li>監視情報より判定情報を取得します。</li>
	 * <li>取得した文字列監視の判定情報を、順序をキーに判定情報マップにセットします。</li>
	 * </ol>
	 * 
	 * @see com.clustercontrol.monitor.run.ejb.entity.MonitorStringValueInfoBean#getOrderNo()
	 * @see com.clustercontrol.monitor.run.bean.MonitorStringValueInfo
	 */
	@Override
	public void setJudgementInfo() {
		m_log.debug("setJudgementInfo() start");
		
		// 文字列監視判定値、ログ出力メッセージ情報を取得
		Collection ct = m_monitor.getMonitorStringValueInfo();
        Iterator itr = ct.iterator();
		
        m_judgementInfoList = new TreeMap<Integer, MonitorJudgementInfo>();
        MonitorStringValueInfoLocal local = null;
        while(itr.hasNext()){
        	
            local = (MonitorStringValueInfoLocal)itr.next();
            Integer order = local.getOrderNo().intValue();
            
            MonitorStringValueInfo info = new MonitorStringValueInfo();
            info.setOrderNo(order.intValue());
            info.setDescription(local.getDescription());
            info.setPattern(local.getPattern());
            info.setProcessType(local.getProcessType().intValue());
            info.setNotifyGroupId(local.getNotifyGroupId());
         //   info.setNotifyId(local.getNotifyId());
            info.setPriority(local.getPriority().intValue());
            info.setMessageId(local.getMessageId());
            info.setMessage(local.getMessage());
          /*  info.setJobRun(local.getJobRun().intValue());
            info.setJobId(local.getJobId());
            info.setJobInhibitionFlg(local.getJobInhibitionFlg().intValue());
            info.setJobFailurePriority(local.getJobFailurePriority().intValue());*/
            info.setValidFlg(ValidConstant.typeToBoolean(local.getValidFlg().intValue()));

            m_judgementInfoList.put(order, info);
            
            if(m_log.isDebugEnabled()){
        		m_log.debug("setJudgementInfo() MonitorStringValue OrderNo = " + order.intValue());
        		m_log.debug("setJudgementInfo() MonitorStringValue Description = " + local.getDescription());
        		m_log.debug("setJudgementInfo() MonitorStringValue Pattern = " + local.getPattern());
        		m_log.debug("setJudgementInfo() MonitorStringValue ProcessType = " + local.getProcessType().intValue());
        		m_log.debug("setJudgementInfo() MonitorStringValue NotifyGroupId = " + local.getNotifyGroupId());
        		m_log.debug("setJudgementInfo() MonitorStringValue Priority = " + local.getPriority().intValue());
        		m_log.debug("setJudgementInfo() MonitorStringValue MessageId = " + local.getMessageId());
        		m_log.debug("setJudgementInfo() MonitorStringValue Message = " + local.getMessage());
        		m_log.debug("setJudgementInfo() MonitorStringValue ValidFlg = " + ValidConstant.typeToBoolean(local.getValidFlg().intValue()));
            }
        }

		m_log.debug("setJudgementInfo() end");
	}
	
	/**
	 * 通知IDを返します。
	 * <p>
	 * 個々の文字列監視の判定情報に通知IDが指定されていない場合は、
	 * 監視情報の通知ID（デフォルト）を返します。<BR>
	 * 条件に一致したら処理しない または 条件に一致しない場合は、<code>null</code>を返します。
	 */
	@Override
	public String getNotifyGroupId() {
		return m_separateNotifyGroupId;
	}
}
