/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.monitor.factory;

import java.util.ArrayList;
import java.util.Date;
import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.FacilityNotFoundException;
import com.clustercontrol.HinemosUnknownException;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.monitor.bean.ScopeInfoData;
import com.clustercontrol.notify.monitor.ejb.entity.EventLogLocal;
import com.clustercontrol.notify.monitor.ejb.entity.StatusInfoLocal;
import com.clustercontrol.monitor.ejb.session.MonitorControllerBean;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * スコープ情報を検索するクラス<BR>
 *
 * @version 1.0.0
 * @since 1.0.0
 */
public class SelectScope {
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog( SelectScope.class );
	
	/**
	 * スコープ情報一覧を返します。
	 * <p>
	 * <ol>
	 * <li>引数で指定されたファシリティ(A)の直下のファシリティ（B）を取得します。</li>
	 * <li>取得したファシリティ（B）の配下全てのファシリティ（C）を取得します。</li>
	 * <li>ファシリティ（C）に属するステータス情報／イベント情報から重要度が最高で受信日時が最新のログを取得し、
	 *     ファシリティ（B）のスコープ情報とします。</li>
	 * <li>ファシリティ（B）の重要度が最高で受信日時が最新のログを、ファシリティ（A）のスコープ情報とします。</li>
	 * <li>この１スコープ情報をScopeInfoDataとし、スコープ情報一覧を保持するリスト（{@link ArrayList}）に格納し返します。<BR>
	 * </li>
	 * </ol>
	 * 
	 * @param facilityId 取得対象の親ファシリティID
	 * @return スコープ情報一覧（ScopeInfoDataが格納されたList）
	 * @throws CreateException
	 * @throws NamingException
	 * @throws HinemosUnknownException 
	 * @throws FacilityNotFoundException 
	 * @throws FinderException 
	 * 
	 * @see com.clustercontrol.monitor.bean.ScopeInfoData
	 * @see com.clustercontrol.monitor.factory.SelectStatus#getHighPriorityStatus(String, int)
	 * @see com.clustercontrol.monitor.factory.SelectEvent#getHighPriorityEvent(String, int)
	 * @see com.clustercontrol.monitor.bean.ScopeTabelDefine
	 */
	
	public ArrayList<ScopeInfoData> getScopeList(String facilityId)
			throws CreateException, NamingException, FacilityNotFoundException, HinemosUnknownException, FinderException {
		return getScopeList(facilityId, true, true, true);
	}	
	
	public ArrayList<ScopeInfoData> getScopeList(String facilityId,
			boolean statusFlag, boolean eventFlag, boolean orderFlg)
			throws CreateException, NamingException, FacilityNotFoundException, HinemosUnknownException {
		
		ArrayList<ScopeInfoData> list = new ArrayList();
		
		SelectStatus selectStatus = new SelectStatus();
		SelectEvent selectEvent = new SelectEvent();
		
		int parentPriority = PriorityConstant.TYPE_NONE;
		String parentFacilityId = null;
		String parentFacilityPath = null;
		Date parentOutputDate = null;
		
		try 
		{
			RepositoryControllerLocal repository = RepositoryControllerUtil.getLocalHome().create();
			
			//ソートの順位を取得するために、ホームインターフェースを取得しておく。
			// FacilityTreeLocalHome ftHome =  FacilityTreeUtil.getLocalHome();
			
			// 指定したファシリティの重要度が一番高い情報を取得
			if(facilityId != null && !"".equals(facilityId)){
				// 重要度が一番高いステータス情報を取得
				StatusInfoLocal status = null;
				m_log.debug("flag " + statusFlag + "," + eventFlag);
				if (statusFlag) {
					status = selectStatus.getHighPriorityStatus(facilityId, MonitorControllerBean.ONLY, orderFlg);
				}
				
				// 重要度が一番高いイベントログを取得
				EventLogLocal event = null;
				if (eventFlag) {
					event = selectEvent.getHighPriorityEvent(facilityId, MonitorControllerBean.ONLY, orderFlg);
				}
				
				// 重要度が一番高い情報を設定する
//				ArrayList info = this.getHighPriorityScope(status, event, repository, facilityId, facilityId);
//
//				if(info != null){
//					parentPriority = ((Integer)info.get(ScopeTabelDefine.PRIORITY)).intValue();
//					parentFacilityId = (String)info.get(ScopeTabelDefine.FACILITY_ID);
//					parentFacilityPath = (String)info.get(ScopeTabelDefine.SCOPE);
//					parentOutputDate = (Date)info.get(ScopeTabelDefine.UPDATE_TIME);
//				}
				
				ScopeInfoData scopeInfo = this.getHighPriorityScope(status, event, repository, facilityId, facilityId);
				if (scopeInfo != null) {
					parentPriority = scopeInfo.getPriority().intValue();
					parentFacilityId = scopeInfo.getFacilityId();
					parentFacilityPath = scopeInfo.getFacilityPath();
					parentOutputDate = scopeInfo.getOutputDate();
					
				}
				
				
			}
			
			
			// 直下のファシリティIDを取得
			ArrayList facilityIdList = repository.getFacilityIdList(facilityId, MonitorControllerBean.ONE_LEVEL);
			
			String[] facilityIds = null;
			if(facilityIdList != null && facilityIdList.size() > 0){
				// スコープの場合
				facilityIds = new String[facilityIdList.size()];
				facilityIdList.toArray(facilityIds);
			}
			else{
				if(facilityId != null){
					// ノードの場合
					facilityIds = new String[1];
					facilityIds[0] = facilityId;
				}
				else{
					// リポジトリが1件も登録されていない場合
					return null;
				}
			}
			
			
			// 直下のファシリティIDに属する配下全てのファシリティIDより、各々重要度が一番高い情報を取得
			boolean statusInfoFlg = false;
			boolean eventLogFlg = false;
			boolean highPriorityFlg = false;
			for(int index=0; ((facilityIds!=null) && (index<facilityIds.length)); index++){
				
				// 直下のファシリティの場合
				if(!facilityIds[index].equals(facilityId)){
					
					// 重要度が一番高いステータス情報を取得
					StatusInfoLocal status = null;
					if (statusFlag) {
						status = selectStatus.getHighPriorityStatus(facilityIds[index], MonitorControllerBean.ALL, orderFlg);
					}
					// 重要度が一番高いイベントログを取得
					EventLogLocal event = null;
					if (eventFlag) {
						event = selectEvent.getHighPriorityEvent(facilityIds[index], MonitorControllerBean.ALL, orderFlg);
					}
					
					// 重要度が一番高い情報を設定する
//					ArrayList info = this.getHighPriorityScope(status, event, repository, facilityIds[index], facilityId);
					ScopeInfoData scopeInfo = this.getHighPriorityScope(status, event, repository, facilityIds[index], facilityId);
								
//					if(info != null){
					if(scopeInfo != null){
									
						//そのfacilityTreeのソート順を読み出す。ArrayListに追加する。
						Integer sortValue = repository.getDisplaySortOrder(facilityIds[index]);
						
						scopeInfo.setSortValue(sortValue);
						/*
						 * TODO
						 * nodemap3.1の時の実装の一部を外す。
						 * 後で解析して、仕様変更がどこに現れているか確認すること！
						 */
						/*
						Collection fts = (Collection)ftHome.findByFacilityId(facilityIds[index]);
						Iterator it = fts.iterator();
						if(fts.size() != 0 ){
							FacilityTreeLocal ftl =(FacilityTreeLocal) (it.next());
							Integer sortValue = (Integer)(ftl.getSortValue());
							
							// とりあえずコメントアウト
//							info.add(sortValue);
							scopeInfo.setSortValue(sortValue);

						}else{
							// とりあえずコメントアウト
//							info.add(new Integer(100)); //デフォルト値100をとりあえず入れておく。

						}
						*/
						
//						list.add(info);
						list.add(scopeInfo);
						
						// 最上位のファシリティの情報を設定する
//						if(parentPriority > ((Integer)info.get(ScopeTabelDefine.PRIORITY)).intValue()){
						if(parentPriority > scopeInfo.getPriority().intValue()){
							highPriorityFlg = true;
						}
//						else if(parentPriority == ((Integer)info.get(ScopeTabelDefine.PRIORITY)).intValue()){
						else if(parentPriority == scopeInfo.getPriority().intValue()){							
							if(parentOutputDate != null){
//								if(parentOutputDate.before((Date)info.get(ScopeTabelDeafine.UPDATE_TIME))){
								if(parentOutputDate.before(scopeInfo.getOutputDate())){
									highPriorityFlg = true;
								}
							}
							else{
								highPriorityFlg = true;
							}
						}
						
						if(highPriorityFlg){
//							parentPriority = ((Integer)info.get(ScopeTabelDefine.PRIORITY)).intValue();
//							parentOutputDate = (Date)info.get(ScopeTabelDefine.UPDATE_TIME);
							parentPriority = scopeInfo.getPriority().intValue();
							parentOutputDate = scopeInfo.getOutputDate();
							
							if(parentFacilityId == null){
								if(facilityId != null && !"".equals(facilityId)){
									parentFacilityId = facilityId;
									parentFacilityPath = repository.getFacilityPath(facilityId, facilityId);	
								}
							}
						}
					}
				}
				statusInfoFlg = false;
				eventLogFlg = false;
				highPriorityFlg = false;
			}
			
			if(parentFacilityId != null){
				// 最上位のファシリティの情報を設定する
//				ArrayList info = new ArrayList();
//				info.add(new Integer(parentPriority));
//				info.add(parentFacilityId);
//				info.add(parentFacilityPath);
//				info.add(parentOutputDate);
//				info.add(new Integer(0)); //最上位スコープの表示位置は一番上にするため、Sort_ORDERの最小値をである0をセット
				
				ScopeInfoData scopeInfo = new ScopeInfoData(parentFacilityId, parentFacilityPath, parentPriority, parentOutputDate, 0);
				
//				list.add(info);
				list.add(scopeInfo);
				
			}
			
		} catch (CreateException e) {
			AplLogger apllog = new AplLogger("MON", "mon");
			String[] args = {facilityId};
			apllog.put("SYS", "001", args);
			m_log.debug("getScopeList():" + e.getMessage());
			throw e;
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("MON", "mon");
			String[] args = {facilityId};
			apllog.put("SYS", "001", args);
			m_log.debug("getScopeList():" + e.getMessage());
			throw new FacilityNotFoundException(e.getMessage(), e);
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("MON", "mon");
			String[] args = {facilityId};
			apllog.put("SYS", "001", args);
			m_log.debug("getScopeList():" + e.getMessage());
			throw e;
		}
		return list;
	}
	
	/**
	 * 重要度が最高で受信日時が最新のスコープ情報を返します。<BR>
	 * 引数で指定されたステータス情報／イベント情報を比較し、
	 * 重要度が最高で受信日時が最新のログの情報を、テーブルのカラム順にリストにセットし返します。
	 * 
	 * @param status ステータス情報
	 * @param event イベント情報
	 * @param repository リポジトリ
	 * @param facilityId 親ファシリティID 
	 * @return スコープ情報
	 */
//	private ArrayList getHighPriorityScope(StatusInfoLocal status, 
	private ScopeInfoData getHighPriorityScope(StatusInfoLocal status, 
			EventLogLocal event, 
			RepositoryControllerLocal repository,
			String facilityId,
			String parentFacilityId) throws CreateException, FinderException, NamingException{
		
		ArrayList info = new ArrayList();
		ScopeInfoData scopeInfo = new ScopeInfoData();
		
		boolean statusInfoFlg = false;
		boolean eventLogFlg = false;
		
		// 重要度が一番高い情報を設定する
		if(status == null && event == null){
			return null;
		}
		else if(status != null && event == null){
			statusInfoFlg = true;
		}
		else if(status == null && event != null){
			eventLogFlg = true;
		}
		else if(status != null && event != null){
			if((status.getPriority()).intValue() < (event.getPriority()).intValue()){
				statusInfoFlg = true;
			}
			else if((status.getPriority()).intValue() > (event.getPriority()).intValue()){
				eventLogFlg = true;
			}
			else if((status.getPriority()).intValue() == (event.getPriority()).intValue()){
				
				// 重要度が等しい場合、更新日時は最新の情報を設定する
				if((status.getOutputDate()).after(event.getOutputDate())){
					statusInfoFlg = true;
				}
				else{
					eventLogFlg = true;
				}
			}
		}
		
		String facilityPath = repository.getFacilityPath(facilityId, parentFacilityId);
		if(statusInfoFlg){
//			info.add(status.getPriority());
//			info.add(facilityId);
//			info.add(facilityPath);
//			info.add(new Date(status.getOutputDate().getTime()));
			scopeInfo.setPriority(status.getPriority());
			scopeInfo.setFacilityId(facilityId);
			scopeInfo.setFacilityPath(facilityPath);
			scopeInfo.setOutputDate(status.getOutputDate());
		}
		else if(eventLogFlg){
//			info.add(event.getPriority());
//			info.add(facilityId);
//			info.add(facilityPath);
//			info.add(new Date(event.getOutputDate().getTime()));
			scopeInfo.setPriority(event.getPriority());
			scopeInfo.setFacilityId(facilityId);
			scopeInfo.setFacilityPath(facilityPath);
			scopeInfo.setOutputDate(event.getOutputDate());

		}
//		return info;
		return scopeInfo;
	}
}
