/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.monitor.factory;

import java.io.File;
import java.io.FileInputStream;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.Properties;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.ConfirmConstant;
import com.clustercontrol.bean.FacilityTargetConstant;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.Property;
import com.clustercontrol.bean.ViewListInfo;
import com.clustercontrol.monitor.bean.EventFilterConstant;
import com.clustercontrol.monitor.bean.ReportEventInfo;
import com.clustercontrol.monitor.bean.StatusFilterConstant;
import com.clustercontrol.notify.monitor.ejb.entity.EventLogData;
import com.clustercontrol.notify.monitor.ejb.entity.EventLogLocal;
import com.clustercontrol.notify.monitor.ejb.entity.EventLogUtil;
import com.clustercontrol.monitor.ejb.session.MonitorControllerBean;
import com.clustercontrol.repository.ejb.session.RepositoryControllerBean;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.util.PropertyUtil;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * イベント情報を検索するクラス<BR>
 * 
 * @version 3.0.0
 * @since 1.0.0
 */
public class SelectEvent {
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog( SelectEvent.class );
	
	/** 
	 * 表示イベント数（デフォルト値）。<BR>
	 * 監視[イベント]ビューに表示するイベント表示数を格納します。
	 */
	private final static int MAX_DISPLAY_NUMBER = 500;
	
	/**
	 * ダウンロード可能な最大イベント数
	 * CSVおよびPDFで出力される最大のイベント数を格納します。
	 */
	private static final String MAX_REPORT_EVENT_COUNT_KEY = "monitor.common.report.event.count";
	private static volatile Integer MAX_REPORT_EVENT_COUNT = null;
	private static final Integer MAX_REPORT_EVENT_COUNT_DEFAULT = 1000;
	
	
	
	/**
	 * コンストラクタ
	 * 各種設定値を格納します
	 */
	public SelectEvent() {
		synchronized (this) {
			if (MAX_REPORT_EVENT_COUNT == null) {
				String maxReportEventCountStr = null;
				Properties properties = new Properties();
				
				try {
					String etcdir = System.getProperty("hinemos.manager.etc.dir");
					String propertyFile = etcdir + File.separator + "hinemos.properties";
					
					// プロパティファイルからキーと値のリストを読み込みます
					properties.load(new FileInputStream(propertyFile));
					
					// ダウンロード可能な最大イベント数を取得
					if (properties.containsKey("monitor.common.report.event.count")) {
						maxReportEventCountStr = properties.getProperty(MAX_REPORT_EVENT_COUNT_KEY);
						MAX_REPORT_EVENT_COUNT = Integer.parseInt(maxReportEventCountStr);
					} else {
						MAX_REPORT_EVENT_COUNT = MAX_REPORT_EVENT_COUNT_DEFAULT;
					}
				} catch (Exception e) {
					m_log.warn("a configuration in hinemos.properties is invalid(" + MAX_REPORT_EVENT_COUNT_KEY + " = " + maxReportEventCountStr + ").");
					MAX_REPORT_EVENT_COUNT = MAX_REPORT_EVENT_COUNT_DEFAULT;
				} finally {
					if (MAX_REPORT_EVENT_COUNT < -1) {
						m_log.warn("a configuration in hinemos.properties is invalid(" + MAX_REPORT_EVENT_COUNT_KEY + " = " + maxReportEventCountStr + ").");
						MAX_REPORT_EVENT_COUNT = MAX_REPORT_EVENT_COUNT_DEFAULT;
					} else if (MAX_REPORT_EVENT_COUNT == -1) {
						MAX_REPORT_EVENT_COUNT = null;
					}
					m_log.info("initialized max event count for report to " + (MAX_REPORT_EVENT_COUNT == null ? "unlimit" : MAX_REPORT_EVENT_COUNT.toString()) + ".");
				}
			}
		}
	}	
	
	/**
	 * イベント一覧情報を返します。<BR>
	 * 表示イベント数を越えた場合は、表示イベント数分のイベント情報一覧を返します。
	 * 各イベント情報は、EventLogDataインスタンスとして保持されます。<BR>
	 * 戻り値のViewListInfoは、クライアントにて表示用の形式に変換されます。
	 * 
	 * @param facilityId 取得対象の親ファシリティID
	 * @param level 取得対象のファシリティのターゲット（配下全て／直下のみ）
	 * @param messages 表示イベント数
	 * @return ビュー一覧情報
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * @throws SQLException
	 * 
	 * @see com.clustercontrol.monitor.ejb.entity.EventLogData
	 * @see com.clustercontrol.repository.ejb.session.RepositoryControllerBean#ALL
	 * @see com.clustercontrol.repository.ejb.session.RepositoryControllerBean#ONE_LEVEL
	 * @see com.clustercontrol.repository.ejb.session.RepositoryControllerBean#getFacilityIdList(String, int)
	 * @see com.clustercontrol.monitor.ejb.entity.EventLogBean#ejbFindEvent(String[], Integer, Timestamp, Timestamp, Timestamp, Timestamp, String, String, Integer, boolean, Integer)
	 * @see com.clustercontrol.monitor.ejb.entity.EventLogBean#ejbHomeCountEvent(String[], Integer, Timestamp, Timestamp, Timestamp, Timestamp, String, String, Integer)
	 * @see com.clustercontrol.monitor.bean.EventTabelDefine
	 */
	public ViewListInfo getEventList(String facilityId, int level, int messages)
			throws CreateException, FinderException, NamingException, SQLException {

		ViewListInfo list = new ViewListInfo();
		try {
			String[] facilityIds = null;

			// ファシリティが指定されない（最上位）場合は、ファシリティIDを指定せずに検索を行う
			if (facilityId != null && !"".equals(facilityId)) {

				if (level == MonitorControllerBean.ONLY) {
					if (facilityId != null && !"".equals(facilityId)) {
						facilityIds = new String[1];
						facilityIds[0] = facilityId;
					} else {
						return null;
					}
				} else {
					// 直下 または 配下すべてのファシリティIDを取得
					ArrayList facilityIdList = 
						RepositoryControllerUtil.getLocalHome().create().getFacilityIdList(
									facilityId, level);

					if (facilityIdList != null && facilityIdList.size() > 0) {
						// スコープの場合
						facilityIds = new String[facilityIdList.size()];
						facilityIdList.toArray(facilityIds);
					} else {
						if(facilityId != null){
							// ノードの場合
							facilityIds = new String[1];
							facilityIds[0] = facilityId;
						}
						else{
							// リポジトリが1件も登録されていない場合
							return null;
						}
					}
				}
			}
			
			if(messages <= 0){
				messages = MAX_DISPLAY_NUMBER;
			}
			Integer limit = new Integer(messages + 1);

			// イベントログ情報一覧を取得
			Collection ct = EventLogUtil.getLocalHome().findEvent(
					facilityIds, 
					null, 
					null, 
					null, 
					null,
					null, 
					null, 
					null,  
					new Integer(ConfirmConstant.TYPE_UNCONFIRMED),
					null, 
					false,
					limit);
			
			// 2次元配列に変換
			this.collectionToEventList(ct, list, messages);

		} catch (CreateException e) {
			outputLog(e, "getEventList()", facilityId, null, null, null, null, null, null, null, null, null);
			throw e;
		} catch (FinderException e) {
			outputLog(e, "getEventList()", facilityId, null, null, null, null, null, null, null, null, null);
			throw e;
		} catch (NamingException e) {
			outputLog(e, "getEventList()", facilityId, null, null, null, null, null, null, null, null, null);
			throw e;
		}
		
		return list;
	}

	/**
	 * 引数で指定された条件に一致するイベント一覧情報を返します。<BR>
	 * 表示イベント数を越えた場合は、表示イベント数分のイベント情報一覧を返します。
	 * 各イベント情報は、EventLogDataインスタンスとして保持されます。<BR>
	 * 戻り値のViewListInfoは、クライアントにて表示用の形式に変換されます。
	 * 
	 * @param facilityId 取得対象の親ファシリティID
	 * @param property 検索条件
	 * @param messages 表示イベント数
	 * @return ビュー一覧情報
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * @throws SQLException
	 * 
	 * @see com.clustercontrol.monitor.ejb.entity.EventLogData
	 * @see com.clustercontrol.util.PropertyUtil#getPropertyValue(com.clustercontrol.bean.Property, java.lang.String)
	 * @see com.clustercontrol.repository.ejb.session.RepositoryControllerBean#getFacilityIdList(String, int)
	 * @see com.clustercontrol.monitor.ejb.entity.EventLogBean#ejbFindEvent(String[], Integer, Timestamp, Timestamp, Timestamp, Timestamp, String, String, Integer, boolean, Integer)
	 * @see com.clustercontrol.monitor.ejb.entity.EventLogBean#ejbHomeCountEvent(String[], Integer, Timestamp, Timestamp, Timestamp, Timestamp, String, String, Integer)
	 * @see com.clustercontrol.monitor.bean.EventTabelDefine
	 */
	public ViewListInfo getEventList(String facilityId, Property property, int messages)
			throws CreateException, FinderException, NamingException, SQLException {

		ViewListInfo list = new ViewListInfo();
		
		Integer priority = null;
		Timestamp outputFromDate = null;
		Timestamp outputToDate = null;
		Timestamp generationFromDate = null;
		Timestamp generationToDate = null;
		String facilityType = null;
		String application = null;
		String message = null;
		Integer confirmFlg = null;
		String confirmUser = null;
		
		try {
			ArrayList values = null;

			//重要度取得
			values = PropertyUtil.getPropertyValue(property,
					StatusFilterConstant.PRIORITY);
			if (!"".equals(values.get(0))) {
				int priorityType = PriorityConstant
						.stringToType((String) values.get(0));
				if (priorityType != -1) {
					priority = new Integer(priorityType);
				}
			}

			//更新日時（自）取得
			values = PropertyUtil.getPropertyValue(property,
					EventFilterConstant.OUTPUT_FROM_DATE);
			if (values.get(0) instanceof Date) {
				outputFromDate = new Timestamp(((Date) values.get(0)).getTime());
				outputFromDate.setNanos(0);
			}

			//更新日時（至）取得
			values = PropertyUtil.getPropertyValue(property,
					EventFilterConstant.OUTPUT_TO_DATE);
			if (values.get(0) instanceof Date) {
				outputToDate = new Timestamp(((Date) values.get(0)).getTime());
				outputToDate.setNanos(999999999);
			}

			//出力日時（自）取得
			values = PropertyUtil.getPropertyValue(property,
					EventFilterConstant.GENERATION_FROM_DATE);
			if (values.get(0) instanceof Date) {
				generationFromDate = new Timestamp(((Date) values.get(0))
						.getTime());
				generationFromDate.setNanos(0);
			}

			//出力日時（至）取得
			values = PropertyUtil.getPropertyValue(property,
					EventFilterConstant.GENERATION_TO_DATE);
			if (values.get(0) instanceof Date) {
				generationToDate = new Timestamp(((Date) values.get(0))
						.getTime());
				generationToDate.setNanos(999999999);
			}

			//対象ファシリティ種別取得
			values = PropertyUtil.getPropertyValue(property,
					StatusFilterConstant.FACILITY_TYPE);
			if (!"".equals(values.get(0))) {
				facilityType = (String) values.get(0);
			}

			//アプリケーション取得
			values = PropertyUtil.getPropertyValue(property,
					StatusFilterConstant.APPLICATION);
			if (!"".equals(values.get(0))) {
				application = (String) values.get(0);
			}

			//メッセージ取得
			values = PropertyUtil.getPropertyValue(property,
					StatusFilterConstant.MESSAGE);
			if (!"".equals(values.get(0))) {
				message = (String) values.get(0);
			}

			// 確認有無取得
			values = PropertyUtil.getPropertyValue(property,
					EventFilterConstant.CONFIRMED);
			int confirmFlgType = ConfirmConstant.stringToType((String) values
					.get(0));
			if (confirmFlgType != -1) {
				confirmFlg = new Integer(confirmFlgType);
			}
			
			// 確認ユーザ取得
			values = PropertyUtil.getPropertyValue(property, EventFilterConstant.CONFIRM_USER);

			if (!"".equals(values.get(0))) {
				confirmUser = (String) values.get(0);	
			}

			// 対象ファシリティのファシリティIDを取得
			// ファシリティが指定されない（最上位）場合は、ファシリティIDを指定せずに検索を行う
			String[] facilityIds = null;
			if (facilityId != null && !"".equals(facilityId)) {

				int level = RepositoryControllerBean.ALL;
				if (FacilityTargetConstant.STRING_BENEATH.equals(facilityType)) {
					level = RepositoryControllerBean.ONE_LEVEL;
				}

				ArrayList facilityIdList = RepositoryControllerUtil
						.getLocalHome().create().getFacilityIdList(facilityId,
								level);

				if (facilityIdList != null && facilityIdList.size() > 0) {
					// スコープの場合
					facilityIds = new String[facilityIdList.size()];
					facilityIdList.toArray(facilityIds);
				}
				else {
					if(facilityId != null){
						// ノードの場合
						facilityIds = new String[1];
						facilityIds[0] = facilityId;
					}
					else{
						// リポジトリが1件も登録されていない場合
						return null;
					}
				}
			}

			if(messages <= 0){
				messages = MAX_DISPLAY_NUMBER;
			}
			Integer limit = new Integer(messages + 1);
			
			// イベントログ情報一覧を取得
			Collection ct = EventLogUtil.getLocalHome().findEvent(facilityIds, 
					priority, 
					outputFromDate, 
					outputToDate,
					generationFromDate, 
					generationToDate, 
					application, 
					message,
					confirmFlg,
					confirmUser,
					false,
					limit);
			
			// 2次元配列に変換
			this.collectionToEventList(ct, list, messages);

		} catch (CreateException e) {
			outputLog(e, "getEventList()", facilityId, priority, outputFromDate, outputToDate, generationFromDate, generationToDate, facilityType, application, message, confirmFlg);
			throw e;
		} catch (FinderException e) {
			outputLog(e, "getEventList()", facilityId, priority, outputFromDate, outputToDate, generationFromDate, generationToDate, facilityType, application, message, confirmFlg);
			throw e;
		} catch (NamingException e) {
			outputLog(e, "getEventList()", facilityId, priority, outputFromDate, outputToDate, generationFromDate, generationToDate, facilityType, application, message, confirmFlg);
			throw e;
		}
		
		return list;
	}

	/**
	 * 重要度が最高で受信日時が最新のイベント情報を返します。
	 * <p>
	 * <ol>
	 * <li>引数で指定されたファシリティ配下のファシリティを、指定されたファシリティのターゲットで取得します。</li>
	 * <li>取得したファシリティに属する重要度が最高 かつ 受信日時が最新の未確認のイベント情報を取得し返します。</li>
	 * </ol>
	 * 
	 * @param facilityId 取得対象の親ファシリティID
	 * @param level 取得対象のファシリティのターゲット（配下全て／直下のみ）
	 * @return イベントのローカルコンポーネントインターフェース
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.repository.ejb.session.RepositoryControllerBean#ALL
	 * @see com.clustercontrol.repository.ejb.session.RepositoryControllerBean#ONE_LEVEL
	 * @see com.clustercontrol.repository.ejb.session.RepositoryControllerBean#getFacilityIdList(String, int)
	 * @see com.clustercontrol.monitor.ejb.entity.EventLogBean#ejbFindHighPriorityEvent(String[], Integer, Timestamp, Timestamp, Timestamp, Timestamp, String, String, Integer)
	 */
	public EventLogLocal getHighPriorityEvent(String facilityId, int level)
			throws CreateException, FinderException, NamingException {
		return getHighPriorityEvent(facilityId, level, true);
	}
	
	/**
	 * 重要度が最高で受信日時が最新のイベント情報を返します。
	 * <p>
	 * <ol>
	 * <li>引数で指定されたファシリティ配下のファシリティを、指定されたファシリティのターゲットで取得します。</li>
	 * <li>取得したファシリティに属する重要度が最高 かつ 受信日時が最新の未確認のイベント情報を取得し返します。</li>
	 * </ol>
	 * 
	 * @param facilityId 取得対象の親ファシリティID
	 * @param level 取得対象のファシリティのターゲット（配下全て／直下のみ）
	 * @param orderFlg ソートの有無
	 * @return イベントのローカルコンポーネントインターフェース
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.repository.ejb.session.RepositoryControllerBean#ALL
	 * @see com.clustercontrol.repository.ejb.session.RepositoryControllerBean#ONE_LEVEL
	 * @see com.clustercontrol.repository.ejb.session.RepositoryControllerBean#getFacilityIdList(String, int)
	 * @see com.clustercontrol.monitor.ejb.entity.EventLogBean#ejbFindHighPriorityEvent(String[], Integer, Timestamp, Timestamp, Timestamp, Timestamp, String, String, Integer)
	 */
	public EventLogLocal getHighPriorityEvent(String facilityId, int level, boolean orderFlg)
			throws CreateException, FinderException, NamingException {

		EventLogLocal event = null;
		try {
			String[] facilityIds = null;
			if (level == MonitorControllerBean.ONLY) {
				if (facilityId != null && !"".equals(facilityId)) {
					facilityIds = new String[1];
					facilityIds[0] = facilityId;
				} else {
					return null;
				}
			} else {
				// 直下 または 配下すべてのファシリティIDを取得
				ArrayList facilityIdList = RepositoryControllerUtil
						.getLocalHome().create().getFacilityIdList(facilityId,
								level);

				if (facilityIdList != null && facilityIdList.size() > 0) {
					// スコープの場合
					if(facilityId != null){
						facilityIdList.add(facilityId);
					}
					facilityIds = new String[facilityIdList.size()];
					facilityIdList.toArray(facilityIds);
				} else {
					if(facilityId != null){
						// ノードの場合
						facilityIds = new String[1];
						facilityIds[0] = facilityId;
					}
					else{
						// リポジトリが1件も登録されていない場合
						return null;
					}
				}
			}

//			// イベントログ情報一覧を取得
//			Collection ct = EventLogUtil.getLocalHome().findHighPriorityEvent(facilityIds, 
//					null, 
//					null, 
//					null, 
//					null, 
//					null, 
//					null, 
//					null, 
//					new Integer(ConfirmConstant.TYPE_UNCONFIRMED));
	
			// 重要度のリストを取得する
			int[] priorityList = PriorityConstant.PRIORITY_LIST;			
			for(int i=0; i<priorityList.length; i++){
				// イベントログ情報一覧から重要度が危険のもので、最近出力されたイベントを取得する。
				Collection ct = EventLogUtil.getLocalHome().findHighPriorityEvent(facilityIds, 
						priorityList[i], 
						null, 
						null, 
						null, 
						null, 
						null, 
						null, 
						new Integer(ConfirmConstant.TYPE_UNCONFIRMED), 
						null,
						orderFlg);
				// 重要度の高いもの順にループされるため、取得できた場合は、それを返す。
				Iterator itr = ct.iterator();
				// イテレータで参照するが、0件か１件しかない。
				if (itr.hasNext()) {
					event = (EventLogLocal) itr.next();
					return event;
				}
			}

		} catch (CreateException e) {
			m_log.debug("getHighPriorityEvent():" + e.getMessage());
			throw e;
		} catch (FinderException e) {
			m_log.debug("getHighPriorityEvent():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			m_log.debug("getHighPriorityEvent():" + e.getMessage());
			throw e;
		}
		return event;
	}
	
	/**
	 * 引数で指定された条件に一致する帳票出力用のイベント情報一覧を返します。
	 * <p>
	 * <ol>
	 * <li>引数で指定されたプロパティに格納された検索条件を、プロパティユーティリティ（{@link com.clustercontrol.util.PropertyUtil}）を使用して取得します。</li>
	 * <li>引数で指定されたファシリティ配下のファシリティと検索条件を基に、イベント情報を取得します。</li>
	 * <li>１イベント情報を帳票出力用イベント情報（{@link com.clustercontrol.monitor.bean.ReportEventInfo}）にセットします。</li>
	 * <li>この帳票出力用イベント情報を、イベント情報一覧を保持するリスト（{@link ArrayList}）にセットし返します。<BR>
	 * </ol>
	 * 
	 * @param facilityId 取得対象の親ファシリティID
	 * @param property 検索条件
	 * @return 帳票出力用イベント情報一覧（{@link com.clustercontrol.monitor.bean.ReportEventInfo}のリスト）
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * @throws IndexOutOfBoundsException
	 * 
	 * @since 2.1.0
	 * 
	 * @see com.clustercontrol.util.PropertyUtil#getPropertyValue(com.clustercontrol.bean.Property, java.lang.String)
	 * @see com.clustercontrol.repository.ejb.session.RepositoryControllerBean#getFacilityIdList(String, int)
	 * @see com.clustercontrol.monitor.ejb.entity.EventLogBean#ejbFindEvent(String[], Integer, Timestamp, Timestamp, Timestamp, Timestamp, String, String, Integer, boolean, Integer)
	 * @see com.clustercontrol.monitor.bean.ReportEventInfo
	 */
	public ArrayList getEventListForReport(String facilityId, Property property)
			throws CreateException, FinderException, NamingException, IndexOutOfBoundsException {

		ArrayList list = null;
		
		Integer priority = null;
		Timestamp outputFromDate = null;
		Timestamp outputToDate = null;
		Timestamp generationFromDate = null;
		Timestamp generationToDate = null;
		String facilityType = null;
		String application = null;
		String message = null;
		Integer confirmFlg = null;
		String confirmUser = null;
		
		try {
			ArrayList values = null;

			//重要度取得
			values = PropertyUtil.getPropertyValue(property,
					StatusFilterConstant.PRIORITY);
			if (!"".equals(values.get(0))) {
				int priorityType = PriorityConstant
						.stringToType((String) values.get(0));
				if (priorityType != -1) {
					priority = new Integer(priorityType);
				}
			}

			//更新日時（自）取得
			values = PropertyUtil.getPropertyValue(property,
					EventFilterConstant.OUTPUT_FROM_DATE);
			if (values.get(0) instanceof Date) {
				outputFromDate = new Timestamp(((Date) values.get(0)).getTime());
				outputFromDate.setNanos(0);
			}

			//更新日時（至）取得
			values = PropertyUtil.getPropertyValue(property,
					EventFilterConstant.OUTPUT_TO_DATE);
			if (values.get(0) instanceof Date) {
				outputToDate = new Timestamp(((Date) values.get(0)).getTime());
				outputToDate.setNanos(999999999);
			}

			//出力日時（自）取得
			values = PropertyUtil.getPropertyValue(property,
					EventFilterConstant.GENERATION_FROM_DATE);
			if (values.get(0) instanceof Date) {
				generationFromDate = new Timestamp(((Date) values.get(0))
						.getTime());
				generationFromDate.setNanos(0);
			}

			//出力日時（至）取得
			values = PropertyUtil.getPropertyValue(property,
					EventFilterConstant.GENERATION_TO_DATE);
			if (values.get(0) instanceof Date) {
				generationToDate = new Timestamp(((Date) values.get(0))
						.getTime());
				generationToDate.setNanos(999999999);
			}

			//対象ファシリティ種別取得
			values = PropertyUtil.getPropertyValue(property,
					StatusFilterConstant.FACILITY_TYPE);
			if (!"".equals(values.get(0))) {
				facilityType = (String) values.get(0);
			}

			//アプリケーション取得
			values = PropertyUtil.getPropertyValue(property,
					StatusFilterConstant.APPLICATION);
			if (!"".equals(values.get(0))) {
				application = (String) values.get(0);
			}

			//メッセージ取得
			values = PropertyUtil.getPropertyValue(property,
					StatusFilterConstant.MESSAGE);
			if (!"".equals(values.get(0))) {
				message = (String) values.get(0);
			}

			// 確認有無取得
			values = PropertyUtil.getPropertyValue(property,
					EventFilterConstant.CONFIRMED);
			int confirmFlgType = ConfirmConstant.stringToType((String) values
					.get(0));
			if (confirmFlgType != -1) {
				confirmFlg = new Integer(confirmFlgType);
			}
			
			// 確認ユーザ
			values = PropertyUtil.getPropertyValue(property,
					EventFilterConstant.CONFIRM_USER);
			if (!"".equals(values.get(0))) {
				confirmUser = (String) values.get(0);
			}

			// 対象ファシリティのファシリティIDを取得
			// ファシリティが指定されない（最上位）場合は、ファシリティIDを指定せずに検索を行う
			String[] facilityIds = null;
			if (facilityId != null && !"".equals(facilityId)) {

				int level = RepositoryControllerBean.ALL;
				if (FacilityTargetConstant.STRING_BENEATH.equals(facilityType)) {
					level = RepositoryControllerBean.ONE_LEVEL;
				}

				ArrayList facilityIdList = RepositoryControllerUtil
						.getLocalHome().create().getFacilityIdList(facilityId,
								level);

				if (facilityIdList != null && facilityIdList.size() > 0) {
					// スコープの場合
					facilityIds = new String[facilityIdList.size()];
					facilityIdList.toArray(facilityIds);
				}
				else {
					if(facilityId != null){
						// ノードの場合
						facilityIds = new String[1];
						facilityIds[0] = facilityId;
					}
					else{
						// リポジトリが1件も登録されていない場合
						return null;
					}
				}
			}

			// イベントログ情報一覧を取得
			Collection ct = EventLogUtil.getLocalHome().findEvent(facilityIds, 
					priority, 
					outputFromDate, 
					outputToDate,
					generationFromDate, 
					generationToDate, 
					application, 
					message,
					confirmFlg,
					confirmUser, 
					true,
					MAX_REPORT_EVENT_COUNT);

			// 帳票出力用に変換
			list = this.collectionToArrayForReportd(ct);

		} catch (CreateException e) {
			outputLog(e, "getEventListForReport()", facilityId, priority, outputFromDate, outputToDate, generationFromDate, generationToDate, facilityType, application, message, confirmFlg);
			throw e;
		} catch (FinderException e) {
			outputLog(e, "getEventListForReport()", facilityId, priority, outputFromDate, outputToDate, generationFromDate, generationToDate, facilityType, application, message, confirmFlg);
			throw e;
		} catch (NamingException e) {
			outputLog(e, "getEventListForReport()", facilityId, priority, outputFromDate, outputToDate, generationFromDate, generationToDate, facilityType, application, message, confirmFlg);
			throw e;
		}
		return list;
	}

	/**
	 * DBより取得したイベント情報をイベント一覧情報に格納します。
	 * <p>
	 * <ol>
	 * <li>１イベント情報をEventLogDataのインスタンスとし、イベント情報一覧を保持するリスト（{@link ArrayList}）に格納します。<BR>
	 * <li>イベント情報一覧を、引数で指定されたビュー一覧情報（{@link com.clustercontrol.bean.ViewListInfo}）にセットします。</li>
	 * </ol>
	 * <p>
	 * また、イベント数を重要度毎にカウントし、
	 * 表示イベント数よりもイベント数が少ない場合は、重要度別イベント数を引数で指定されたビュー一覧情報にセットします。
	 * 
	 * @param ct イベント情報取得結果
	 * @param eventList ビュー一覧情報
	 * @param messages イベント最大表示件数
	 * 
	 * @see com.clustercontrol.monitor.ejb.entity.EventLogData
	 * @see com.clustercontrol.monitor.bean.EventTabelDefine
	 */
	@SuppressWarnings("unchecked")
	private void collectionToEventList(Collection ct, ViewListInfo eventList, int messages) {
		int critical = 0;
	    int warning = 0;
	    int info = 0;
	    int unknown = 0;
	    
		ArrayList<EventLogData> list = new ArrayList();
		Iterator itr = ct.iterator();
		while (itr.hasNext()) {
			EventLogLocal event = (EventLogLocal) itr.next();
			
			EventLogData eventLogData = new EventLogData();			
			eventLogData.setPriority(event.getPriority());
			eventLogData.setOutputDate(event.getOutputDate());
			eventLogData.setGenerationDate(event.getGenerationDate());
			eventLogData.setPluginId(event.getPluginId());
			eventLogData.setMonitorId(event.getMonitorId());
			eventLogData.setFacilityId(event.getFacilityId());
			eventLogData.setScopeText(event.getScopeText());
			eventLogData.setApplication(event.getApplication());
			eventLogData.setMessageId(event.getMessageId());
			eventLogData.setMessage(event.getMessage());
			eventLogData.setConfirmFlg(event.getConfirmFlg());
			eventLogData.setConfirmUser(event.getConfirmUser());
			
			list.add(eventLogData);
			
			//最大表示件数以下の場合
			if(event.getPriority().intValue() == PriorityConstant.TYPE_CRITICAL)
				critical++;
			else if(event.getPriority().intValue() == PriorityConstant.TYPE_WARNING)
				warning++;
			else if(event.getPriority().intValue() == PriorityConstant.TYPE_INFO)
				info++;
			else if(event.getPriority().intValue() == PriorityConstant.TYPE_UNKNOWN)
				unknown++;
			
			//取得したイベントを最大表示件数まで格納したら終了
			if(list.size() >= messages)
				break;
		}
		
		//イベント数を設定
		eventList.setCritical(critical);
		eventList.setWarning(warning);
		eventList.setInfo(info);
		eventList.setUnKnown(unknown);
		eventList.setTotal(ct.size());

		eventList.setList(list);
	}
	
	/**
	 * DBより取得したイベント情報を帳票出力用イベント情報一覧に格納します。
	 * 
	 * @param ct イベント情報取得結果
	 * @return 帳票出力用イベント情報一覧
	 * 
	 * @version 2.1.0
	 * @since 2.1.0
	 */
	private ArrayList<ReportEventInfo> collectionToArrayForReportd(Collection ct) {

		ArrayList<ReportEventInfo> list = new ArrayList<ReportEventInfo>();
		Iterator itr = ct.iterator();
		while (itr.hasNext()) {
			EventLogLocal event = (EventLogLocal) itr.next();

			ReportEventInfo info = new ReportEventInfo();
			
			info.setApplication(event.getApplication());
			info.setConfirmFlg(ConfirmConstant.typeToString(event.getConfirmFlg().intValue()));
			info.setFacilityId(event.getFacilityId());
			info.setGenerationDate(new Date(event.getGenerationDate().getTime()));
			info.setMessage(event.getMessage());
			info.setMessageId(event.getMessageId());
			info.setMessageOrg(event.getMessageOrg());
			info.setMonitorId(event.getMonitorId());
			info.setOutputDate(new Date(event.getOutputDate().getTime()));
			info.setPluginId(event.getPluginId());
			info.setPriority(PriorityConstant.typeToString(event.getPriority().intValue()));
			info.setScopeText(event.getScopeText());
			info.setConfirmUser(event.getConfirmUser());
			
			list.add(info);
		}
		return list;
	}
	
	/**
     * アプリケーションログにログを出力
     * 
     * @param e 例外
     */
	private void outputLog(
			Exception e, 
			String method,
			String facilityId,
			Integer priority,
			Timestamp outputFromDate,
			Timestamp outputToDate,
			Timestamp generationFromDate,
			Timestamp generationToDate,
			String facilityType,
			String application,
			String message,
			Integer confirmFlg){
		
		int priorityInt = -1;
		String outputFromDateString = null;
		String outputToDateString = null;
		String generationFromDateString = null;
		String generationToDateString = null;
		int confirmFlgInt = -1;
		
		if(priority != null)
			priorityInt = priority.intValue();
		if(outputFromDate != null)
			outputFromDateString = outputFromDate.toString();
		if(outputToDate != null)
			outputToDateString = outputToDate.toString();
		if(generationFromDate != null)
			generationFromDateString = generationFromDate.toString();
		if(generationToDate != null)
			generationToDateString = generationToDate.toString();
		if(confirmFlg != null)
			confirmFlgInt = confirmFlg.intValue();
		
        AplLogger apllog = new AplLogger("MON", "mon");
        String[] args = {facilityId,PriorityConstant.typeToString(priorityInt),outputFromDateString,outputToDateString,generationFromDateString,generationToDateString,facilityType,application,message,ConfirmConstant.typeToString(confirmFlgInt)};
        apllog.put("SYS", "003", args);
		m_log.debug(method + ":" + e.getMessage());
	}
}
