/*
 
Copyright (C) since 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.accesscontrol.factory;

import java.sql.Timestamp;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.EJBException;
import javax.ejb.RemoveException;
import javax.management.ObjectName;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.jmx.adaptor.rmi.RMIAdaptor;

import com.clustercontrol.accesscontrol.bean.RoleConstant;
import com.clustercontrol.accesscontrol.bean.UserConstant;
import com.clustercontrol.accesscontrol.bean.UserTypeConstant;
import com.clustercontrol.accesscontrol.ejb.entity.UserLocal;
import com.clustercontrol.accesscontrol.ejb.entity.UserRoleLocal;
import com.clustercontrol.accesscontrol.ejb.entity.UserRoleUtil;
import com.clustercontrol.accesscontrol.ejb.entity.UserUtil;
import com.clustercontrol.accesscontrol.util.UserValidator;
import com.clustercontrol.bean.Property;
import com.clustercontrol.commons.util.ObjectValidator;
import com.clustercontrol.util.PropertyUtil;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * ユーザ情報を更新するファクトリクラス<BR>
 *
 * @version 1.0.0
 * @since 3.2.0
 */
public class LoginUserModifier {
	
	/** ログ出力のインスタンス */
	protected static Log m_log = LogFactory.getLog(LoginUserModifier.class);
	
	/** JMXのルックアップ名 */
	public static final String LOOKUP_NAME = "jmx/invoker/RMIAdaptor";
	/** JMXのオブジェクト名 */
	public static final String OBJECT_NAME = "jboss.security:service=JaasSecurityManager";
	/** JMXのオブジェクトのメソッド名 */
	public static final String OPERATION_NAME = "flushAuthenticationCache";
	
	/**
	 * ログインユーザを追加する。<BR>
	 * 
	 * @param property 追加するユーザ情報
	 * @param modifyUserId 作業ユーザID
	 * @throws CreateException ユーザインスタンスが生成できなかった場合
	 * @throws ParseException ユーザIDが不正な場合
	 */
	public static void addUser(Property property, String modifyUserId) throws CreateException, ParseException {
		/** ローカル変数 */
		UserLocal user = null;
		ArrayList inputValue = null;
		String userId = null;
		
		/** メイン処理 */
		m_log.debug("adding a user...");
		
		try {
			// 更新排他制御
			AccessLock.lock(AccessLock.ACCESS);
			
			// ユーザIDの入力値チェック
			inputValue = PropertyUtil.getPropertyValue(property, UserConstant.UID);
			userId = (String)inputValue.get(0);
			UserValidator.validateUserId(userId);
			
			// ユーザインスタンスの作成
			user = UserUtil.getLocalHome().create(userId);
			
			// ユーザ種別の格納
			user.setUserType(UserTypeConstant.LOGIN_USER);
			
			// ユーザ情報の反映
			setUser(user, property, modifyUserId);
		} catch (DuplicateKeyException e) {
			String[] args = {userId};
			AplLogger apllog = new AplLogger("ACC", "acc");
			apllog.put("SYS", "006", args);
			
			m_log.info("failure to add a user. a user'id is duplicated. (userId = " + userId + ")", e);
			throw e;
		} catch (ParseException e) {
			String[] args = {userId};
			AplLogger apllog = new AplLogger("ACC", "acc");
			apllog.put("SYS", "006", args);
			
			m_log.warn("failure to add a user. (userId = " + userId + ")", e);
			throw e;
		} catch (Exception e) {
			String[] args = {userId};
			AplLogger apllog = new AplLogger("ACC", "acc");
			apllog.put("SYS", "006", args);
			
			m_log.warn("failure to add a user. (userId = " + userId + ")", e);
			throw new EJBException("failure to add a user. (userId = " + userId + ")", e);
		}
		
		m_log.info("successful in adding a user. (userId = " + userId + ")");
	}
	
	/**
	 * ログインユーザを削除する。<BR>
	 * 
	 * @param userId 削除するユーザID
	 * @param modifyUserId 作業ユーザID
	 */
	public static void deleteUser(String userId, String modifyUserId) {
		/** ローカル変数 */
		UserLocal user = null;
		
		/** メイン処理 */
		m_log.debug("deleting a user...");
		
		try {
			// 更新排他制御
			AccessLock.lock(AccessLock.ACCESS);
			
			// 作業ユーザと削除対象のユーザが一致している場合、削除不可とする
			if (userId.compareTo(modifyUserId) == 0) {
				throw new EJBException("a user will be deleted is equal to current login user.");
			}
			
			// 該当するユーザを検索して取得
			user = UserUtil.getLocalHome().findByPrimaryKey(userId);
			
			// ユーザを削除する（DELETE CASCADEによりユーザ権限も削除される）
			user.remove();
		} catch (Exception e) {
			String[] args = {userId};
			AplLogger apllog = new AplLogger("ACC", "acc");
			apllog.put("SYS", "007", args);
			
			m_log.warn("failure to delete a user. (userId = " + userId + ")", e);
			throw new EJBException("failure to delete a user. (userId = " + userId + ")", e);
		}
		
		//認証キャッシュ更新
		flushAuthenticationCache();
		
		m_log.info("successful in deleting a user. (userId = " + userId + ")");
	}
	
	/**
	 * ログインユーザを変更する。<BR>
	 * 
	 * @param property 変更するユーザ情報
	 * @param modifyUserId 作業ユーザID
	 */
	public static void modifyUser(Property property, String modifyUserId) {
		/** ローカル変数 */
		UserLocal user = null;
		ArrayList inputValue = null;
		String userId = null;
		
		/** メイン処理 */
		m_log.debug("modifing a user...");
		
		try {
			// 更新排他制御
			AccessLock.lock(AccessLock.ACCESS);
			
			// ユーザIDの入力値チェック
			inputValue = PropertyUtil.getPropertyValue(property, UserConstant.UID);
			userId = (String)inputValue.get(0);
			UserValidator.validateUserId(userId);
			
			// 該当するユーザを検索して取得
			user = UserUtil.getLocalHome().findByPrimaryKey(userId);
			
			// ユーザ情報の反映
			setUser(user, property, modifyUserId);
		} catch (Exception e) {
			String[] args = {userId};
			AplLogger apllog = new AplLogger("ACC", "acc");
			apllog.put("SYS", "008", args);
			
			m_log.warn("failure to modify a user. (userId = " + userId + ")", e);
			throw new EJBException("failure to modify a user. (userId = " + userId + ")", e);
		}
		
		//認証キャッシュ更新
		flushAuthenticationCache();
		
		m_log.info("successful in modifing a user. (userId = " + userId + ")");
	}
	
	/**
	 * ユーザ情報をユーザエンティティに反映する。<BR>
	 * 
	 * @param user 反映対象のユーザエンティティ
	 * @param property 反映情報が格納されたユーザ情報
	 * @param modifyUserId 作業ユーザID
	 * @throws NamingException ホームインタフェースに接続できなかった場合
	 * @throws CreateException 新規のユーザ権限が生成できなかった場合
	 * @throws RemoveException 既存のユーザ権限インスタンスが生成できなかった場合
	 */
	private static void setUser(UserLocal user, Property property, String modifyUserId) throws CreateException, NamingException, RemoveException {
		/** ローカル変数 */
		ArrayList inputValue = null;
		Timestamp now = null;
		boolean newUserFlg = false;
		ArrayList<String> userRolesAlready = null;
		
		/** メイン処理 */
		// 新規ユーザかどうかの確認
		newUserFlg = ObjectValidator.isEmptyString(user.getCreateUserId()) && user.getCreateDatetime().getTime() == 0 ? true : false;
		
		// 現在日時を取得
		now = new Timestamp(new Date().getTime());
		
		// 入力値（ユーザ名）の格納
		inputValue = PropertyUtil.getPropertyValue(property, UserConstant.NAME); 
		user.setUserName(ObjectValidator.objectToString(inputValue.get(0)));
		
		// 入力値（説明）の格納
		inputValue = PropertyUtil.getPropertyValue(property, UserConstant.DESCRIPTION);
		user.setDescription(ObjectValidator.objectToString(inputValue.get(0)));
		
		if (newUserFlg) {
			// 入力値（作成者・日時）の格納
			user.setCreateUserId(modifyUserId);
			user.setCreateDatetime(now);
		}
		
		// 入力値（更新者・日時）の格納
		user.setModifyUserId(modifyUserId);
		user.setModifyDatetime(now);
		
		// ユーザ権限インスタンスの設定
		userRolesAlready = new ArrayList<String>();
		if (user.getUserRole() != null) {
			for (UserRoleLocal userRole : (Collection<UserRoleLocal>)user.getUserRole()) {
				inputValue = PropertyUtil.getPropertyValue(property, userRole.getUserRole());
				if (inputValue.size() > 0 && ! ((Boolean)inputValue.get(0)).booleanValue()) {
					try {
						userRole.remove();
					} catch (RemoveException e) {
						m_log.warn("failure to delete a user's role. (userId = " + user.getUserId() + ", userRole = " + userRole.getUserRole() + ")", e);
						throw e;
					}
				} else {
					userRolesAlready.add(userRole.getUserRole());
				}
			}
		}
		if (newUserFlg) {
			// クライアントから送信されないユーザ権限の登録
			UserRoleUtil.getLocalHome().create(user.getUserId(), RoleConstant.HINEMOS_USER);
		}
		for (String roleName : UserProperty.getRoleList().getRoles()) {
			if (userRolesAlready.contains(roleName)) {
				continue;
			}
			inputValue = PropertyUtil.getPropertyValue(property, roleName);
			if (inputValue.size() > 0 && ((Boolean)inputValue.get(0)).booleanValue()) {
				try {
					UserRoleUtil.getLocalHome().create(user.getUserId(), roleName);
				} catch (CreateException e) {
					m_log.warn("failure to add a user's role. (userId = " + user.getUserId() + ", userRole = " + roleName + ")", e);
					throw e;
				}
			}
		}
	}
	
	/**
	 * ログインユーザに設定されたパスワードを変更する。<BR>
	 * 
	 * @param userId ユーザID
	 * @param password 新しいパスワード文字列
	 */
	public static void modifyUserPassword(String userId, String password) {
		/** ローカル変数 */
		UserLocal user = null;
		
		/** メイン処理 */
		m_log.debug("modifing a user's password...");
		
		try {
			// 更新排他制御
			AccessLock.lock(AccessLock.ACCESS);
			
			// 該当するユーザを検索して取得
			user = UserUtil.getLocalHome().findByPrimaryKey(userId);
			
			// パスワードを反映する
			user.setPassword(password);
		} catch (Exception e) {
			AplLogger apllog = new AplLogger("ACC", "acc");
			String[] args = {userId};
			apllog.put("SYS", "009", args);
			
			m_log.warn("failure to modify user's password. (userId = " + userId + ")", e);
			throw new EJBException("failure to modify user's password. (userId = " + userId + ")", e);
		}
		
		//認証キャッシュ更新
		flushAuthenticationCache();
		
		m_log.info("successful in modifing a user's password. (userId = " + userId + ")");
	}
	
	
	/**
	 * JMX経由で、認証キャッシュを初期化する。<BR>
	 */
	public static void flushAuthenticationCache() {
		/** ローカル変数 */
		InitialContext ic = null;
		RMIAdaptor server = null;
		ObjectName name = null;
		
		/** メイン処理 */
		m_log.debug("clearing authentication cache...");
		
		try {
			ic = new InitialContext();
			
			// RMIAdaptorを取得
			server = (RMIAdaptor) ic.lookup(LOOKUP_NAME);
			
			// ObjectNameを設定
			name = new ObjectName(OBJECT_NAME);
			
			// ObjectNameのOperationNameのメソッドを実行
			Object[]args = {"hinemos"};
			String[] signature = {String.class.getName()};
			server.invoke(name, OPERATION_NAME, args, signature);
		} catch (Exception e) {
			m_log.warn("failure to clear authentication cache", e);
		}
	}
	
}
