/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.syslogng.forward;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.repository.bean.FacilityAttributeConstant;
import com.clustercontrol.repository.bean.FacilityTreeAttributeConstant;
import com.clustercontrol.syslogng.bean.LogFilterInfo;

/**
 * フィルタとリポジトリ情報を保持するクラスです。
 * フィルタ情報と関連するファシリティ情報の一覧を格納します。
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class LogFilterRepositoryInfo {
	
	/**
	 * ファシリティ情報マップ。
	 * <dl>
	 *  <dt>ファシリティ情報マップ</dt>
	 *  <dd>キー：ファシリティID</dd>
	 *  <dd>値　：属性名と値のHashMap</dd>
	 * </dl>
	 */
	private HashMap m_map = new HashMap();
	
	/** フィルタ情報。 */
	private LogFilterInfo m_filter = null;
	
	/** ログ出力のインスタンス。 */
	@SuppressWarnings("unused")
	static Log log = LogFactory.getLog(LogFilterRepositoryInfo.class);
	
	/**
	 * コンストラクタ。
	 * 
	 * @param filter フィルタ情報
	 */
	public LogFilterRepositoryInfo(LogFilterInfo filter) {
		super();
		m_filter = filter;
	}
	
	/**
	 * ファシリティ情報を設定します。
	 * 
	 * @param map ファシリティ情報マップ
	 */
	public void setMap(HashMap map) {
		m_map = map;
		
		
	}
	
	
	
	
	/**
	 * ファシリティIDを返します。<BR>
	 * 引数で指定されたノード名のファシリティIDを返します。
	 * 存在しない場合は、<code> null </code>を返します。
	 * 
	 * @param value ノード名/IPアドレス
	 * @return ファシリティID
	 */
	public String contains(String value, boolean include) {
		
		if(log.isDebugEnabled()){
			log.debug("contains() start value = " + value + ", include = " + include);
		}
		
		if(value == null)
			return null;

		
		if(include){ //普通の場合		

			if(log.isDebugEnabled()){
				log.debug("contains() inlcude route");
			}

			//HashMap(の要素)→Set
			Set entrySet = m_map.entrySet();
			for (Iterator iter = entrySet.iterator(); iter.hasNext();) {
				//Set(の要素）→Map.Entry
				Map.Entry  entry = (Map.Entry ) iter.next();
				//Map.Entry(変換)→HashMap
				HashMap attr = (HashMap)entry.getValue();

				//ノード名による一致確認
				String host = (String)attr.get(FacilityAttributeConstant.NODENAME);
				if(host != null && host.equalsIgnoreCase(value) ){
					if(log.isDebugEnabled())
						log.debug("contains() Node Name Match facilityId = " + (String)entry.getKey());
					return (String)entry.getKey();
				}
				
				
				// for debug
				Integer ipproto = (Integer)attr.get(FacilityAttributeConstant.IPPROTOCOLNUMBER);
				if(log.isDebugEnabled()){
					log.debug("contains() ipproto = " + (Integer)attr.get(FacilityAttributeConstant.IPPROTOCOLNUMBER));
					log.debug("contains() ipv4 = " + (String)attr.get(FacilityAttributeConstant.IPNETWORKNUMBER));
					log.debug("contains() ipv6 = " + (String)attr.get(FacilityAttributeConstant.IPNETWORKNUMBERV6));
				}
				//IPアドレスによる一致確認
				if(ipproto == null || ipproto.equals(4)){
					// IPPROTOCOLNUMBERが指定がない、または4の場合はv4
					String address = (String)attr.get(FacilityAttributeConstant.IPNETWORKNUMBER);
					if(address != null && address.equalsIgnoreCase(value)){
						if(log.isDebugEnabled())
							log.debug("contains() IPv4 Address Match facilityId = " + (String)entry.getKey());
						return (String)entry.getKey();
					}
				}else if(ipproto.equals(6)){
					// 6指定の場合はv6
					String address = (String)attr.get(FacilityAttributeConstant.IPNETWORKNUMBERV6);
					if(address != null && address.equalsIgnoreCase(value)){
						if(log.isDebugEnabled())
							log.debug("contains() IPv6 Address Match facilityId = " + (String)entry.getKey());
						return (String)entry.getKey();
					}
				}
			}
			if(log.isDebugEnabled())
				log.debug("contains() Not Match");
			return null;
		}
		else{
			
			//UNREGISTEREDスコープ対応
			
			Set entrySet = m_map.entrySet();
			for (Iterator iter = entrySet.iterator(); iter.hasNext();) {
				//Set(の要素）→Map.Entry
				Map.Entry  entry = (Map.Entry ) iter.next();
				//Map.Entry(変換)→HashMap
				HashMap attr = (HashMap)entry.getValue();
				//HashMap(の要素）→String 
				String host = (String)attr.get(FacilityAttributeConstant.NODENAME);
				if(host != null && host.equalsIgnoreCase(value) ){
					//1つホストでもマッチしたらNULL
					return null;
				}
			}
			return FacilityTreeAttributeConstant.UNREGISTEREFD_SCOPE;	
		}
	}
	
	
	
	
	
	/**
	 * ファシリティ名を返します。
	 * 引数で指定されたファシリティIDのファシリティ名を返します。
	 * 
	 * @param facilityID ファシリティID
	 * @return ファシリティ名
	 */
	public String getNodeName(String facilityID) {
		
		HashMap attr = (HashMap)m_map.get(facilityID);
		String facilityName;
		
		if(attr == null){
			facilityName = FacilityTreeAttributeConstant.UNREGISTERED_SCOPE_TEXT;
			
		}else{
			facilityName=  (String)attr.get(FacilityAttributeConstant.CN);
			
		}
		return facilityName;
		
	}
	
	/**
	 * フィルタ情報を返します。
	 * 
	 * @return フィルタ情報
	 */
	public LogFilterInfo getFilter() {
		return m_filter;
	}
}
