/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.agent;

import java.io.File;
import java.util.ArrayList;
import java.util.Date;
import java.util.Hashtable;
import java.util.List;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.agent.util.RunHistoryUtil;
import com.clustercontrol.jobmanagement.bean.RunStatusConstant;
import com.clustercontrol.jobmanagement.message.RunInstructionInfo;
import com.clustercontrol.jobmanagement.message.RunResultInfo;

/**
 * ファイル転送ジョブ用ファイルリスト取得スレッドクラス<BR>
 * 
 * Hinemosのファイル転送ジョブでは、ファイル指定にワールドカード指定を
 * 許します。<BR>
 * そこで、このスレッドにより、そのワイルドカードの展開を
 * システムに問い合わせる形で行います。<BR>
 * <BR>
 * また、ファイルが存在しているかの存在確認も行います。
 *
 * @version 2.1.0
 * @since 2.0.0
 */
public class FileListThread extends AgentThread {
	
	//ロガー
	private static Log log = LogFactory.getLog(FileListThread.class);
	
	/**
	 * コンストラクタ
	 * 
	 * @param props
	 */
	public FileListThread(
			RunInstructionInfo info, 
			SendQueue sendQueue, 
			Hashtable<String, Date> runHistory) {
		super(info, sendQueue, runHistory);
	}
	
	/**
	 * ファイルリストを取得します。<BR>
	 * 
	 * ReceiveTopicで受け取ったジョブの指示がファイルリストの場合に
	 * このメソッドが実行されます。
	 */
	/* (non-Javadoc)
	 * @see java.lang.Runnable#run()
	 */
	@Override
	public void run() {
		log.debug("run start");
		
        Date startDate = new Date();
        
        //実行履歴に追加
		RunHistoryUtil.addRunHistory(m_info, m_runHistory, startDate);
		
		//---------------------------
		//-- 開始メッセージ送信
		//---------------------------
		
		//メッセージ作成
		RunResultInfo info = new RunResultInfo();
		info.setSessionId(m_info.getSessionId());
		info.setJobunitId(m_info.getJobunitId());
		info.setJobId(m_info.getJobId());
		info.setFacilityId(m_info.getFacilityId());
		info.setCommand(m_info.getCommand());
		info.setCommandType(m_info.getCommandType());
		info.setStatus(RunStatusConstant.START);
		info.setTime(startDate);
		
		log.info("run SessionID=" + m_info.getSessionId() + ", JobID=" + m_info.getJobId());
		
		//送信
		m_sendQueue.put(info);
		
		//ファイルリスト取得
		List<String> fileList = getFileList(m_info.getFilePath());
		if(fileList != null && fileList.size() > 0){
			info.setStatus(RunStatusConstant.END);
			info.setFileList(fileList);
			info.setTime(new Date());
			info.setErrorMessage("");
			info.setMessage("");
			info.setEndValue(0);
		}
		else if(fileList != null){
			//FIXME ファイルが１つもないときはagent.logに
			//ログを出力したほうがよい。
			info.setStatus(RunStatusConstant.END);
			info.setTime(new Date());
			info.setErrorMessage("");
			info.setMessage("");
			info.setEndValue(1);
		}
		else{
			//FIXME ファイルリストがnullの場合ってExceptionが発生した場合。
			info.setStatus(RunStatusConstant.END);
			info.setTime(new Date());
			info.setErrorMessage("");
			info.setMessage("");
			info.setEndValue(-1);
		}
		
		//送信
		m_sendQueue.put(info);
		
        //実行履歴削除メッセージ送信
        sendDeleteHistory(info);
		
		log.debug("run end");
	}
	
	/**
	 * run()から呼び出されるファイルリスト取得部分のメソッド<BR>
	 * 
	 * @param path
	 * @return
	 */
	public synchronized List<String> getFileList(String path) {
		log.debug("get file list start");
		
		List<String> fileList = new ArrayList<String>();
		
		try{
			String dir = path.substring(0, path.lastIndexOf("/") + 1);
			String fileName = path.substring(path.lastIndexOf("/") + 1);
			fileName = fileName.replaceAll("[.]", "[.]");
			fileName = fileName.replaceAll("[*]", ".*");
			if(fileName.length() == 0)
				fileName = ".*";
		
			//Fileを指定パスで作成
			File fi = new File(dir);

			File[] files = fi.listFiles();
			for(int i = 0; i < files.length; i++){
				if(files[i].isFile() && !files[i].isHidden()){
					if(files[i].getName().matches(fileName))
						fileList.add(files[i].getCanonicalPath());
				}
			}
		}
		catch(Exception e){
			//FIXME Exceptionをキャッチしてnullは無しでお願いします。
			fileList = null;
		}
		
		log.debug("get file list end");
		return fileList;
	}
}
