/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.notify.factory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.bean.NotifyTypeConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.notify.bean.NotifyEventInfo;
import com.clustercontrol.notify.bean.NotifyInfo;
import com.clustercontrol.notify.bean.NotifyInfoDetail;
import com.clustercontrol.notify.bean.NotifyJobInfo;
import com.clustercontrol.notify.bean.NotifyLogEscalateInfo;
import com.clustercontrol.notify.bean.NotifyMailInfo;
import com.clustercontrol.notify.bean.NotifyStatusInfo;
import com.clustercontrol.notify.bean.NotifyUseTyperConstant;
import com.clustercontrol.notify.ejb.entity.NotifyEventInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyInfoUtil;
import com.clustercontrol.notify.ejb.entity.NotifyJobInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyLogEscalateInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyMailInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyStatusInfoLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * 通知情報を検索するクラスです。
 *
 * @version 3.0.0
 * @since 1.0.0
 */
public class SelectNotify {
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog( SelectNotify.class );
	
	/**
	 * 通知情報を返します。
	 * 
	 * @param notifyId 取得対象の通知ID
	 * @return 通知情報
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.notify.ejb.entity.NotifyInfoBean
	 * @see com.clustercontrol.notify.ejb.entity.NotifyEventInfoBean
	 */
	public NotifyInfo getNotify(String notifyId) throws FinderException, NamingException {

		
			// 通知情報を取得
			NotifyInfoLocal notify = NotifyInfoUtil.getLocalHome().findByPrimaryKey(notifyId);
			
			return getNotify(notify);
			
		
		
	}
	
	public NotifyInfo getNotify(NotifyInfoLocal notify) throws FinderException, NamingException {
		
		NotifyInfo bean = null;
		String notifyId = notify.getNotifyId();
		
		ArrayList<NotifyInfoDetail> notifyEvents = getNotifyDetails(notifyId,notify);
		
		bean = new NotifyInfo(
				notify.getNotifyId(),
				notify.getDescription(),
				notify.getNotifyType(),
				notify.getInhibitionFlg(),
				notify.getInhibitionFrequency(),
				notify.getInhibitionPeriod(),
				notify.getRegDate(),
				notify.getUpdateDate(),
				notify.getRegUser(),
				notify.getUpdateUser(),
				notify.getValidFlg(),
				notifyEvents
		);
		return bean;
	}
	
	/**
	 * 通知詳細情報を取得します。
	 * 
	 * @param notifyId
	 * @param notify
	 * @return
	 */
	public ArrayList getNotifyDetails(String notifyId, NotifyInfoLocal notify)	{

		
		ArrayList notifyEvents = new ArrayList();
	
		try {
			RepositoryControllerLocal repository = RepositoryControllerUtil.getLocalHome().create();
			
			if(notify.getNotifyType() ==  com.clustercontrol.bean.NotifyTypeConstant.TYPE_STATUS ){
				Collection ct = notify.getNotifyStatusInfo();
				
			    Iterator itr = ct.iterator();
			    NotifyStatusInfoLocal local = null;
			    while(itr.hasNext()){
			        local = (NotifyStatusInfoLocal)itr.next();
			        NotifyStatusInfo status = new NotifyStatusInfo(
			        		notifyId,
							local.getPriority(),
							local.getStatusFlg(),
							local.getStatusInhibitionFlg(),
							local.getStatusInvalidFlg(),
			        		local.getStatusUpdatePriority(),
			        		local.getStatusValidPeriod());
			        		
			        notifyEvents.add(status);
			    }
			
			}else if(notify.getNotifyType() ==  com.clustercontrol.bean.NotifyTypeConstant.TYPE_EVENT){
				
				Collection ct = notify.getNotifyEventInfo();
				
			    Iterator itr = ct.iterator();
			    NotifyEventInfoLocal local = null;
			    while(itr.hasNext()){
			        local = (NotifyEventInfoLocal)itr.next();
			        NotifyEventInfo event = new NotifyEventInfo(
			        		notifyId,
							local.getPriority(),
							local.getEventNormalFlg(),
							local.getEventNormalState(),
							local.getEventInhibitionFlg(),
							local.getEventInhibitionState());
			        
			        notifyEvents.add(event);
			    }
			
			}else if(notify.getNotifyType() ==  com.clustercontrol.bean.NotifyTypeConstant.TYPE_MAIL){
				Collection ct = notify.getNotifyMailInfo();
				
			    Iterator itr = ct.iterator();
			    NotifyMailInfoLocal local = null;
			    while(itr.hasNext()){
			        local = (NotifyMailInfoLocal)itr.next();
			        NotifyMailInfo status = new NotifyMailInfo(
			        		notifyId,
							local.getPriority(),
							local.getMailTemplateId(),
							local.getMailFlg(),
							local.getMailInhibitionFlg(),
							local.getMailAddress());
			        		
			        notifyEvents.add(status);
			    }
			
			}else if(notify.getNotifyType() ==  com.clustercontrol.bean.NotifyTypeConstant.TYPE_JOB){
			
				Collection ct = notify.getNotifyJobInfo();
				
			    Iterator itr = ct.iterator();
			    NotifyJobInfoLocal local = null;
			    while(itr.hasNext()){
			        local = (NotifyJobInfoLocal)itr.next();
			        
			        String facilityPath = repository.getFacilityPath(local.getJobExecFacility(), null);
			        
			        NotifyJobInfo status = new NotifyJobInfo(
			        		notifyId,
							local.getPriority(),
							local.getJobRun(),
							local.getJobId(),
							local.getJobInhibitionFlg(),
							local.getJobFailurePriority(),
			        		local.getJobExecFacilityFlg(),
			        		local.getJobExecFacility(),
			        		facilityPath);
							
			        notifyEvents.add(status);
			    }
			    
			}else if(notify.getNotifyType() ==  com.clustercontrol.bean.NotifyTypeConstant.TYPE_LOG_ESCALATE){
			
				Collection ct = notify.getNotifyLogEscalateInfo();
				
			    Iterator itr = ct.iterator();
			    NotifyLogEscalateInfoLocal local = null;
			    while(itr.hasNext()){
			        local = (NotifyLogEscalateInfoLocal)itr.next();
			        
			        String facilityPath = repository.getFacilityPath(local.getEscalateFacility(), null);
			        
			        NotifyLogEscalateInfo status = new NotifyLogEscalateInfo(
			        		notifyId,
							local.getPriority(),
							local.getEscalateFlg(),
							local.getEscalateInhibitionFlg(),
							local.getEscalateMessage(),
							local.getSyslogPriority(),
							local.getSyslogFacility(),
							local.getEscalateFacilityFlg(),
							local.getEscalateFacility(),
							facilityPath,
							local.getEscalatePort());
							
			        notifyEvents.add(status);
			    }
			}
		}
		catch (NamingException e) {
			m_log.error("getNotifyDetails() : " + e.getMessage(), e);
		}
		catch (FinderException e) {
			m_log.error("getNotifyDetails() : " + e.getMessage(), e);
		} 
		catch (CreateException e) {
			m_log.error("getNotifyDetails() : " + e.getMessage(), e);
		}
	
    return notifyEvents;
}
	
	/**
	 * 通知ID一覧を返します。
	 * <p>
	 * 通知IDの昇順に並んだ通知ID一覧を返します。
	 * 
	 * @return 通知ID一覧
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.notify.ejb.entity.NotifyInfoBean
	 */
	public ArrayList getNotifyIdList() throws FinderException, NamingException {

		// 戻り値の初期化
		ArrayList<String> list = new ArrayList<String>();
		
		try 
		{
			// 通知情報一覧を取得
			Collection ct = NotifyInfoUtil.getLocalHome().findAllOrderByNotifyId();
			
			Iterator itr = ct.iterator();
			while(itr.hasNext())
			{
				NotifyInfoLocal notify = (NotifyInfoLocal)itr.next();
				list.add(notify.getNotifyId());
			}
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            apllog.put("SYS", "005");
            m_log.debug("getNotifyList():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            apllog.put("SYS", "005");
            m_log.debug("getNotifyList():" + e.getMessage());
			throw e;
		}
		return list;
	}

	/**
	 * 通知情報一覧を返します(障害検知用通知を除く)。
	 * <p>
	 * <ol>
	 * <li>通知IDの昇順に並んだ全ての通知情報を取得します。</li>
	 * <li>１通知情報をテーブルのカラム順（{@link com.clustercontrol.notify.bean.NotifyTableDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 * <li>この１通知情報を保持するリストを、通知情報一覧を保持するリスト（{@link ArrayList}）に格納し返します。<BR>
	 *  <dl>
	 *  <dt>通知情報一覧（Objectの2次元配列）</dt>
	 *  <dd>{ 通知情報1 {カラム1の値, カラム2の値, … }, 通知情報2{カラム1の値, カラム2の値, …}, … }</dd>
	 *  </dl>
	 * </li>
	 * </ol>
	 * 
	 * @return 通知情報一覧（Objectの2次元配列）
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.notify.bean.NotifyTableDefine
	 * @see #collectionToArray(Collection)
	 */
	public ArrayList<NotifyInfoLocal> getNotifyList() throws CreateException, FinderException, NamingException {

		ArrayList<NotifyInfoLocal> list = null;
		try 
		{
			// 通知情報一覧を取得
			Collection<NotifyInfoLocal> ct = NotifyInfoUtil.getLocalHome().findAllOrderByNotifyId();
			Collection<NotifyInfoLocal> retct = NotifyInfoUtil.getLocalHome().findAllOrderByNotifyId();
			
			if(ct.size() != 0) {
				
				Iterator<NotifyInfoLocal> itr = ct.iterator();
				
				// 障害検知用の通知IDを除く
				while(itr.hasNext()){
					
					NotifyInfoLocal notify = itr.next();
					String regex = "^" + HinemosModuleConstant.SYSYTEM_TROUBLE_DETECTION + ".*";
					
					// FIXME : 文字列の比較による障害検知用通知IDの特定方法を変更する
					// 結果用Collectionから削除対象のCollectionを削除する
					if(notify.getNotifyId().matches(regex)){
						retct.remove(notify);
					}
				}
			}
			
			// 2次元配列に変換
			list = this.collectionToArray(retct);
			
		} catch (CreateException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            apllog.put("SYS", "006");
            m_log.debug("getNotifyList():" + e.getMessage());
			throw e;
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            apllog.put("SYS", "006");
            m_log.debug("getNotifyList():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            apllog.put("SYS", "006");
            m_log.debug("getNotifyList():" + e.getMessage());
			throw e;
		}
		return list;
	}
	
	/**
	 * 通知情報マップを返します。
	 * <p>
	 * <ol>
	 * <li>全ての通知情報を取得します。</li>
	 * <li>通知情報マップに、通知情報をセットします。</li>
	 *  <dl>
	 *   <dt>通知情報マップ</dt>
	 *   <dd>キー：通知ID</dd>
	 *   <dd>値　：通知情報</dd>
	 *  </dl>
	 * </li>
	 * </ol>
	 * 
	 * @return 通知情報マップ
	 * @throws FinderException
	 * @throws NamingException
	 */
	public HashMap<String, NotifyInfo> getNotifyMap() throws FinderException, NamingException {

		HashMap<String, NotifyInfo> map = new HashMap<String, NotifyInfo>();
		try 
		{
			// 通知情報一覧を取得
			Collection ct = NotifyInfoUtil.getLocalHome().findAll();
			
			Iterator itr = ct.iterator();
			NotifyInfo bean = null;
			NotifyInfoLocal notify = null;

			while(itr.hasNext()){
				
				notify = (NotifyInfoLocal)itr.next();
			    bean = getNotify(notify);
				
		        map.put(notify.getNotifyId(), bean);
			}
			
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            apllog.put("SYS", "006");
            m_log.debug("getNotifyMap():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            apllog.put("SYS", "006");
            m_log.debug("getNotifyMap():" + e.getMessage());
			throw e;
		}
		return map;
	}

	/**
	 * 障害検知用通知情報一覧を返します。
	 * <p>
	 * <ol>
	 * <li>通知IDの昇順に並んだ全ての通知情報を取得します。</li>
	 * <li>１通知情報をテーブルのカラム順（{@link com.clustercontrol.notify.bean.NotifyTableDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 * <li>この１通知情報を保持するリストを、通知情報一覧を保持するリスト（{@link ArrayList}）に格納し返します。<BR>
	 *  <dl>
	 *  <dt>通知情報一覧（Objectの2次元配列）</dt>
	 *  <dd>{ 通知情報1 {カラム1の値, カラム2の値, … }, 通知情報2{カラム1の値, カラム2の値, …}, … }</dd>
	 *  </dl>
	 * </li>
	 * </ol>
	 * 
	 * @return 通知情報一覧（Objectの2次元配列）
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.notify.bean.NotifyTableDefine
	 * @see #collectionToArray(Collection)
	 */
	public ArrayList<NotifyInfoLocal> getTroubleDetectionNotifyList() throws CreateException, FinderException, NamingException {

		ArrayList<NotifyInfoLocal> list = null;
		try 
		{
			// 通知情報一覧を取得
			Collection<NotifyInfoLocal> ct = NotifyInfoUtil.getLocalHome().findAllOrderByNotifyId();
			Collection<NotifyInfoLocal> retct = NotifyInfoUtil.getLocalHome().findAllOrderByNotifyId();
			
			if(ct.size() != 0) {
			
				Iterator<NotifyInfoLocal> itr = ct.iterator();
				
				// 障害検知用の通知IDのみを取得
				while(itr.hasNext()){
					
					NotifyInfoLocal notify = itr.next();
					String regex = "^" + HinemosModuleConstant.SYSYTEM_TROUBLE_DETECTION + ".*";
					
					// FIXME : 文字列の比較による障害検知用通知IDの特定方法を変更する
					// 結果用Collectionから削除対象のCollectionを削除する
					if(!notify.getNotifyId().matches(regex)){
						retct.remove(notify);
					}
				}
			}
			
			// 2次元配列に変換
			list = this.collectionToArray(retct);
			
		} catch (CreateException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            apllog.put("SYS", "006");
            m_log.debug("getNotifyList():" + e.getMessage());
			throw e;
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            apllog.put("SYS", "006");
            m_log.debug("getNotifyList():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            apllog.put("SYS", "006");
            m_log.debug("getNotifyList():" + e.getMessage());
			throw e;
		}
		return list;
	}
	
	/**
	 * 障害検知用通知情報マップを返します。
	 * <p>
	 * <ol>
	 * <li>障害検知用の通知情報を取得します。</li>
	 * <li>通知情報マップに、通知情報をセットします。</li>
	 *  <dl>
	 *   <dt>通知情報マップ</dt>
	 *   <dd>キー：通知ID</dd>
	 *   <dd>値　：通知情報</dd>
	 *  </dl>
	 * </li>
	 * </ol>
	 * 
	 * @return 通知情報マップ
	 * @throws FinderException
	 * @throws NamingException
	 */
	public HashMap<String, NotifyInfo> getTroubleDetectionNotifyMap() throws FinderException, NamingException {

		HashMap<String, NotifyInfo> map = new HashMap<String, NotifyInfo>();
		try 
		{
			// 通知情報一覧を取得
			Collection<NotifyInfoLocal> ct = NotifyInfoUtil.getLocalHome().findByUseType(HinemosModuleConstant.SYSYTEM_TROUBLE_DETECTION+"%");
			
			Iterator<NotifyInfoLocal> itr = ct.iterator();
			NotifyInfo bean = null;
			NotifyInfoLocal notify = null;

			while(itr.hasNext()){
				
				notify = (NotifyInfoLocal)itr.next();
				bean = getNotify(notify);
				
				map.put(notify.getNotifyId(), bean);
			}
			
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			apllog.put("SYS", "006");
			m_log.debug("getNotifyMap():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			apllog.put("SYS", "006");
			m_log.debug("getTroubleDetectionNotifyMap():" + e.getMessage());
			throw e;
		}
		return map;
	}
	
	
	/**
	 * DBより取得した通知情報のObjectの2次元配列を返します。
	 * <p>
	 * <ol>
	 * <li>通知情報をテーブルのカラム順（{@link com.clustercontrol.notify.bean.NotifyTableDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 * <li>1つの通知情報を保持するリストを、通知情報一覧を保持するリスト（{@link ArrayList}）に格納します。
	 * <p>
	 *  <dl>
	 *  <dt>通知情報一覧（Objectの2次元配列）</dt>
	 *  <dd>{ 通知情報1 {カラム1の値, カラム2の値, … }, 通知情報2{カラム1の値, カラム2の値, …}, … }</dd>
	 *  </dl>
	 * </li>
	 * </ol>
	 * 
	 * @param ct 通知情報のコレクション
	 * @return 通知情報一覧（Objectの2次元配列）
	 * 
	 * @see com.clustercontrol.notify.bean.NotifyTableDefine
	 */
	@SuppressWarnings("unchecked")
	private ArrayList collectionToArray(Collection ct) throws CreateException, FinderException, NamingException{
		
		ArrayList list = new ArrayList();
		Iterator itr = ct.iterator();
		while(itr.hasNext())
		{
			NotifyInfoLocal notify = (NotifyInfoLocal)itr.next();
			
			ArrayList info = new ArrayList();
			info.add(YesNoConstant.BOOLEAN_NO);
			info.add(notify.getValidFlg().intValue());
			info.add(notify.getNotifyId());
			info.add(notify.getDescription());
			info.add(NotifyTypeConstant.typeToString(notify.getNotifyType()));
			info.add(notify.getUpdateDate() == null ? null:new Date(notify.getUpdateDate().getTime()));
			list.add(info);
		}
		return list;
	}
}
