/*
 
Copyright (C) 2009 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.audit.log;

import org.jboss.aop.Aspect;
import org.jboss.aop.Bind;
import org.jboss.aop.advice.Scope;
import org.jboss.aop.joinpoint.MethodInvocation;

import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.monitor.run.bean.MonitorInfo;

/**
 * 監視機能 操作ログ出力クラス
 * 
 * @since 1.0.0
 * @version 1.0.0
 * @see com.clustercontrol.audit.log.Logger
 */
@Aspect (scope= Scope.PER_VM)
public class MonitorRun extends Logger {

	/**
	 * ログ出力<br>
	 * SNMPTRAPマスタ関連のメソッドの場合ログ出力しない 
	 * 
	 * @see com.clustercontrol.audit.log.Logger#outputLogMessage(java.lang.String, java.lang.String, org.jboss.aop.joinpoint.MethodInvocation)
	 */
	@Override
	public void outputLogMessage(String title, String argMessage,
			MethodInvocation invocation) throws Exception {

		String name = invocation.getMethod().getName();
		if(name.matches(".*MibMaster") || name.matches(".*OidMaster"))
			return;

		super.outputLogMessage(title, argMessage, invocation);
	}

	/**
	 * 監視設定追加時のログ出力
	 * 
	 * @param invocation 
	 * @return invokeNext()の結果
	 * @throws Throwable
	 */
	@Bind (pointcut="execution(public boolean com.clustercontrol.*.ejb.session.Monitor*ControllerBean->add*(..))")
	public Object addAdvice(MethodInvocation invocation) throws Throwable {
		Object result = invocation.invokeNext();

		StringBuffer title = new StringBuffer();
		StringBuffer msg = new StringBuffer();

		Object[] args = ((MethodInvocation)invocation).getArguments();
		if(args.length > 0 && args[0] instanceof MonitorInfo){
			MonitorInfo info = (MonitorInfo)args[0];

			title.append(monitorTypeToTitle(info.getMonitorTypeId()));
			title.append(" Add");

			msg.append(", MonitorID=");
			msg.append(info.getMonitorId());
		}

		outputLogMessage(title.toString(), msg.toString(), invocation);

		return result;
	}

	/**
	 * 監視設定変更時のログ出力
	 * 
	 * @param invocation 
	 * @return invokeNext()の結果
	 * @throws Throwable
	 */
	@Bind (pointcut="execution(public boolean com.clustercontrol.*.ejb.session.Monitor*ControllerBean->modify*(..))")
	public Object modifyAdvice(MethodInvocation invocation) throws Throwable {
		Object result = invocation.invokeNext();

		StringBuffer title = new StringBuffer();
		StringBuffer msg = new StringBuffer();

		Object[] args = ((MethodInvocation)invocation).getArguments();
		if(args.length > 0 && args[0] instanceof MonitorInfo){
			MonitorInfo info = (MonitorInfo)args[0];

			title.append(monitorTypeToTitle(info.getMonitorTypeId()));
			title.append(" Change");

			msg.append(", MonitorID=");
			msg.append(info.getMonitorId());
		}

		outputLogMessage(title.toString(), msg.toString(), invocation);

		return result;
	}

	/**
	 * 監視設定削除時のログ出力
	 * 
	 * @param invocation 
	 * @return invokeNext()の結果
	 * @throws Throwable
	 */
	@Bind (pointcut="execution(public boolean com.clustercontrol.*.ejb.session.Monitor*ControllerBean->delete*(..))")
	public Object deleteAdvice(MethodInvocation invocation) throws Throwable {
		Object result = invocation.invokeNext();

		StringBuffer title = new StringBuffer();
		title.append(classNameToTitle(((MethodInvocation)invocation).getTargetObject().getClass().getName()));
		title.append(" Delete");

		StringBuffer msg = new StringBuffer();

		Object[] args = ((MethodInvocation)invocation).getArguments();
		if(args.length > 0 && args[0] instanceof String){
			msg.append(", MonitorID=");
			msg.append((String)args[0]);
		} else{
			
			// もし該当する監視種別がない場合はログを出さずに終了する
			return result;
		}
			

		outputLogMessage(title.toString(), msg.toString(), invocation);

		return result;
	}

	/**
	 * 監視種別からタイトルを取得する
	 * 
	 * @param type 監視種別
	 * @return タイトル
	 */
	protected String monitorTypeToTitle(String type){
		if (type.equals(HinemosModuleConstant.MONITOR_AGENT)) {
			return "[Agent]";
		} else if (type.equals(HinemosModuleConstant.MONITOR_HTTP)) {
			return "[Http]";
		} else if (type.equals(HinemosModuleConstant.MONITOR_PING)) {
			return "[Ping]";
		} else if (type.equals(HinemosModuleConstant.MONITOR_PROCESS)) {
			return "[Process]";
		} else if (type.equals(HinemosModuleConstant.MONITOR_SNMP)) {
			return "[Snmp]";
		} else if (type.equals(HinemosModuleConstant.MONITOR_SNMPTRAP)) {
			return "[Snmptrap]";
		} else if (type.equals(HinemosModuleConstant.MONITOR_SQL)) {
			return "[Sql]";
		} else if (type.equals(HinemosModuleConstant.MONITOR_PERFORMANCE)) {
			return "[Resource]";
		} else if (type.equals(HinemosModuleConstant.MONITOR_PORT)) {
			return "[Port]";
		}
		return "";
	}

	/**
	 * クラス名からタイトルを取得する
	 * 
	 * @param name クラス名
	 * @return タイトル
	 */
	protected String classNameToTitle(String name){
		if (name.matches("com\\.clustercontrol\\.hinemosagent.*")) {
			return "[Agent]";
		} else if (name.matches("com\\.clustercontrol\\.http.*")) {
			return "[Http]";
		} else if (name.matches("com\\.clustercontrol\\.ping.*")) {
			return "[Ping]";
		} else if (name.matches("com\\.clustercontrol\\.process.*")) {
			return "[Process]";
		} else if (name.matches("com\\.clustercontrol\\.snmptrap.*")) {
			return "[Snmptrap]";
		} else if (name.matches("com\\.clustercontrol\\.snmp.*")) {
			return "[Snmp]";
		} else if (name.matches("com\\.clustercontrol\\.sql.*")) {
			return "[Sql]";
		} else if (name.matches("com\\.clustercontrol\\.performance.*")) {
			return "[Resource]";
		} else if (name.matches("com\\.clustercontrol\\.port.*")) {
			return "[Port]";
		}
		else{
			return name;
		}
	}
}
