/*
Copyright (C) 2010 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
*/
package com.clustercontrol.ws.endpoint;

import java.util.ArrayList;
import java.util.Locale;

import javax.ejb.CreateException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.CalendarNotFountException;
import com.clustercontrol.FacilityNotFoundException;
import com.clustercontrol.HinemosUnknownException;
import com.clustercontrol.JobInfoNotFoundException;
import com.clustercontrol.JobInvalidException;
import com.clustercontrol.JobMasterNotFoundException;
import com.clustercontrol.NotifyNotFoundException;
import com.clustercontrol.UsedFacilityException;
import com.clustercontrol.UserNotFoundException;
//import com.clustercontrol.bean.CommonTableTreeItem;
import com.clustercontrol.bean.OutputBasicInfo;
import com.clustercontrol.jobmanagement.bean.JobForwardFile;
import com.clustercontrol.jobmanagement.bean.JobHistoryFilter;
import com.clustercontrol.jobmanagement.bean.JobHistoryList;
import com.clustercontrol.jobmanagement.bean.JobNodeDetail;
import com.clustercontrol.jobmanagement.bean.JobSchedule;
import com.clustercontrol.jobmanagement.bean.JobTreeItem;
import com.clustercontrol.jobmanagement.bean.JobTriggerInfo;
import com.clustercontrol.jobmanagement.bean.NodeOperationInfo;
import com.clustercontrol.jobmanagement.ejb.session.JobControllerLocal;
import com.clustercontrol.jobmanagement.ejb.session.JobControllerUtil;
import com.clustercontrol.notify.bean.NotifyRequestMessage;

@javax.jws.WebService
public class JobEndpoint {
	protected static Log m_log = LogFactory.getLog( JobEndpoint.class );
	
	public String echo(String str) {
		return str + ", " + str;
	}
	
	private JobControllerLocal getJobControllerLocal() throws HinemosUnknownException {
		JobControllerLocal monitor = null;
		try {
			monitor = JobControllerUtil.getLocalHome().create();
		} catch (CreateException e) {
			m_log.error("getMonitorSyslogNGControllerLocal CreateException : " + e,e);
			throw new HinemosUnknownException(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error("getMonitorSyslogNGControllerLocal NamingException : " + e,e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return monitor;
	}
	
	
	/**
	 * ジョブツリー情報を取得する。<BR>
	 * 
	 * @param treeOnly
	 * @throws HinemosUnknownException 
	 * @throws UserNotFoundException 
	 * @throws JobMasterNotFoundException 
	 * @throws NotifyNotFoundException 
	 */
	public JobTreeItem getJobTree(boolean treeOnly) throws HinemosUnknownException, NotifyNotFoundException, JobMasterNotFoundException, UserNotFoundException {
		JobControllerLocal job = getJobControllerLocal();
		return job.getJobTree(treeOnly, Locale.getDefault());
	}
	
	/**
	 * 同一階層のジョブツリー情報を取得する。<BR>
	 * 
	 * @param parentJobunitId 親ジョブのジョブユニットID
	 * @param parentJobId 親ジョブID
	 * @param locale ロケール情報
	 * @return ジョブツリー情報{@link com.clustercontrol.jobmanagement.bean.JobTreeItem}の階層オブジェクト
	 * @throws JobMasterNotFoundException 
	 * @throws HinemosUnknownException 
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getJobTreeOneLevel(String, Locale)
	 */
	public JobTreeItem getJobTreeOneLevel(String parentJobunitId, String parentJobId) throws JobMasterNotFoundException, HinemosUnknownException {
		JobControllerLocal job = getJobControllerLocal();
		return job.getJobTreeOneLevel(parentJobunitId, parentJobId, Locale.getDefault());
	}
	
	/**
	 * ジョブツリー情報を登録する。<BR>
	 * 
	 * @param item ジョブツリー情報{@link com.clustercontrol.jobmanagement.bean.JobTreeItem}の階層オブジェクト
	 * @throws HinemosUnknownException 
	 * @throws JobMasterNotFoundException 
	 * @throws JobInvalidException 
	 * @see com.clustercontrol.jobmanagement.factory.ModifyJob#registerJob(JobTreeItem, String)
	 */
	public void registerJob(JobTreeItem item) throws HinemosUnknownException, JobMasterNotFoundException, JobInvalidException {
		JobControllerLocal job = getJobControllerLocal();
		job.registerJob(item);
	}
	
	/**
	 * ジョブユニット単位でジョブツリー情報を登録する。<BR>
	 * 
	 * @param item ジョブユニットのジョブツリー情報{@link com.clustercontrol.jobmanagement.bean.JobTreeItem}の階層オブジェクト
	 * @throws NotifyNotFoundException 
	 * @throws HinemosUnknownException 
	 * @throws JobMasterNotFoundException 
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.ModifyJob#registerJobunit(JobTreeItem, String)
	 */
	public void registerJobunit(JobTreeItem jobunit) throws NotifyNotFoundException, HinemosUnknownException, JobMasterNotFoundException {
		JobControllerLocal job = getJobControllerLocal();
		job.registerJobunit(jobunit);
	}
	
	/**
	 * ジョブユニット単位でジョブツリー情報を削除する。<BR>
	 * 
	 * @param jobunitId 削除対象ジョブユニットのジョブID
	 * @throws NotifyNotFoundException 
	 * @throws HinemosUnknownException 
	 * @throws JobMasterNotFoundException 
	 */
	public void deleteJobunit(String jobunitId) throws NotifyNotFoundException, HinemosUnknownException, JobMasterNotFoundException {
		JobControllerLocal job = getJobControllerLocal();
		job.deleteJobunit(jobunitId);
	}
	
	
	/**
	 * ジョブ操作開始用プロパティを返します。<BR>
	 * 
	 * @param sessionId セッションID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @param facilityId ファシリティID
	 * @param locale ロケール情報
	 * @return ジョブ操作開始用プロパティ
	 * @throws HinemosUnknownException 
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.JobOperationProperty#getStartProperty(String, String, String, String, Locale)
	 */
	public ArrayList<String> getAvailableStartOperation(String sessionId, String jobunitId, String jobId, String facilityId) throws HinemosUnknownException {
		JobControllerLocal job = getJobControllerLocal();
		return job.getAvailableStartOperation(sessionId, jobunitId, jobId, facilityId, Locale.getDefault());
	}
	
	/**
	 * ジョブ操作停止用プロパティを返します。<BR>
	 * 
	 * @param sessionId セッションID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @param facilityId ファシリティID
	 * @param locale ロケール情報
	 * @return ジョブ操作停止用プロパティ
	 * @throws HinemosUnknownException 
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.JobOperationProperty#getStopProperty(String, String, String, String, Locale)
	 */
	public ArrayList<String> getAvailableStopOperation(String sessionId, String jobunitId,  String jobId, String facilityId) throws HinemosUnknownException {
		JobControllerLocal job = getJobControllerLocal();
		return job.getAvailableStopOperation(sessionId, jobunitId, jobId, facilityId, Locale.getDefault());
	}
	
	/**
	 * ジョブを実行します。<BR>
	 * 
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @param info ログ出力情報
	 * @param triggerInfo 実行契機情報
	 * @throws HinemosUnknownException 
	 * @throws JobInfoNotFoundException 
	 * @throws JobMasterNotFoundException 
	 * @see com.clustercontrol.jobmanagement.ejb.session.JobControllerBean#createJobInfo(String, String, NotifyRequestMessage, JobTriggerInfo}
	 * @see com.clustercontrol.jobmanagement.ejb.session.JobRunManagementBean#runJob(String, String)
	 */
	public void runJob(String jobunitId, String jobId, OutputBasicInfo info, JobTriggerInfo triggerInfo) 
			throws FacilityNotFoundException, HinemosUnknownException, JobInfoNotFoundException, JobMasterNotFoundException {
		JobControllerLocal job = getJobControllerLocal();
		job.runJob(jobunitId, jobId, info, triggerInfo);
	}
	
	/**
	 * ジョブをスケジュール実行します。<BR>
	 * Quartzからスケジュール実行時に呼び出されます。<BR>
	 *  
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @param calendarId カレンダID
	 * @param triggerInfo 実行契機情報
	 * @throws FacilityNotFoundException
	 * @throws HinemosUnknownException 
	 * @throws JobInfoNotFoundException 
	 * @throws JobMasterNotFoundException 
	 * @throws CalendarNotFountException 
	 * @see com.clustercontrol.calendar.ejb.session.CalendarControllerBean#isRun(String, Date)}
	 * @see com.clustercontrol.jobmanagement.ejb.session.JobControllerBean#runJob(String, String, JobTriggerInfo)
	 */
	public void scheduleRunJob(String jobunitId, String jobId, String calendarId, JobTriggerInfo triggerInfo)
			throws FacilityNotFoundException, HinemosUnknownException, JobMasterNotFoundException, JobInfoNotFoundException, CalendarNotFountException {
		JobControllerLocal job = getJobControllerLocal();
		job.scheduleRunJob(jobunitId, jobId, calendarId, triggerInfo);
	}
	
	/**
	 * ジョブの実行用情報を作成します。<BR>
	 * 
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @param info ログ出力情報
	 * @param triggerInfo 実行契機情報
	 * @return セッションID
	 * @throws FacilityNotFoundException
	 * @throws HinemosUnknownException 
	 * @throws JobInfoNotFoundException 
	 * @throws JobMasterNotFoundException 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobSessionLocalHome#makeSession(java.lang.String, com.clustercontrol.notify.message.NotifyRequestMessage)
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobSessionBean#ejbHomeMakeSession(String, NotifyRequestMessage)
	 */
	public String createJobInfo(String jobunitId, String jobId, OutputBasicInfo info, JobTriggerInfo triggerInfo)
			throws FacilityNotFoundException, HinemosUnknownException, JobMasterNotFoundException, JobInfoNotFoundException {
		JobControllerLocal job = getJobControllerLocal();
		return job.createJobInfo(jobunitId, jobId, info, triggerInfo);
	}

	/**
	 * ジョブ操作を行います。<BR>
	 * 
	 * @param property ジョブ操作用プロパティ
	 * @throws HinemosUnknownException 
	 * @throws JobInfoNotFoundException
	 * @see com.clustercontrol.jobmanagement.ejb.session.JobRunManagementBean#operationJob(Property)
	 */
	public void operationJob(NodeOperationInfo property) throws HinemosUnknownException, JobInfoNotFoundException {
		JobControllerLocal job = getJobControllerLocal();
		job.operationJob(property);
	}
	
	
	/**
	 * ジョブ履歴一覧情報を返します。<BR>
	 * 
	 * @param property 履歴フィルタ用プロパティ
	 * @param histories 表示履歴数
	 * @return ジョブ履歴一覧情報（Objectの2次元配列）
	 * @throws JobInfoNotFoundException
	 * @throws HinemosUnknownException 
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getHistoryList(Property, int)
	 */
	public JobHistoryList getHistoryList(JobHistoryFilter property, int histories) throws JobInfoNotFoundException, HinemosUnknownException {
		JobControllerLocal job = getJobControllerLocal();
		return job.getHistoryList(property, histories);
	}
	
	/**
	 * ジョブ詳細一覧情報を返します。<BR>
	 * TODO 未実装・・・。返り値をArrayList<JobDetailInfo>のsession beanを作成する必要あり。
	 * 
	 * @param sessionId セッションID
	 * @return ジョブ詳細一覧情報（Objectの2次元配列）
	 * @throws JobInfoNotFoundException
	 * @throws HinemosUnknownException 
	 * @throws JobMasterNotFoundException 
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getDetailList(String)
	 */
	/*
	public CommonTableTreeItem getDetailList(String sessionId) throws JobInfoNotFoundException, HinemosUnknownException, JobMasterNotFoundException {
		JobControllerLocal job = getJobControllerLocal();
		return job.getDetailList(sessionId);
	}
	*/
	
	/**
	 * ノード詳細一覧情報を返します。<BR>
	 * getDetailListが未実装のため、こちらも合わせて未実装とする。
	 * 
	 * @param sessionId セッションID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @param locale ロケール情報
	 * @return ノード詳細一覧情報（Objectの2次元配列）
	 * @throws JobInfoNotFoundException
	 * @throws HinemosUnknownException 
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getNodeDetailList(String, String, String, Locale)
	 */
	/*
	public ArrayList<JobNodeDetail> getNodeDetailList(String sessionId, String jobunitId, String jobId) throws JobInfoNotFoundException, HinemosUnknownException {
		JobControllerLocal job = getJobControllerLocal();
		return job.getNodeDetailList(sessionId, jobunitId, jobId, Locale.getDefault());
	}
	*/
	
	/**
	 * ファイル転送一覧情報を返します。<BR>
	 * getDetailListが未実装のため、こちらも合わせて未実装とする。
	 * 
	 * @param sessionId セッションID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @return ファイル転送一覧情報（Objectの2次元配列）
	 * @throws JobInfoNotFoundException
	 * @throws HinemosUnknownException 
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getForwardFileList(String, String)
	 */
	/*
	public ArrayList<JobForwardFile> getForwardFileList(String sessionId, String jobunitId, String jobId) throws JobInfoNotFoundException, HinemosUnknownException {
		JobControllerLocal job = getJobControllerLocal();
		return job.getForwardFileList(sessionId, jobunitId, jobId);
	}
	*/
	
	/**
	 * スケジュール情報を登録します。<BR>
	 * 
	 * @param info スケジュール情報
	 * @throws HinemosUnknownException 
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.ModifySchedule#addSchedule(JobSchedule, String)
	 */
	public void addSchedule(JobSchedule info) throws HinemosUnknownException {
		JobControllerLocal job = getJobControllerLocal();
		job.addSchedule(info);
	}
	
	/**
	 * スケジュール情報を変更します。<BR>
	 * 
	 * @param info スケジュール情報
	 * @throws HinemosUnknownException 
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.ModifySchedule#modifySchedule(JobSchedule, String)
	 */
	public void modifySchedule(JobSchedule info) throws HinemosUnknownException {
		JobControllerLocal job = getJobControllerLocal();
		job.modifySchedule(info);
	}
	
	/**
	 * スケジュール情報を削除します。<BR>
	 * 
	 * @param scheduleId スケジュールID
	 * @throws HinemosUnknownException 
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.ModifySchedule#deleteSchedule(String)
	 */
	public void deleteSchedule(String scheduleId) throws HinemosUnknownException {
		JobControllerLocal job = getJobControllerLocal();
		job.deleteSchedule(scheduleId);
	}
	
	/**
	 * スケジュール一覧情報を返します。<BR>
	 * 
	 * @return スケジュール一覧情報（Objectの2次元配列）
	 * @throws JobMasterNotFoundException
	 * @throws HinemosUnknownException 
	 * @see com.clustercontrol.jobmanagement.factory.SelectSchedule#getScheduleList()
	 */
	public ArrayList<JobSchedule> getScheduleList() throws JobMasterNotFoundException, HinemosUnknownException  {
		JobControllerLocal job = getJobControllerLocal();
		return job.getScheduleList();
	}
	
	
	/**
	 * カレンダIDリストを返します。<BR>
	 * 
	 * @return カレンダIDのリスト
	 * @throws HinemosUnknownException 
	 * @throws CalendarNotFountException 
	 * @throws  
	 * 
	 * @see com.clustercontrol.calendar.ejb.session.CalendarControllerBean#getCalendarIdList()
	 */
	public ArrayList<String> getCalendarIdList() throws HinemosUnknownException, CalendarNotFountException {
		JobControllerLocal job = getJobControllerLocal();
		return job.getCalendarIdList();
	}
	
	/**
	 * セッションジョブ情報を返します。<BR>
	 * 
	 * @param sessionId セッションID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @return ジョブツリー情報{@link com.clustercontrol.jobmanagement.bean.JobTreeItem}
	 * @throws HinemosUnknownException 
	 * @throws NotifyNotFoundException 
	 * @throws JobInfoNotFoundException 
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getSessionJobInfo(String, String, String)
	 */
	public JobTreeItem getSessionJobInfo(String sessionId, String jobunitId, String jobId) throws HinemosUnknownException, NotifyNotFoundException, JobInfoNotFoundException {
		JobControllerLocal job = getJobControllerLocal();
		return job.getSessionJobInfo(sessionId, jobunitId, jobId);
	}
	
	/**
	 * ファシリティIDが使用されているかチェックします。<BR>
	 * <P>
	 * 使用されていたら、UsedFacilityException がスローされる。<BR>
	 * 
	 * @param facilityId ファシリティ
	 * @throws UsedFacilityException
	 * @throws HinemosUnknownException 
	 * 
	 * @see com.clustercontrol.commons.ejb.session.CheckFacility#isUseFacilityId(java.lang.String)
	 * @see com.clustercontrol.bean.PluginConstant;
	 */
	/*
	public void isUseFacilityId(String facilityId) throws UsedFacilityException, HinemosUnknownException {
		JobControllerLocal job = getJobControllerLocal();
		job.isUseFacilityId(facilityId);
	}
	*/
}
