/*
 
Copyright (C) 2010 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.selfcheck.monitor;

import java.util.Date;
import java.util.List;

import javax.management.ObjectName;
import javax.naming.InitialContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.jms.server.destination.SubscriptionInfo;
import org.jboss.jmx.adaptor.rmi.RMIAdaptor;

import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.bean.OutputNotifyGroupInfo;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.util.Messages;

/**
 * SyslogForwardからの接続状態を確認する処理の実装クラス
 */
public class SyslogForwardMonitor extends SelfCheckMonitorBase {
	
	private static Log m_log = LogFactory.getLog( SyslogForwardMonitor.class );
	
	private static final String LOOKUP_NAME = "jmx/invoker/RMIAdaptor";
	private static final String OBJECT_NAME = "jboss.messaging.destination:name=clustercontrolRepositoryUpdate,service=Topic";
	private static final String OPERATION_NAME = "listAllSubscriptions";
	
	private String ipAddress = "";
	
	private static final String monitorIdPrefix = "SYS_EXT";
	private String monitorId = "";
	private String application = "SELFCHECK (SyslogForward)";
	
	
	/**
	 * コンストラクタ
	 * @param jvmMinHeapThreshold Java VMの利用可能なヒープ容量の最小値
	 */
	public SyslogForwardMonitor(String ipAddress) {
		this.ipAddress = ipAddress;
		
		this.monitorId = monitorIdPrefix + "_SyslogForward";
	}
	
	/**
	 * セルフチェック処理名
	 */
	@Override
	public String toString() {
		return "monitoring syslogforward connection";
	}
	
	/**
	 * 監視項目ID
	 */
	public String getMonitorId() {
		return monitorId;
	}
	
	/**
	 * SyslogForwardからの接続状態を確認する処理
	 * @return 通知情報（アプリケーション名は未格納）
	 */
	@SuppressWarnings("unchecked")
	public OutputNotifyGroupInfo execute() {
		/** ローカル変数 */
		InitialContext initCtx = null;
		RMIAdaptor rmiAdapter = null;
		
		List<SubscriptionInfo> subscriptions = null;
		String selector = null;
		String checkString = null;
		
		OutputNotifyGroupInfo notifyInfo = null;
		
		boolean avaiable = false;
		
		/** メイン処理 */
		if (m_log.isDebugEnabled()) m_log.debug("monitoring syslogforward's connection.");
		
		try {
			// JMSのオブジェクトをJNDI経由で取得し、属性値の値を取得できるかを確認する
			initCtx = new InitialContext();
			rmiAdapter = (RMIAdaptor)initCtx.lookup(LOOKUP_NAME);
			
			subscriptions = (List<SubscriptionInfo>)rmiAdapter.invoke(new ObjectName(OBJECT_NAME), OPERATION_NAME, null, null);
			
			checkString = "MultiId='" + ipAddress + "'";
			
			for (SubscriptionInfo subscription : subscriptions) {
				selector = subscription.getSelector();
				if (selector != null && selector.indexOf(checkString) != -1) {
					avaiable = true;
				}
			}
		} catch (Exception e) {
			if (m_log.isWarnEnabled()) m_log.warn("jmx access failure.", e);
			avaiable = false;
		}
		
		if (avaiable) {
			if (m_log.isDebugEnabled()) m_log.debug("syslogforward's connection is avaiable.");
			
			// set result, but do not notify (priority == info)
			notifyInfo = new OutputNotifyGroupInfo();
			notifyInfo.setPluginId(PLUGIN_ID);
			notifyInfo.setMonitorId(monitorId);
			notifyInfo.setPriority(PriorityConstant.TYPE_INFO);
		} else {
			if (m_log.isWarnEnabled()) m_log.warn("syslogforward's connection is not avaiable.");
			notifyInfo = new OutputNotifyGroupInfo();
			
			String[] msgAttr1 = { ipAddress };
			String[] msgAttr2 = { ipAddress };
			
			notifyInfo.setPluginId(PLUGIN_ID);
			notifyInfo.setMonitorId(monitorId);
			notifyInfo.setApplication(application);
			notifyInfo.setNotifyGroupId(HinemosModuleConstant.SYSYTEM_SELFCHECK + "_" + monitorId);
			notifyInfo.setMessageId("001");
			notifyInfo.setMessage(Messages.getString("message.selfcheck.notify.syslogforward.failure.msg", msgAttr1));
			notifyInfo.setMessageOrg(Messages.getString("message.selfcheck.notify.syslogforward.failure.origmsg", msgAttr2));
			notifyInfo.setFacilityId(FACILITY_ID);
			notifyInfo.setScopeText(FACILITY_TEXT);
			notifyInfo.setPriority(PriorityConstant.TYPE_WARNING);
			notifyInfo.setGenerationDate(new Date());
		}
		
		return notifyInfo;
	}
	
}
