/*
 
Copyright (C) 2010 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.selfcheck.monitor;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Date;

import javax.naming.InitialContext;
import javax.sql.DataSource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.bean.OutputNotifyGroupInfo;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.util.Messages;

/**
 * ログテーブルの蓄積量を確認する処理の実装クラス
 */
public class DBSizeMonitor extends SelfCheckMonitorBase {
	
	private static Log m_log = LogFactory.getLog( DBSizeMonitor.class );
	
	private String dataSourceName = "";
	private String jndiName = "";
	private final String jndiPrefix = "java:/";
	
	private String tableName = "";
	private String tableDesc = "";
	private long threshold = 0;
	private int thresholdType = THRESHOLD_MBYTE;
	
	private static final String monitorIdPrefix = "SYS_DBSIZE";
	private String monitorId = "";
	private String application = "SELFCHECK (DataSource)";
	
	public static final int THRESHOLD_MBYTE = 1;
	public static final int THRESHOLD_COUNT = 2;
	
	/**
	 * コンストラクタ
	 * @param dataSourceName データソース名（"HinemosDS"など）
	 */
	public DBSizeMonitor(String dataSourceName, String tableName, String tableDesc, long threshold, int thresholdType) {
		this.dataSourceName = dataSourceName;
		this.jndiName = jndiPrefix + dataSourceName;
		
		this.tableName = tableName;
		this.tableDesc = tableDesc;
		this.threshold = threshold;
		this.thresholdType = thresholdType;
		
		this.monitorId = monitorIdPrefix + "_" + tableName + "_" + thresholdType;
	}
	
	/**
	 * セルフチェック処理名
	 */
	@Override
	public String toString() {
		return "monitoring log table - " + tableName + "_" + thresholdType;
	}
	
	/**
	 * 監視項目ID
	 */
	public String getMonitorId() {
		return monitorId;
	}
	
	/**
	 * データソースへの疎通確認
	 * @return 通知情報（アプリケーション名は未格納）
	 */
	public OutputNotifyGroupInfo execute() {
		/** ローカル変数 */
		InitialContext initCtx = null;
		DataSource dataSource = null;
		Connection conn = null;
		
		long size = -1;
		long thresholdOrig = threshold;
		long physicalSize = -1;
		double physicalSizeMByte = -1.0;
		long count = -1;
		
		OutputNotifyGroupInfo notifyInfo = null;
		
		/** メイン処理 */
		if (m_log.isDebugEnabled()) 
			m_log.debug("monitoring log table. (dataSource = " + dataSourceName + ", jndi=" + jndiName
					+ ", tableName=" + tableName + ", threshold=" + threshold + " [" + getThresholdUnit(thresholdType) + "])");
		try {
			// データソースのオブジェクトをJNDI経由で取得し、取得したコネクションが正しく動作するかを確認する
			initCtx = new InitialContext();
			dataSource = (DataSource)initCtx.lookup(jndiName);
			
			conn = dataSource.getConnection();
			
			// 判定対象値を取得する
			switch (thresholdType) {
			case THRESHOLD_MBYTE : 
				// convert MByte to byte
				threshold = threshold * 1024 * 1024;
				size = getTableSize(conn, tableName);
				break;
			case THRESHOLD_COUNT :
				size = getTableCount(conn, tableName);
				break;
			default :
				m_log.warn("monitoring type is invalid. (type=" + thresholdType + ")");
			}
			
			if (size == -1) {
				if (m_log.isInfoEnabled()) {
					m_log.info("skipped monitoring log table. (dataSource = " + dataSourceName + ", jndi=" + jndiName
							+ ", tableName=" + tableName + ", threshold=" + thresholdOrig + " [" + getThresholdUnit(thresholdType) + "])");
				}
			} else if (size <= threshold) {
				if (m_log.isDebugEnabled()) {
					m_log.debug("log table's size is low. (dataSource = " + dataSourceName + ", jndi=" + jndiName
							+ ", tableName=" + tableName + ", size=" + size + ", threshold=" + thresholdOrig + " [" + getThresholdUnit(thresholdType) + "])");
				}
				
				// set result, but do not notify (priority == info)
				notifyInfo = new OutputNotifyGroupInfo();
				notifyInfo.setPluginId(PLUGIN_ID);
				notifyInfo.setMonitorId(monitorId);
				notifyInfo.setPriority(PriorityConstant.TYPE_INFO);
			} else {
				m_log.warn("log table's size is too high. (dataSource = " + dataSourceName + ", jndi=" + jndiName
						+ ", tableName=" + tableName + ", size=" + size + ", threshold=" + thresholdOrig + " [" + getThresholdUnit(thresholdType) + "])");
				
				switch (thresholdType) {
				case THRESHOLD_MBYTE : 
					physicalSize = size;
					count = getTableCount(conn, tableName);
					break;
				case THRESHOLD_COUNT :
					physicalSize = getTableSize(conn, tableName);
					count = size;
					break;
				default :
					m_log.warn("monitoring type is invalid. (type=" + thresholdType + ")");
				}
				
				physicalSizeMByte = (double)physicalSize / 1024.0 / 1024.0;
				
				notifyInfo = new OutputNotifyGroupInfo();
				
				String[] msgAttr1 = { tableDesc, String.format("%.2f", physicalSizeMByte), getThresholdUnit(thresholdType), new Long(count).toString(), new Long(thresholdOrig).toString() };
				String[] msgAttr2 = { tableName, String.format("%.2f", physicalSizeMByte), getThresholdUnit(thresholdType), new Long(count).toString(), new Long(thresholdOrig).toString() };
				
				notifyInfo.setPluginId(PLUGIN_ID);
				notifyInfo.setMonitorId(monitorId);
				notifyInfo.setApplication(application);
				notifyInfo.setNotifyGroupId(HinemosModuleConstant.SYSYTEM_SELFCHECK + "_" + monitorId);
				notifyInfo.setMessageId("001");
				notifyInfo.setMessage(Messages.getString("message.selfcheck.notify.dbsize.failure.msg", msgAttr1));
				notifyInfo.setMessageOrg(Messages.getString("message.selfcheck.notify.dbsize.failure.origmsg", msgAttr2));
				notifyInfo.setFacilityId(FACILITY_ID);
				notifyInfo.setScopeText(FACILITY_TEXT);
				notifyInfo.setPriority(PriorityConstant.TYPE_WARNING);
				notifyInfo.setGenerationDate(new Date());
			}
		} catch (Exception e) {
			m_log.warn("monitoring log table failure. (dataSource = " + dataSourceName + ", jndi=" + jndiName
					+ ", tableName=" + tableName + ", threshold=" + threshold + " [" + getThresholdUnit(thresholdType) + "])");
		} finally {
			try {
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e) {
				m_log.warn("closing connection failure. (dataSource = " + dataSourceName + ", jndi=" + jndiName + ")", e);
			}
		}
		
		return notifyInfo;
	}
	
	/**
	 * 特定のテーブルの物理サイズを返すメソッド
	 * @param conn 問い合わせに利用するコネクション
	 * @param tableName 対象とするテーブル名
	 * @return 物理サイズ
	 */
	private static long getTableSize(Connection conn, String tableName) {
		// ローカル変数
		Statement stmt = null;
		ResultSet res = null;
		
		String query = "SELECT pg_total_relation_size('" + tableName + "') as size";
		long physicalSize = -1;
		
		// メイン処理
		try {
			stmt = conn.createStatement();
			res = stmt.executeQuery(query);
			if (res.next()) {
				physicalSize = res.getLong("size");
			}
		} catch (Exception e) {
			m_log.warn("database query execution failure. (" + query + ")", e);
		} finally {
			try {
				if (stmt != null) {
					stmt.close();
				}
			} catch (Exception e) {
				m_log.warn("closing statement failure.", e);
			}
		}
		
		return physicalSize;
	}
	
	/**
	 * 特定のテーブルのレコード数を返すメソッド
	 * @param conn 問い合わせに利用するコネクション
	 * @param tableName 対象とするテーブル名
	 * @return レコード数
	 */
	private static long getTableCount(Connection conn, String tableName) {
		// ローカル変数
		Statement stmt = null;
		ResultSet res = null;
		
		String query = "SELECT count(*) as count FROM " + tableName;
		long count = -1;
		
		// メイン処理
		try {
			stmt = conn.createStatement();
			res = stmt.executeQuery(query);
			if (res.next()) {
				count = res.getLong("count");
			}
		} catch (Exception e) {
			m_log.warn("database query execution failure. (" + query + ")", e);
		} finally {
			try {
				if (stmt != null) {
					stmt.close();
				}
			} catch (Exception e) {
				m_log.warn("closing statement failure.", e);
			}
		}
		
		return count;
	}
	
	public static String getThresholdUnit(int type) {
		// ローカル変数
		String unit = "";
		
		// メイン処理
		switch (type) {
		case THRESHOLD_MBYTE :
			unit = "mbyte";
			break;
		case THRESHOLD_COUNT :
			unit = "rows";
			break;
		default :
		}
		
		return unit;
	}
	
}
