/*
 
Copyright (C) 2010 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.selfcheck;

import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.EventConfirmConstant;
import com.clustercontrol.bean.OutputNotifyGroupInfo;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.notify.util.OutputEvent;
import com.clustercontrol.notify.util.SendMail;
import com.clustercontrol.notify.util.SendSyslog;
import com.clustercontrol.selfcheck.monitor.SelfCheckMonitor;

/**
 * セルフチェック処理実行処理の実装クラス
 */
public class SelfCheckTask implements Runnable {

	private static Log m_log = LogFactory.getLog(SelfCheckTask.class);

	private final SelfCheckMonitor monitor;
	
	private static volatile ConcurrentHashMap<String, Integer> invalidCounter = new ConcurrentHashMap<String, Integer>();

	/**
	 * コンストラクタ
	 * @param config
	 */
	public SelfCheckTask(SelfCheckMonitor monitor){
		this.monitor = monitor;
	}

	/**
	 * セルフチェック処理の実行
	 */
	public void run() {
		/** ローカル変数 */
		OutputNotifyGroupInfo notifyInfo = null;
		String mailSubject = null;
		String mailBody = null;
		SendMail sendmail = null;

		/** メイン処理 */
		if (m_log.isDebugEnabled()) m_log.debug("executing self-check. (" + toString() + ")");
		
		// counterの初期化
		if (! invalidCounter.containsKey(monitor.getMonitorId())) {
			if (m_log.isInfoEnabled()) m_log.info("initializing monitoring counter. (monitorId=" + monitor.getMonitorId() + ")");
			invalidCounter.putIfAbsent(monitor.getMonitorId(), 0);
		}
		
		// 監視処理の実行
		notifyInfo = monitor.execute();
		
		// 通知処理
		if (notifyInfo == null) {
			if (m_log.isDebugEnabled()) m_log.debug("skipped monitoring. (monitorId=" + monitor.getMonitorId() + ")");
		} else {
			if (notifyInfo.getPriority() == PriorityConstant.TYPE_INFO) {
				// 出力判定用カウンターをリセット
				if (m_log.isDebugEnabled()) m_log.debug("resetting monitoring counter. (monitorId=" + monitor.getMonitorId() + ")");
				invalidCounter.put(monitor.getMonitorId(), 0);
			} else {
				// 出力判定用カウンタをインクリメント
				if (invalidCounter.get(monitor.getMonitorId()) < Integer.MAX_VALUE) {
					if (m_log.isDebugEnabled()) 
						m_log.debug("incrementing monitoring counter. (monitorId=" + monitor.getMonitorId() + ", current counter=" + invalidCounter.get(monitor.getMonitorId()) + ")");
					invalidCounter.replace(monitor.getMonitorId(), invalidCounter.get(monitor.getMonitorId()) + 1);
				}
				
				if (invalidCounter.get(monitor.getMonitorId()) >= SelfCheckConfig.getAlertThreshold()) {
					if (m_log.isDebugEnabled()) m_log.debug("output result of selfcheck monitoring. (" + monitor.toString() + ")");
					
					// 閾値を超えて異常が通知された場合、外部へ通知する
					if (SelfCheckConfig.getEvent()) {
						// イベント通知（ベストエフォートのため、DBに疎通できない場合は失敗）
						if (m_log.isDebugEnabled()) m_log.debug("creating event of selfcheck monitoring. (" + monitor.toString() + ")");
						new OutputEvent().insertEventLog(notifyInfo, EventConfirmConstant.TYPE_UNCONFIRMED);
					}
					
					if (SelfCheckConfig.getSyslog()) {
						// syslog通知
						for (String ipAddress : SelfCheckConfig.getSyslogHost()) {
							if (m_log.isDebugEnabled()) m_log.debug("sending syslog of selfcheck monitoring. (" + monitor.toString() + ", host=" + ipAddress + ")");
							new SendSyslog().sendAfterConvertHostname(ipAddress, SelfCheckConfig.getSyslogPort(), SelfCheckConfig.getSyslogFacility(), SelfCheckConfig.getSyslogSeverity(), notifyInfo.getFacilityId(), notifyInfo.getMessageOrg());
						}
					}
					
					if (SelfCheckConfig.getMail()) {
						// メール通知（デフォルトテンプレート）
						try {
							if (m_log.isDebugEnabled()) m_log.debug("sending mail of selfcheck monitoring. (" + monitor.toString() + ", mail_address=" + SelfCheckConfig.getMailAddress() + ")");
							sendmail = new SendMail();
							mailSubject = sendmail.getSubject(notifyInfo, null);
							mailBody = sendmail.getContent(notifyInfo, null);
							sendmail.sendMail(SelfCheckConfig.getMailAddress(), mailSubject, mailBody);
						} catch (Exception e) {
							m_log.warn("failed sending e-mail of selfcheck monitoring. (" + monitor.toString() + ")", e);
						}
					}
				}
			}
		}
		
		if (m_log.isDebugEnabled()) m_log.debug("selfcheck scheduler task is executed. (" + toString() + ")");
	}

	/**
	 * セルフチェック処理名の取得
	 */
	@Override
	public String toString() {
		/** メイン処理 */
		return monitor.toString();
	}
}
