/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.process.ejb.session;

import java.rmi.RemoteException;
import java.security.Principal;
import java.util.ArrayList;
import javax.ejb.EJBException;
import javax.ejb.SessionBean;
import javax.ejb.SessionContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.HinemosUnknownException;
import com.clustercontrol.MonitorDuplicateException;
import com.clustercontrol.MonitorNotFoundException;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.process.factory.AddMonitorProcess;
import com.clustercontrol.process.factory.DeleteMonitorProcess;
import com.clustercontrol.process.factory.ModifyMonitorProcess;
import com.clustercontrol.process.factory.SelectMonitorProcess;

/**
 * プロセス監視を制御するSesison Bean<BR>
 * 
 * @ejb.bean name="MonitorProcessController"
 *           jndi-name="MonitorProcessController"
 *           type="Stateless"
 *           transaction-type="Container"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MonitorProcessInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MonitorProcessMethodMstLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MonitorProcessPollingMstLocal"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Home"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 * 
 */
public abstract class MonitorProcessControllerBean implements SessionBean {
	
	protected static Log m_log = LogFactory.getLog( MonitorProcessControllerBean.class );
	
	@SuppressWarnings("unused")
	private SessionContext m_context;
	
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}
	
	/**
	 * プロセス監視情報をマネージャに登録します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="ProcessMonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param info 通知情報
	 * @return 登録に成功した場合、true
	 * @throws MonitorDuplicateException
	 * @throws HinemosUnknownException 
	 * @see com.clustercontrol.monitor.run.bean.MonitorInfo
	 */
	public boolean addProcess(MonitorInfo info) throws MonitorDuplicateException, HinemosUnknownException{
		
		Principal pri = m_context.getCallerPrincipal();

		// プロセス監視情報を登録
		AddMonitorProcess process = new AddMonitorProcess();
		boolean flag;
		try {
			flag = process.add(info, pri.getName());
		} catch (MonitorDuplicateException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("addProcess " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return flag;
	}
	
	/**
	 * マネージャのプロセス監視情報を更新します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="ProcessMonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param info 通知情報
	 * @return 変更に成功した場合、true
	 * @throws MonitorNotFoundException 
	 * @throws HinemosUnknownException
	 * @see com.clustercontrol.monitor.run.bean.MonitorInfo
	 */
	public boolean modifyProcess(MonitorInfo info) throws MonitorNotFoundException, HinemosUnknownException {
		
		Principal pri = m_context.getCallerPrincipal();

		// プロセス監視情報を更新
		ModifyMonitorProcess process = new ModifyMonitorProcess();
		boolean flag;
		try {
			flag = process.modify(info, pri.getName());
		} catch (MonitorNotFoundException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("modifyProcess " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return flag;
	}

	/**
	 * プロセス監視情報マネージャから削除します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="ProcessMonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param monitorId 監視項目ID
	 * @return 削除に成功した場合、true
	 * @throws MonitorNotFoundException
	 * @throws HinemosUnknownException 
	 */
	public boolean deleteProcess(String monitorId) throws MonitorNotFoundException, HinemosUnknownException {

		// プロセス監視情報を削除
		DeleteMonitorProcess process = new DeleteMonitorProcess();
		boolean flag;
		try {
			flag = process.delete(HinemosModuleConstant.MONITOR_PROCESS, monitorId);
		} catch (MonitorNotFoundException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("deleteProcess " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return flag;
	}
	
	/**
	 * プロセス監視情報を取得します。<BR>
	 * 
	 * 引数に指定されたmonitorIdに対応するプロセス監視設定情報を取得します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="ProcessMonitorRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param monitorId 状態監視ID
	 * @return プロセス監視情報
	 * @throws MonitorNotFoundException 
	 * @throws HinemosUnknownException
	 */
	public MonitorInfo getProcess(String monitorId) throws MonitorNotFoundException, HinemosUnknownException{

		// プロセス監視情報を取得
		SelectMonitorProcess process = new SelectMonitorProcess();
		MonitorInfo info = null;
		try {
			info = process.getMonitor(HinemosModuleConstant.MONITOR_PROCESS, monitorId);
		} catch (MonitorNotFoundException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getProcess " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return info;
	}

	/**
	 * プロセス監視一覧リストを取得します。<BR>
	 * 
	 * 戻り値はArrayListのArrayListであり内部のArrayListは以下のように
	 * 値が格納される。
	 * 
	 * 
	 * 
	 * <Pre>
	 * 
	 * ArrayList info = new ArrayList();
	 *  info.add(monitor.getMonitorId());
	 *  info.add(monitor.getDescription());
	 *  info.add(facilityPath);
	 *  info.add(MonitorTypeConstant.typeToString(monitor.getMonitorType().intValue()));
	 *  info.add(monitor.getCalendarId());
	 *  info.add(RunIntervalConstant.typeToString(monitor.getRunInterval().intValue()));
	 *  info.add(schedule.getValid(monitor.getMonitorTypeId(), monitor.getMonitorId()));
	 *  info.add(monitor.getRegUser());
	 *  info.add(monitor.getRegDate() == null ? null:new Date(monitor.getRegDate().getTime()));
	 *  info.add(monitor.getUpdateUser());
	 *  info.add(monitor.getUpdateDate() == null ? null:new Date(monitor.getUpdateDate().getTime()));
	 *
	 *  list.add(info);
	 * </Pre>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="ProcessMonitorRead"
     *     method-intf="Remote"
     *     
     * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return Objectの2次元配列
	 * @throws MonitorNotFoundException 
	 * @throws HinemosUnknownException
	 */
	public ArrayList<MonitorInfo> getProcessList() throws MonitorNotFoundException, HinemosUnknownException{

		// プロセス監視一覧を取得
		SelectMonitorProcess process = new SelectMonitorProcess();
		ArrayList<MonitorInfo> list = null;
		try {
			list = process.getMonitorList(HinemosModuleConstant.MONITOR_PROCESS);
		} catch (MonitorNotFoundException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getProcessList " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return list;
	}
	
}
