/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.port.ejb.session;

import java.rmi.RemoteException;
import java.security.Principal;
import java.util.ArrayList;
import javax.ejb.EJBException;
import javax.ejb.SessionBean;
import javax.ejb.SessionContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.HinemosUnknownException;
import com.clustercontrol.MonitorDuplicateException;
import com.clustercontrol.MonitorNotFoundException;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.port.factory.AddMonitorPort;
import com.clustercontrol.port.factory.DeleteMonitorPort;
import com.clustercontrol.port.factory.ModifyMonitorPort;
import com.clustercontrol.port.factory.SelectMonitorPort;

/**
 * @ejb.bean name="MonitorPortController"
 *           jndi-name="MonitorPortController"
 *           type="Stateless"
 *           transaction-type="Container"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MonitorPortInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MonitorProtocolMasterLocal"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Home"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 * 
 */
public abstract class MonitorPortControllerBean implements SessionBean {
	
	protected static Log m_log = LogFactory.getLog( MonitorPortControllerBean.class );
	
	@SuppressWarnings("unused")
	private SessionContext m_context;
	
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}
	
	/**
	 * port監視情報を登録します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="PortMonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param info 通知情報
	 * @return 登録に成功した場合、true
	 * @throws MonitorDuplicateException 
	 * @throws HinemosUnknownException
	 */
	public boolean addPort(MonitorInfo info) throws MonitorDuplicateException, HinemosUnknownException {
		
		Principal pri = m_context.getCallerPrincipal();

		// port監視情報を登録
		AddMonitorPort port = new AddMonitorPort();
		boolean flag;
		try {
			flag = port.add(info, pri.getName());
		} catch (MonitorDuplicateException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("addPort " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return flag;
	}
	
	/**
	 * port監視情報を更新します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="PortMonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param info 通知情報
	 * @return 変更に成功した場合、true
	 * @throws MonitorNotFoundException 
	 * @throws HinemosUnknownException
	 */
	public boolean modifyPort(MonitorInfo info) throws MonitorNotFoundException, HinemosUnknownException {
		
		Principal pri = m_context.getCallerPrincipal();

		// port監視情報を更新
		ModifyMonitorPort port = new ModifyMonitorPort();
		boolean flag;
		try {
			flag = port.modify(info, pri.getName());
		} catch (MonitorNotFoundException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("modifyPort " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return flag;
	}
	
	/**
	 * port監視情報を削除します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="PortMonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param monitorId 監視項目ID
	 * @return 削除に成功した場合、true
	 * @throws MonitorNotFoundException
	 * @throws HinemosUnknownException 
	 */
	public boolean deletePort(String monitorId) throws MonitorNotFoundException, HinemosUnknownException {

		// port監視情報を削除
		DeleteMonitorPort port = new DeleteMonitorPort();
		boolean flag;
		try {
			flag = port.delete(HinemosModuleConstant.MONITOR_PORT, monitorId);
		} catch (MonitorNotFoundException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("deletePort " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return flag;
	}
	
	/**
	 * port監視情報を返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="PortMonitorRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param monitorId 監視項目ID
	 * @return port監視情報
	 * @throws MonitorNotFoundException
	 * @throws HinemosUnknownException 
	 */
	public MonitorInfo getPort(String monitorId) throws MonitorNotFoundException, HinemosUnknownException {

		// port監視情報を取得
		SelectMonitorPort port = new SelectMonitorPort();
		MonitorInfo info = null;
		try {
			info = port.getMonitor(HinemosModuleConstant.MONITOR_PORT, monitorId);
		} catch (MonitorNotFoundException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getPort " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}

		return info;
	}

	/**
	 * port監視一覧リストを返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="PortMonitorRead"
     *     method-intf="Remote"
     *     
     * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return Objectの2次元配列
	 * @throws MonitorNotFoundException 
	 * @throws HinemosUnknownException
	 */
	public ArrayList<MonitorInfo> getPortList() throws MonitorNotFoundException, HinemosUnknownException{

		// port監視一覧を取得
		SelectMonitorPort port = new SelectMonitorPort();
		ArrayList<MonitorInfo> list = null;
		try {
			list = port.getMonitorList(HinemosModuleConstant.MONITOR_PORT);
		} catch (MonitorNotFoundException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getPortList " + e.getMessage(), e);
			throw new HinemosUnknownException (e.getMessage(), e);
		}

		return list;
	}
}
