/*

 Copyright (C) 2006 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 */

package com.clustercontrol.performance.monitor.ejb.session;

import java.rmi.RemoteException;
import java.security.Principal;
import java.util.ArrayList;
import javax.ejb.EJBException;
import javax.ejb.SessionBean;
import javax.ejb.SessionContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.HinemosUnknownException;
import com.clustercontrol.MonitorDuplicateException;
import com.clustercontrol.MonitorNotFoundException;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.performance.monitor.factory.AddMonitorPerformance;
import com.clustercontrol.performance.monitor.factory.DeleteMonitorPerformance;
import com.clustercontrol.performance.monitor.factory.ModifyMonitorPerformance;
import com.clustercontrol.performance.monitor.factory.SelectMonitorPerformance;

/**
 * リソース監視を制御するSession Bean <BR>
 * @ejb.bean name="MonitorPerfController"
 *           jndi-name="MonitorPerfController"
 *           type="Stateless"
 *           transaction-type="Container"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MonitorPerfInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=SnmpValueTypeMstLocal"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Home"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"   
 *     
 */
public abstract class MonitorPerfControllerBean implements SessionBean {
	
	protected static Log m_log = LogFactory.getLog( MonitorPerfControllerBean.class );
	
	@SuppressWarnings("unused")
	private SessionContext m_context;
	
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}
	
	/**
	 * リソース監視情報を登録します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="PerformanceWrite"
     *     method-intf="Remote"
	 * 
	 * @param info 通知情報
	 * @return 登録に成功した場合、true
	 * @throws MonitorDuplicateException
	 * @throws HinemosUnknownException 
	 */
	public boolean addPerformance(MonitorInfo info) throws MonitorDuplicateException, HinemosUnknownException {
		Principal pri = m_context.getCallerPrincipal();

		// 閾値監視情報を登録
		AddMonitorPerformance addPerf = new AddMonitorPerformance();
		boolean flag;
		try {
			flag = addPerf.add(info, pri.getName());
		} catch (MonitorDuplicateException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("addPerformance " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return flag;
	}
	
	/**
	 * リソース監視情報を更新します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="PerformanceWrite"
     *     method-intf="Remote"
	 * 
	 * @param info 通知情報
	 * @return 変更に成功した場合、true
	 * @throws MonitorNotFoundException 
	 * @throws HinemosUnknownException
	 */
	public boolean modifyPerformance(MonitorInfo info) throws MonitorNotFoundException, HinemosUnknownException {
		Principal pri = m_context.getCallerPrincipal();
		
		// リソース監視情報を更新
		ModifyMonitorPerformance modifyPerf = new ModifyMonitorPerformance();
		boolean flag;
		try {
			flag = modifyPerf.modify(info, pri.getName());
		} catch (MonitorNotFoundException e) {
			throw e;
		} catch (Exception e) {
			m_log.error(e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(),e);
		}
		return flag;
	}

	/**
	 * リソース監視情報を削除します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="PerformanceWrite"
     *     method-intf="Remote"
	 * 
	 * @param monitorId 監視項目ID
	 * @return 削除に成功した場合、true
	 * @throws MonitorNotFoundException 
	 * @throws HinemosUnknownException 
	 */
	public boolean deletePerformance(String monitorId) throws MonitorNotFoundException, HinemosUnknownException {

		// リソース監視情報を削除
		DeleteMonitorPerformance delPerf = new DeleteMonitorPerformance();
		boolean flag;
		try {
			flag = delPerf.delete(HinemosModuleConstant.MONITOR_PERFORMANCE, monitorId);
		} catch (MonitorNotFoundException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("deletePerformance " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(),e);
		}
		return flag;
	}
	
	/**
	 * リソース監視情報を返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="PerformanceRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param monitorId 状態監視ID
	 * @return リソース監視情報
	 * @throws MonitorNotFoundException 
	 * @throws HinemosUnknownException
	 */
	public MonitorInfo getPerformance(String monitorId) throws MonitorNotFoundException, HinemosUnknownException {

		// リソース監視情報を取得
		SelectMonitorPerformance selectPerf = new SelectMonitorPerformance();
		MonitorInfo info = null;
		try {
			info = selectPerf.getMonitor(HinemosModuleConstant.MONITOR_PERFORMANCE, monitorId);
		} catch (MonitorNotFoundException e) {
			throw e;
		} catch (Exception e) {
			m_log.error(e.getMessage(), e);
			throw new HinemosUnknownException("getPerformance " + e.getMessage(), e); 
		}

		return info;
	}

	/**
	 * リソース監視一覧リストを返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="PerformanceRead"
     *     method-intf="Remote"
     *     
     * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return Objectの2次元配列
	 * @throws MonitorNotFoundException
	 * @throws HinemosUnknownException
	 */
	public ArrayList<MonitorInfo> getPerformanceList() throws MonitorNotFoundException, HinemosUnknownException{

		// リソース監視一覧を取得
		SelectMonitorPerformance selectPerf = new SelectMonitorPerformance();
		ArrayList<MonitorInfo> list;
		try {
			list = selectPerf.getMonitorList(HinemosModuleConstant.MONITOR_PERFORMANCE);
		} catch (MonitorNotFoundException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getPerformanceList " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}

		return list;
	}
}
