/*
                                                                                                                                                                 
Copyright (C) 2006 NTT DATA Corporation
                                                                                                                                                                 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
                                                                                                                                                                 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
                                                                                                                                                                 
*/

package com.clustercontrol.performance.ejb.session;

import java.rmi.RemoteException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.SessionBean;
import javax.ejb.SessionContext;
import javax.ejb.TransactionRolledbackLocalException;
import javax.naming.NamingException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.quartz.CronTrigger;
import org.quartz.JobDetail;
import org.quartz.SchedulerException;
import org.quartz.jobs.ee.ejb.EJBInvokerJob;


import com.clustercontrol.performance.bean.CollectorRunStatus;
import com.clustercontrol.performance.bean.QuartzConstant;
import com.clustercontrol.performance.util.JNDIConnectionManager;
import com.clustercontrol.performance.ejb.bmp.CollectorLocal;
import com.clustercontrol.performance.ejb.bmp.CollectorLocalHome;
import com.clustercontrol.performance.ejb.bmp.CollectorPK;
import com.clustercontrol.quartzmanager.util.QuartzUtil;
import com.clustercontrol.quartzmanager.ejb.session.QuartzManager;
import com.clustercontrol.util.apllog.AplLogger;



/**
 *　無期限収集のチェックを行うQuartzスケジュールを登録するクラス
 * 通常はマネージャDBのイニシャライズ時にのみ使用します。
 * 
 * @ejb.bean name="CollectorRunController"
 *           display-name="Name for CollectorRunController"
 *           description="Description for CollectorRunController"
 *           jndi-name="CollectorRunController"
 *           type="Stateless"
 *           view-type="remote"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorStatusLocalEJB"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorCalcMethodMstLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorCategoryCollectMstLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorCategoryMstLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorDeviceInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorItemCalcMethodMstLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorItemCodeMstLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorPollingMstLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorScopeRelationLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=CollectorScopeSnapLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=DeviceTypeMstLocal"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Home"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 * 
 */
public class CollectorRunControllerBean implements SessionBean {
	
	private static final long serialVersionUID = 1L;
	protected static Log m_log = LogFactory.getLog( CollectorRunControllerBean.class );
	
	
	/**
	 * Quartzからのコールバックメソッド
	 * 
	 * @ejb.interface-method
	 * @generated
	 * 
	 * 	@ejb.permission
	 *     unchecked="true"
	 *     method-intf="Remote"
	 *     
	 * @ejb.transaction
	 *     type="NotSupported"
	 *     
	 * @throws FinderException
	 * @throws NamingException
	 * @since 2.0.0
	 */
	public void managePresavePeriod() throws FinderException, NamingException {
		m_log.debug("managePresavePeriod() :  start");
		
		//無期限収集のcollectorIdリスト取得
		ArrayList<CollectorPK> list = getPresaveCollectorList();
		
		//collectorId毎に定期削除処理を実行する
		Iterator<CollectorPK> iterator = list.iterator();
		while (iterator.hasNext()) {
			CollectorPK pk = iterator.next();
			m_log.debug("managePresavePeriod() :  do collectorId = " + pk.getCollectorID());
			_managePresavePeriod(pk);
		}
		
		m_log.debug("managePresavePeriod() :  end");
	}

	/**
	 * 無期限収集のcollectorIdリスト取得
	 * 
	 * @ejb.interface-method
	 * @generated
	 * 
	 * 	@ejb.permission
	 *     unchecked="true"
	 *     method-intf="Local"
	 *     
	 * @ejb.transaction
	 *     type="RequiresNew"
	 *     
	 * @throws FinderException
	 * @throws NamingException
	 * @since 2.0.0
	 */
	public ArrayList<CollectorPK> getPresaveCollectorList() throws FinderException, NamingException {
		m_log.debug("getPresaveCollectorList() :  start");
		
		ArrayList<CollectorPK> list = new ArrayList<CollectorPK>();
		
		//get Hame Interface 
		CollectorLocalHome home = JNDIConnectionManager.getCollectorLocalHome();
		Collection myBeans;
		
		try {
			myBeans = home.findByPresave();	
			Iterator it = myBeans.iterator();
			
			//while Loop
			while(it.hasNext()){
				//無期限収集それぞれにたいして、主キーを取得します。			
				CollectorLocal bean = (CollectorLocal)it.next();
				m_log.debug("getPresaveCollectorList() :  add collectorId = " + bean.getCollectorID());
				CollectorPK pk = new CollectorPK(bean.getCollectorID());
				list.add(pk);
			}
		} catch (FinderException e) {
			e.printStackTrace();
		}
		
		m_log.debug("getPresaveCollectorList() :  end");
		return list;
	}
	/**
	 * collectorId指定の定期削除処理
	 * 
	 * @ejb.interface-method
	 * @generated
	 * 
	 * 	@ejb.permission
	 *     unchecked="true"
	 *     method-intf="Local"
	 *
	 * @ejb.transaction
	 *     type="NotSupported"
	 *     
	 * @throws FinderException
	 * @throws NamingException
	 * @since 2.0.0
	 */
	public void _managePresavePeriod(CollectorPK pk) throws FinderException, NamingException {
		m_log.debug("_managePresavePeriod() :  start collectorId = " + pk.getCollectorID());
		
		//get Hame Interface 
		CollectorLocalHome home = JNDIConnectionManager.getCollectorLocalHome();
		
		try {
			CollectorLocal bean = home.findByPrimaryKey(pk);
			
			// 無期限収集でかつ実行中のものだけをターゲットとする
			if(bean.getPresavePeriod() >= 0 && (bean.getStatus() == CollectorRunStatus.RUNNING)){
				//保存期間と現在日時から、保存期間の日時を計算
				Calendar cal = Calendar.getInstance();
				cal.add(Calendar.DAY_OF_MONTH, -bean.getPresavePeriod());
				cal.set(Calendar.MILLISECOND,0);
				cal.set(Calendar.SECOND,0);
				
				Date startDate = cal.getTime();
				
				m_log.debug("_managePresavePeriod() :  presaveDate " + " " + startDate.toString());
				if(startDate.after(bean.getStartDate())){
					
					//収集開始時刻を書き変える(GUI対策)
					bean.setStartDate(startDate);
					
					try {
						//収集開始時刻以前のデータを削除し、収集回数を再定義する。
						// 処理に時間がかかった場合にはログ出力する。
						m_log.debug("_managePresavePeriod() :  do managePresave collectorId = " + pk.getCollectorID());
						bean.managePresave(startDate);
					} catch (TransactionRolledbackLocalException e) {
						// jboss.logへの出力
						m_log.warn("Failed to delete : " + pk.getCollectorID());
						m_log.warn(e.getMessage(), e);
						// HinemosApl.logへの出力
						AplLogger apllog = new AplLogger("PERF", "perf");
						apllog.put("SYS", "012");
						throw e;
					}
				}
			}
		} catch (FinderException e) {
			e.printStackTrace();
		}
		
		m_log.debug("_managePresavePeriod() :  end collectorId = " + pk.getCollectorID());
	}

	/**
	 * Quartzへスケジュールを登録します
	 * 
	 * @param cronString スケジュール実行ルールのcron表現
	 *  
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     unchecked="true"
	 *     method-intf="Remote"
	 * 
	 * @throws NamingException
	 * @throws ParseException
	 * @throws SchedulerException
	 * 
	 * @since 2.0.0
	 */
	public void addQuartz(String cronString) throws NamingException, ParseException, SchedulerException {
		//QuartzのSchedulerを削除します。
		QuartzManager manager = QuartzUtil.getQuartzManager();
		try {
			//Quartzマネージャを
			manager.deleteSchedule(QuartzConstant.JNDI_NAME_PERF_MNG, 
					QuartzConstant.GROUP_COLLECT);
		} catch (SchedulerException e) {
			m_log.debug("addQuartz() : deleteJob error = " + e.getMessage());
		} catch (RemoteException e) {
		}
		
		//JobDetail
		JobDetail job = new JobDetail(
				QuartzConstant.JNDI_NAME_PERF_MNG, 
				QuartzConstant.GROUP_COLLECT,
				EJBInvokerJob.class);
		
		String serverName = System.getProperty("jboss.server.name");
		if(serverName.equals("all")){
			job.getJobDataMap().put(
					EJBInvokerJob.PROVIDER_URL, 
			"jnp://localhost:1100");
		}
		
		
		job.setDurability(true);
		
		//JobDetail
		job.getJobDataMap().put(
				EJBInvokerJob.EJB_JNDI_NAME_KEY, 
				QuartzConstant.JNDI_NAME_PERF_MNG);
		//
		job.getJobDataMap().put(
				EJBInvokerJob.EJB_METHOD_KEY, 
				QuartzConstant.METHOD_NAME_PERF_MNG);
		
		//CronTrigger?
		CronTrigger cronTrigger = new CronTrigger(QuartzConstant.JNDI_NAME_PERF_MNG, QuartzConstant.GROUP_COLLECT);
		
        //起動失敗した場合は、次の起動予定時刻をセットするように設定
        cronTrigger.setMisfireInstruction(CronTrigger.MISFIRE_INSTRUCTION_DO_NOTHING);
		
		// cronString = "0 */1 * * * ? *";
		cronTrigger.setCronExpression(cronString);
		
		//Scheduler       
		try {
			manager.addSchedule(job, cronTrigger);
		} catch (RemoteException e) {
		}
	}
	
	/**
	 * Quartzからスケジュールを削除します
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     unchecked="true"
	 *     method-intf="Remote"
	 * 
	 * @throws NamingException
	 * @throws ParseException
	 * @throws SchedulerException
	 * 
	 * @since 2.0.0
	 */
	public void deleteQuartz() throws NamingException, ParseException, SchedulerException {
		//Quartz
		QuartzManager manager = QuartzUtil.getQuartzManager();
		
		try {
			manager.deleteSchedule(QuartzConstant.JNDI_NAME_PERF_MNG, 
					QuartzConstant.GROUP_COLLECT);
		} catch (SchedulerException e) {
			m_log.debug("deleteQuartz() : deleteJob error = " + e.getMessage());
		} catch (RemoteException e) {
		}
	}
	
	/**
	 * Beanからセッション・コンテキストを取得します。
	 * （実装されていません）
	 */
	public void setSessionContext(SessionContext ctx)
	throws EJBException,
	RemoteException {
	}
	
	/**
	 * SessionBean削除時にコンテナから呼ばれます。
	 */
	public void ejbRemove() throws EJBException, RemoteException {
	}
	
	/**
	 * SessionBeanインスタンスが「活性化」状態になる前にコンテナから呼ばれます。
	 */
	public void ejbActivate() throws EJBException, RemoteException {
	}
	
	/**
	 * SessionBeanインスタンスが「非活性化」状態になる前にコンテナから呼ばれます。
	 */
	public void ejbPassivate() throws EJBException, RemoteException {
	}
}
