/*
                                                                                                                                                                 
Copyright (C) 2006 NTT DATA Corporation
                                                                                                                                                                 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
                                                                                                                                                                 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
                                                                                                                                                                 
*/

package com.clustercontrol.performance.bean;

import java.lang.Double;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;

import javax.ejb.FinderException;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.rmi.PortableRemoteObject;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.performance.ejb.bmp.CollectorBean;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorDeviceInfoData;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorItemCodeMstLocal;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorItemCodeMstUtil;
import com.clustercontrol.performance.util.CalculationMethod;
import com.clustercontrol.poller.NotInitializedException;
import com.clustercontrol.sharedtable.DataTable;
import com.clustercontrol.sharedtable.DataTableNotFoundException;
import com.clustercontrol.sharedtable.SharedTable;

/**
 * Facilityのノード実装クラス
 * 種別がノードであるファシリティの性能値を保持します。
 * ScopeTreeクラスの内部で使用することを目的としたクラスです。
 * 
 * @version 1.0
 * @since 1.0
 */
public class Node extends Facility {
	private static final long serialVersionUID = -5989756829710363991L;

	//	ログ出力
	protected static Log m_log = LogFactory.getLog( Node.class );
	
	private static Object modifyLock = new Object();
	
	private static SharedTable _sharedTable = null;
	
	/**
	 * キー ： デバイス種別
	 * 値　　：　デバイス情報のリスト
	 */
	private HashMap<String, HashMap<String, CollectorDeviceInfoData>> m_deviceMap;
	
	private String platformId;
	private String subPlatformId;
	
	private DataTable currentTable = null;
	private DataTable previousTable = null;
	
	private long lastCollectTime;   // 最終更新時刻
		
	/**
	 * コンストラクター 
	 * 
	 * @param facilityID
	 * @param faclityName 
	 * @param deviceDataList このノードに含まれる全てのデバイス情報
	 */
	public Node(final String facilityID, final String faclityName, final String platformId, final String subPlatformId,
			final List<CollectorDeviceInfoData> deviceDataList){
		super(facilityID, faclityName, Facility.NODE);
		
		m_log.debug("create Node " + facilityID + ", " + faclityName + ", " + platformId + ", " + subPlatformId);
		
		this.platformId = platformId;
		this.subPlatformId = subPlatformId;
		
		m_deviceMap = new HashMap<String, HashMap<String, CollectorDeviceInfoData>>();
		
		Iterator<CollectorDeviceInfoData> itr = deviceDataList.iterator();
		while(itr.hasNext()){
			CollectorDeviceInfoData deviceData = itr.next();
			String deviceType = deviceData.getDeviceType();

			if (m_deviceMap.containsKey(deviceType) == false) {
				m_deviceMap.put(deviceType, new HashMap<String, CollectorDeviceInfoData>());
			}
			
			// デバイスタイプごとのマップを取得
			HashMap<String, CollectorDeviceInfoData> mapByDeviceType = m_deviceMap.get(deviceType);
			
			// リポジトリ表示名をキーにしてデバイス情報を設定
			mapByDeviceType.put(deviceData.getDisplayName(), deviceData);
		}
		
		initialize();
	}
	
	public void initialize() {
		synchronized (modifyLock) {
			// 既に初期化済みの場合は何もしない。
			if(_sharedTable != null){
				return;
			}

			try {
				// テーブル生成
				InitialContext ctx = new InitialContext();
				Object obj = ctx.lookup(CollectorBean.SHARED_TABLE_JNDI_NAME);
				_sharedTable = (SharedTable)PortableRemoteObject.narrow(obj, SharedTable.class);
			} catch (NamingException e) {
				m_log.error(e.getMessage(), e);
			}
		}
	}
	
	/**
	 * 性能値を戻します。
	 * 
	 * @param  itemCode 収集項目コード
	 * @param  deviceIndex デバイス番号(現在は未使用)
	 * @param  deviceName デバイス名
	 * @return 性能値
	 */
	@Override
	public double calcValue(final CollectorItemInfo itemInfo) throws NotInitializedException{
		CollectorDeviceInfoData deviceData = null;
		
		try {
			// 収集項目から利用されるデバイス種別を特定
			CollectorItemCodeMstLocal mstData = 
				CollectorItemCodeMstUtil.getLocalHome().findByPrimaryKey(itemInfo.getItemCode());
			String deviceType = mstData.getDeviceType();
			
			// デバッグ出力
			m_log.debug("DeviceType : " + deviceType);

			if(deviceType != null && !"".equals(deviceType)){
				HashMap<String, CollectorDeviceInfoData> deviceTypeMap = m_deviceMap.get(deviceType);

				// デバッグ出力
				if(m_log.isDebugEnabled()){
					Iterator<String> itr = deviceTypeMap.keySet().iterator();
					while(itr.hasNext()){
						String key = itr.next();
						m_log.debug(deviceTypeMap.get(key));
					}
				}
				
				if(deviceTypeMap != null){
					deviceData = deviceTypeMap.get(itemInfo.getDisplayName());
				}
			}
		} catch (FinderException e) {
			m_log.error(e.getMessage(), e);
			return Double.NaN;
		} catch (NamingException e) {
			m_log.error(e.getMessage(), e);
			return Double.NaN;
		}
		
		double value = Double.NaN;
		if(currentTable != null && previousTable != null){
			value = CalculationMethod.getPerformance(platformId, subPlatformId, itemInfo, deviceData, currentTable, previousTable);
		} else {
			m_log.debug("currentTable = null and previousTable = null");
			throw new NotInitializedException("fetch error.");
		}
		
		m_log.debug("calcValue() : " + getFacilityId() + " " + 
				itemInfo.getItemCode() + " " + itemInfo.getDisplayName() + "  " + value);
		
		// 性能値が算出できた場合はバッファに保存する
		if(!Double.isNaN(value)){
			setCalcValueBuffer(new CollectorItemInfo(
					itemInfo.getCollectorId(),
//					itemInfo.getCollectMethod(),
					itemInfo.getItemCode(),
//					itemInfo.getDeviceName(),
					itemInfo.getDisplayName()), value);
		}
		
		return value;
	}

	
	/**
	 * ポーラーから収集値を取得します。
	 * 
	 * @param oids 収集対象のOIDの配列
	 * @param interval 収集間隔
	 * @return 収集時刻（全てのノードで一致させるために最後に収集された値の収集時刻とする）
	 * @throws NotInitializedException 
	 */
	public long fetchMibValue(String collectorId) throws NotInitializedException {
		m_log.debug("fetchMibValue() start :");
		
		try {
			if(_sharedTable == null){
				initialize();
			}
			
			// 2枚のテーブルを取得する間に更新される可能性があるためまとめて取得
			List<DataTable> tables = _sharedTable.getLastDataTables(
					HinemosModuleConstant.PERFORMANCE, getFacilityId(), collectorId, 2);

			currentTable = tables.get(0);
			previousTable = tables.get(1);
			
			lastCollectTime = currentTable.getLastModify();
		} catch (DataTableNotFoundException e) {
			String facilityId = getFacilityId();
			m_log.warn("create table : " + " CollectorID=" + collectorId + ", FacilityId=" + facilityId);
			// テーブルが存在しない場合は生成する
			_sharedTable = null;
			currentTable = null;
			previousTable = null;
			initialize();
		} catch (NotInitializedException e) {
			// 2回分のポーリングが実行できていない場合
			String facilityId = getFacilityId();
			m_log.warn(e.getMessage() + " CollectorID=" + collectorId + ", FacilityId=" + facilityId);
			_sharedTable = null;
			currentTable = null;
			previousTable = null;
			throw e;
		}
		
		m_log.debug("fetchMibValue() end :");
		return lastCollectTime;
	}
	
	/**
	 * 自分自身を返す。
	 * @return HashSet 
	 */
	public HashSet<Facility> getNode(HashSet<Facility>  nodeSet){
		nodeSet.add(this);
		return nodeSet;
	}
	
	/**
	 * 自分自身をカウントして返す。
	 */
	@Override
	public int getNodeCount(){
		return 1;
	}
	
//	/**
//	 * 現在保持しているMIB値をDBに出力する
//	 *
//	 */
//	public void storeMibValue(String collectorId){
//		m_log.debug("storeRowMibValue() :" + collectorId);
//
//		// DBへ格納する日時情報
//		Date date = new Date();  // 現在の時刻とする
//		
//		// MIB値を格納するためのテーブルにアクセス可能なオブジェクトを生成
//		RecordDataDAO dao = new RecordDataDAO();
//		
//		// m_mibValueSet のキーはOID(最後のインデックスを除く)
//		Iterator itr = m_mibValueSet.keySet().iterator();  
//
//		while(itr.hasNext()){
//			String baseOid = (String)itr.next();
//
//			long[][] mibValueSetList = (long[][])m_mibValueSet.get(baseOid);
//
//			for(int i=0; i<mibValueSetList.length; i++){
//				String fullOid = baseOid + "." + i;
//				
//				dao.insertRecordData(
//						collectorId, 
//						fullOid,
//						date,
//						getFacilityID(),
//						mibValueSetList[i][0]);
//			}
//		}
//	}
}
