/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.notify.util;

import java.text.SimpleDateFormat;

import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.OutputBasicInfo;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.notify.bean.NotifySubstitutionParameterConstant;
import com.clustercontrol.repository.ejb.entity.FacilityLocal;
import com.clustercontrol.repository.ejb.entity.FacilityUtil;

/**
 * 文字列置換を行うユーティリティクラス
 *
 * @version 3.0.0
 * @since 3.0.0
 */
public class TextReplacer {
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog( TextReplacer.class );
	
	/** 日時フォーマット。 */
	private static final String SUBJECT_DATE_FORMAT = "yyyy/MM/dd HH:mm:ss";

	/**
	 * バックエンドのデータソースにアクセスせずに置換できるもののみを対象として、
	 * 引数で指定された文字列内の変数を置換します。
	 * 
	 * @param text 変換対象となるtext
	 * @param info 出力内容
	 * @return 変換後のtext
	 */
	public static synchronized String staticSubstitution(String text, OutputBasicInfo info) {

		SimpleDateFormat sdf = new SimpleDateFormat(SUBJECT_DATE_FORMAT);

		// 結果格納用
		String retText;

		retText = text.replaceAll(NotifySubstitutionParameterConstant.APPLICATION_REGEX, toLiteral(info.getApplication()));
		retText = retText.replaceAll(NotifySubstitutionParameterConstant.FACILITY_ID_REGEX, toLiteral(info.getFacilityId()));
		retText = retText.replaceAll(NotifySubstitutionParameterConstant.MONITOR_ID_REGEX, toLiteral(info.getMonitorId()));
		retText = retText.replaceAll(NotifySubstitutionParameterConstant.GENERATION_DATE_REGEX, toLiteral(sdf.format(info.getGenerationDate())));
		retText = retText.replaceAll(NotifySubstitutionParameterConstant.PLUGIN_ID_REGEX, toLiteral(info.getPluginId()));
		retText = retText.replaceAll(NotifySubstitutionParameterConstant.PRIORITY_REGEX, toLiteral(PriorityConstant.typeToString(info.getPriority())));
		retText = retText.replaceAll(NotifySubstitutionParameterConstant.PRIORITY_JP_REGEX, toLiteral(PriorityConstant.typeToStringJP(info.getPriority())));
		retText = retText.replaceAll(NotifySubstitutionParameterConstant.PRIORITY_EN_REGEX, toLiteral(PriorityConstant.typeToStringEN(info.getPriority())));
		retText = retText.replaceAll(NotifySubstitutionParameterConstant.PRIORITY_NUM_REGEX, toLiteral(String.valueOf(info.getPriority())));
		retText = retText.replaceAll(NotifySubstitutionParameterConstant.SCOPE_REGEX, toLiteral(info.getScopeText()));
		retText = retText.replaceAll(NotifySubstitutionParameterConstant.MESSAGE_ID_REGEX, toLiteral(info.getMessageId()));
		retText = retText.replaceAll(NotifySubstitutionParameterConstant.MESSAGE_REGEX, toLiteral(info.getMessage()));
		retText = retText.replaceAll(NotifySubstitutionParameterConstant.ORG_MESSAGE_REGEX, toLiteral(info.getMessageOrg()));
		
		return retText;
	}

	
	/**
	 * 引数で指定された文字列内の変数を置換します。
	 * 
	 * @param text 変換対象となるtext
	 * @param info 出力内容
	 * @return 変換後のtext
	 * @throws NamingException 
	 * @throws FinderException 
	 */
	public static synchronized String substitution(String text, OutputBasicInfo info) throws NamingException, FinderException {
		// 結果格納用
		String retText;
		
		m_log.debug("before staticSubstitution() : " + text);
		// バックエンドのデータソースにアクセスせずに置換できるものを置換
		retText = staticSubstitution(text, info);
		m_log.debug("after staticSubstitution() : " + retText);
		
		// IPアドレスとホスト名の置換
		try {
			// 置換元となっている文章に、IPアドレスとノード名の置換用文字列が存在する場合
			if(retText.indexOf(NotifySubstitutionParameterConstant.IP_ADDRESS) != -1
					|| retText.indexOf(NotifySubstitutionParameterConstant.NODE) != -1){

				String nodeName = null;
				String ipNetworkNumber = null;

				String facilityId = info.getFacilityId();

				if(facilityId != null && !facilityId.equals("")){

					FacilityLocal facility = FacilityUtil.getLocalHome().findByPrimaryKey(facilityId);

					// 対象ファシリティIDがノードである場合
					if(facility != null){

						// ノード名の取得
						nodeName = facility.getNode().getNodeName();

						// IPアドレスの取得
						int version = 4;
						if(facility.getNode().getIpAddressVersion() != null){
							version = facility.getNode().getIpAddressVersion();
						}

						// 入力されているバージョンを比較し、対応するIPアドレスを取得する
						if(version == 4) {
							ipNetworkNumber = facility.getNode().getIpAddressV4();
						}
						else {
							ipNetworkNumber = facility.getNode().getIpAddressV6();
						}
					}
				}

				// ノード名の置換
				if(nodeName != null && !nodeName.equals("")) {
					retText = retText.replaceAll(NotifySubstitutionParameterConstant.NODE_REGEX, toLiteral(nodeName));
				}
				// IPアドレスの置換
				if(ipNetworkNumber != null && !ipNetworkNumber.equals("")) {
					retText = retText.replaceAll(NotifySubstitutionParameterConstant.IP_ADDRESS_REGEX, toLiteral(ipNetworkNumber));
				}

			} 

		}
		catch (NamingException e) {
			throw e;
		} 
		catch (FinderException e) {
			throw e;
		}

		return retText;
	}
	
	/**
	 * IllegalArgumentExceptionを回避するため禁止文字($,\)をエスケープします。
	 * 
	 * @param text エスケープチェック対象となる文字列
	 * @return チェック後の文字列
	 */
	private static String toLiteral(String text) {

		if(text == null) {
			return "";
		}

		if (text.indexOf('\\') == -1 && text.indexOf('$') == -1) {
			return text;
		}

		StringBuffer buffer = new StringBuffer();
		for (int i=0; i < text.length(); i++) {
			char c = text.charAt(i);
			if (c == '\\') {
				buffer.append('\\');
				buffer.append('\\');
			} else if (c == '$') {
				buffer.append('\\');
				buffer.append('$');
			} else {
				buffer.append(c);
			}
		}

		return buffer.toString();
	}	

}
